# -*- coding: utf-8 -*-
"""
********************************************************************************
* Name:
* Author: Alan Lemon
* Created On: March 18, 2019
* Copyright: (c) Aquaveo 2019
* License: BSD 2-Clause
********************************************************************************
"""

__all__ = ['run_model_check']


def run_model_check(adh_model):
    """
    Checks the ADH simulation for obvious warnings or errors in the model inputs.

    Args:
        adh_model (:class:`AdhSimulation <adhparam.adh_model.AdhSimulation>`): ADH Simulation Class

    Returns:
        list of warning strings, list of error strings
    """
    mc = ModelChecker()
    warnings, errors = mc.check_model(adh_model)
    return warnings, errors


class ModelChecker:
    """
    Class object to check ADH model for warnings and errors
    """
    def __init__(self):
        self.warnings = []
        self.errors = []
        self.adh_model = None

    def check_model(self, adh_model):
        """
        Checks the model for any obvious warnings or errors
        Args:
            adh_model: ADH Model class

        Returns:
            list of warning strings, list of error strings
        """
        self.adh_model = adh_model
        self.check_mesh()
        self.check_solution_controls()
        return self.warnings, self.errors

    def check_mesh(self):
        """
        Checks for errors in the ADH 2d mesh
        """
        # make sure mesh nodes are defined
        df = self.adh_model.mesh.nodes
        if df is None or len(df.index) < 3:
            self.errors.append('No mesh nodes defined.')
            return
        col_labels = list(df)
        labels = ['CARD', 'ID', 'X', 'Y', 'Z']
        if col_labels != labels:
            self.errors.append(f'Mesh nodes table invalid. Expected column heading: {labels}. '
                               f'Found column headings: {col_labels}.')
            return

        # make sure all elements are triangles
        df = self.adh_model.mesh.elements
        if df is None or len(df.index) < 1:
            self.errors.append('No mesh defined.')
            return
        col_labels = list(df)
        labels = ['CARD', 'ID', 'NODE_0', 'NODE_1', 'NODE_2', 'MATERIAL_ID']
        if col_labels != labels:
            self.errors.append(f'Mesh elements table invalid. Expected column heading: {labels}. '
                               f'Found column headings: {col_labels}.')
        else:
            for index, row in df.iterrows():
                if row['CARD'] != 'E3T':
                    self.errors.append(f'Invalid mesh element in row {index} of elements table. '
                                       f'Expected "ET3" found "{row["CARD"]}"')

    def check_solution_controls(self):
        """
        Checks specified boundary conditions to make sure the material, node, edge, mid string exists and that any
        assigned time series exists
        """
        # create dict of boundary string ids
        df2 = self.adh_model.boundary_conditions.solution_controls
        if df2 is None:
            return
        df = self.adh_model.boundary_conditions.boundary_strings
        if df is None:
            self.errors.append('No boundary strings defined.')
            return
        if 'ID' not in df:
            self.errors.append('Invalid definition of boundary strings. No "ID" column defined.')
            return
        if 'STRING_ID' not in df2:
            self.errors.append('Invalid definition of solution controls. No "STRING_ID" column defined.')
            return

        bs_set = set()
        for index, row in df.iterrows():
            bs_set.add(row['ID'])
        sid_set = set()
        for index, row in df2.iterrows():
            sid_set.add(row['STRING_ID'])
        for sid in sid_set:
            if sid not in bs_set:
                self.errors.append('Invalid solution control specified with "STRING ID" {}. This string id is not '
                                   'defined.'.format(sid))

    def validate_sw2(self):
        """
        Method to validate AdH Parameters to ensure that the required parameters are included and are reasonable.

        Parameters
        ----------

        Returns
        -------
        None
            Has the potential to stop the program with multiple exceptions. Prints a success statement upon completion.
        """
        # ### Operating parameters ###
        # check preconditioner
        if self.operation_parameters.preconditioner_type != 1:
            self.errors.append('Preconditioner is not set to the recommended default value of 1. Proceed with caution.')

        # check blocks
        if self.operation_parameters.blocks_per_processor != 1:
            self.errors.append('Number of blocks is not set to the recommended initial value of 1. Proceed with caution.')

        # ### Iteration Parameters ###
        # check tolerance is reasonable
        if self.iteration_parameters.non_linear_residual_tolerance > 0.5:
            self.errors.append('Iteration tolerance is unrealistic.')

        # check tolerance is reasonable
        if self.iteration_parameters.non_linear_incremental_tolerance > 0.5:
            self.errors.append('Iteration tolerance is unrealistic.')

        # ### Material parameters ###
        # check density
        if not (999.0 < self.model_constants.density < 1001.0) and not (
                        1.93 < self.model_constants.density < 1.95):
            self.errors.append('Density should be specified in SI (1000 kg/cm) or English (1.94 slugs/cft). '
                        'Syntax: MP RHO 1000 ')

        # check manning's n constant
        if not self.model_constants.mannings_unit_constant == 1.0 and not (
                        1.4 < self.model_constants.mannings_unit_constant < 1.5):
            self.errors.append('Mannings unit constant should be specified in SI (1.0) or English (1.486). '
                        'Syntax: MP MUC 1.0 ')

        # check material strings
        max_material = max(self.material_properties, key=int)
        if max_material != len(self.material_properties):
            self.errors.append('*** Material strings not set properly. Material numbering must be sequential ***')

        # check estimated eddy viscosity
        for key, material in self.material_properties.items():
            reasonable_bounds = material.param.estimated_eddy_viscosity_weighting_factor.softbounds
            if not reasonable_bounds[0] <= material.estimated_eddy_viscosity_weighting_factor <= reasonable_bounds[1]:
                self.errors.append('*** Estimated eddy viscosity is outside of suggested range.***')

            # check transport properties
            if self.operation_parameters.transport == 0:
                if material.transport_properties:
                    self.errors.append(
                        '*** Transport properties given for Material {}, but OP TRN is set to 0'.format(key))
            elif len(material.transport_properties) != self.operation_parameters.transport:
                self.errors.append('*** Missing transport properties for some constituents in Material {}'.format(key))

        # ### Time series ###
        # ensure designated time step series is correct
        if self.time_control.time_step_option == 'Time step series (SERIES DT)':
            if not self.time_series[
                self.time_control.max_time_step_size_time_series].series_type == 'SERIES DT':
                self.errors.append('*** SERIES DT number does not match specification in time control ***')

        # get a list of the series types in the simulation
        type_list = [value.series_type for key, value in self.time_series.items()]
        # check time series objects
        if 'SERIES BC' not in type_list:
            self.errors.append('*** No SERIES BC found in the simulation ***')
        if 'SERIES AWRITE' not in type_list and 'SERIES WRITE' not in type_list and 'SERIES OS' not in type_list:
            self.errors.append('*** No output series found in the simulation ***')

        # check_transport(self):
        num_trn = self.adh_model.boundary_conditions.number_of_constituents()
        op_trn = self.operation_parameters.transport
        if num_trn != op_trn:
            self.errors.append('Invalid number of constituents specified. OP TRN {} does not match '
                          'the {} constituents found in the simulation.'.format(op_trn, num_trn))

        errors = self.check_solution_controls()

        # todo: check for boundary condition statements (string/series linkage)
        # this list is HUGE @@ update this

        result = True  # todo collect warnings and return
        return result


