"""This module defines the settings CalcData for the Hydraulic Toolbox CalcData."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
import platform
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.app_data.settings.display import DisplaySettings
from xms.FhwaVariable.core_data.app_data.settings.preferences import PreferenceDefinitions
from xms.FhwaVariable.core_data.app_data.settings.theme_change import get_macos_theme, get_windows_theme
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.personal_details import PersonalDetails
from xms.FhwaVariable.core_data.calculator.setting_group import SettingGroup
from xms.FhwaVariable.core_data.variables.variable import Variable


class AppSettingsData(SettingGroup, CalcData):
    """A class that defines the program settings."""

    def __init__(self, app_name: str = 'HydraulicToolbox', version: str = '0.1', agency: str = 'FHWA',
                 developed_by: str = 'Aquaveo', doc_folder: Path = None):
        """Initializes the Settings CalcData.

        Args:
            app_name (str): The name of the application.
            version (float): The version of the application.
            agency (str): The agency that developed the application.
            developed_by (str): The company that developed the application.
            doc_folder (Path): The path to the documentation folder.
        """
        SettingGroup.__init__(self, app_name, version, agency, developed_by)
        CalcData.__init__(self)

        self.name = 'Settings calc'
        self.type = 'AppSettingsData'
        self.uuid = uuid.uuid4()

        self.calc_support_dual_input = False  # Displays a second column of input (HY-8 Style)
        self.calc_support_warnings = False  # Displays the warnings text box
        self.calc_support_results = False  # Displays the Results table box
        self.calc_support_plot = False  # Displays the plot

        self.dlg_width = 800
        self.dlg_height = 600

        # input
        self.input = {}
        # self.input_dual = {}
        self.calc_support_dual_input = True

        self.unknown = None

        # Results
        self.clear_my_own_results = True
        self.results = {}

        self.warnings = []

        # plot

        # Help/Wiki
        # self.self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Channel_Calculator'

        # Settings Data
        self.input['User details'] = Variable('User details', 'class', PersonalDetails(app_name, version, agency,
                                                                                       developed_by))
        # self.input['User details'].name = 'User details'
        self.input['User details'].type = 'PersonalDetails'
        self.input['Preferences'] = Variable('Preferences', 'class',
                                             PreferenceDefinitions(app_name, version, agency, developed_by,
                                                                   doc_folder=doc_folder))
        # self.input['Constant definitions'] = Variable('Constant definitions',
        #                                               'class',
        #                                               ConstantDefinitions(name, version, agency, developed_by))
        # Consider putting this in the GUI repo
        self.input['Display preferences'] = Variable('Display preferences',
                                                     'class',
                                                     DisplaySettings(app_name, version, agency, developed_by))

    def get_theme(self):
        """Gets the selected theme."""
        self.get_system_theme()
        return self.input['Display preferences'].get_val().get_theme()

    def get_system_theme(self):
        """Gets the selected theme."""
        system = platform.system()

        theme = None
        if system == 'Windows':
            theme = get_windows_theme()
        elif system == 'Darwin':
            theme = get_macos_theme()

        if theme is not None and theme == 'dark':
            self.input['Display preferences'].value.set_setting('Select theme', 'dark')
            self.input['Display preferences'].value.set_theme_colors()
        else:
            self.input['Display preferences'].value.set_setting('Select theme', 'light')
            self.input['Display preferences'].value.set_theme_colors()

        return theme

    def get_input_tab_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}

        for input in self.input:
            if isinstance(self.input[input], dict):
                continue
            input_vars[input] = self.input[input].get_val().get_input_group()

        return input_vars

    # def get_input_dual_tab_group(self, unknown=None):
    #     """Get the input group (for user-input).

    #     Returns
    #         input_vars (list of variables): input group of variables
    #     """
    #     input_vars = {}

    #     if len(self.model_list) != 1:
    #         return input_vars

    #     for model in self.model_list:
    #         for input in self.input[model]:
    #             input_vars[input] = self.input[model][input].get_val().get_input_group()

    #     return input_vars
