"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionLengthCalc:
    """Perform length unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Length
    # ----------------------------------------------------------------------
    # US Units
    mile = ['mi', 'mile', 'Mile', 'MILE', 'miles', 'Miles', 'MILES']
    yard = ['yd', 'yard', 'Yard', 'YARD', 'yards', 'Yards', 'YARDS']
    foot = ["\'", 'ft', 'foot', 'Foot', 'FOOT', 'feet', 'Feet', 'FEET']
    inch = ['"', 'in', 'inch', 'Inch', 'INCH', 'inches', 'Inches', 'INCHES']
    # SI Units
    km = ['km', 'kilometer', 'Kilometer', 'KILOMETER', 'kilometers', 'Kilometers', 'KILOMETERS']
    m = ['m', 'meter', 'Meter', 'METER', 'meters', 'Meters', 'METERS']
    cm = ['cm', 'centimeter', 'Centimeter', 'CENTIMETER', 'centimeters', 'Centimeters', 'CENTIMETERS']
    mm = ['mm', 'millimeter', 'Millimeter', 'MILLIMETER', 'millimeters', 'Millimeters', 'MILLIMETERS']

    stream_value = 12.0
    us_gallon = True

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.mile:
            return True, 'km'
        elif from_unit in self.yard:
            return True, 'm'
        elif from_unit in self.foot:
            return True, 'm'
        elif from_unit in self.inch:
            return True, 'mm'
        return False, ''

    # ----------------------------------------------------------------------
    # Length
    # ----------------------------------------------------------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert the Length unit.

        Args:
            from_unit (string): unit that value is currently in
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Units
        if from_unit in self.mile:
            return self.convert_from_mile(to_unit, value)
        elif from_unit in self.yard:
            return self.convert_from_yard(to_unit, value)
        elif from_unit in self.foot:
            return self.convert_from_foot(to_unit, value)
        elif from_unit in self.inch:
            return self.convert_from_inch(to_unit, value)
        # SI Units
        elif from_unit in self.km:
            return self.convert_from_km(to_unit, value)
        elif from_unit in self.m:
            return self.convert_from_m(to_unit, value)
        elif from_unit in self.cm:
            return self.convert_from_cm(to_unit, value)
        elif from_unit in self.mm:
            return self.convert_from_mm(to_unit, value)

        return False, value

    def convert_from_mile(self, to_unit, value):
        """Convert from the miles unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.mile:
            return True, value
        elif to_unit in self.yard:
            return True, value * 1760.0
        elif to_unit in self.foot:
            return True, value * 5280.0
        elif to_unit in self.inch:
            return True, value * 63360.0
        # SI
        elif to_unit in self.km:
            return True, value * 1.609344
        elif to_unit in self.m:
            return True, value * 1609.344
        elif to_unit in self.cm:
            return True, value * 160934.4
        elif to_unit in self.mm:
            return True, value * 1609344.0

        else:
            return False, value

    def convert_from_yard(self, to_unit, value):
        """Convert from the yard unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.mile:
            return True, value / 1760.0
        elif to_unit in self.yard:
            return True, value
        elif to_unit in self.foot:
            return True, value * 3.0
        elif to_unit in self.inch:
            return True, value * 36.0
        # SI
        elif to_unit in self.km:
            return True, value * 0.0009144
        elif to_unit in self.m:
            return True, value * 0.9144
        elif to_unit in self.cm:
            return True, value * 91.44
        elif to_unit in self.mm:
            return True, value * 914.4

        else:
            return False, value

    def convert_from_foot(self, to_unit, value):
        """Convert from the foot unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.mile:
            return True, value * 0.000189394
        elif to_unit in self.yard:
            return True, value * 0.33333333
        elif to_unit in self.foot:
            return True, value
        elif to_unit in self.inch:
            return True, value * 12.0
        # SI
        elif to_unit in self.km:
            return True, value * 0.0003048
        elif to_unit in self.m:
            return True, value * 0.3048
        elif to_unit in self.cm:
            return True, value * 30.48
        elif to_unit in self.mm:
            return True, value * 304.8

        else:
            return False, value

    def convert_from_inch(self, to_unit, value):
        """Convert from the inch unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.mile:
            return True, value * 1.5783e-5
        elif to_unit in self.yard:
            return True, value * 0.0277778
        elif to_unit in self.foot:
            return True, value * 0.0833333
        elif to_unit in self.inch:
            return True, value
        # SI
        elif to_unit in self.km:
            return True, value * 2.54e-5
        elif to_unit in self.m:
            return True, value * 0.0254
        elif to_unit in self.cm:
            return True, value * 2.54
        elif to_unit in self.mm:
            return True, value * 25.4

        else:
            return False, value

    def convert_from_km(self, to_unit, value):
        """Convert from the km unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.mile:
            return True, value * 0.621371
        elif to_unit in self.yard:
            return True, value * 1093.61
        elif to_unit in self.foot:
            return True, value * 3280.84
        elif to_unit in self.inch:
            return True, value * 39370.1
        # SI
        elif to_unit in self.km:
            return True, value
        elif to_unit in self.m:
            return True, value * 1000.0
        elif to_unit in self.cm:
            return True, value * 100000.0
        elif to_unit in self.mm:
            return True, value * 1.0e+6

        else:
            return False, value

    def convert_from_m(self, to_unit, value):
        """Convert from the m unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.mile:
            return True, value * 0.000621371
        elif to_unit in self.yard:
            return True, value * 1.09361
        elif to_unit in self.foot:
            return True, value * 3.28084
        elif to_unit in self.inch:
            return True, value * 39.3701
        # SI
        elif to_unit in self.km:
            return True, value * 0.001
        elif to_unit in self.m:
            return True, value
        elif to_unit in self.cm:
            return True, value * 100.0
        elif to_unit in self.mm:
            return True, value * 1000.0

        else:
            return False, value

    def convert_from_cm(self, to_unit, value):
        """Convert from the cm unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.mile:
            return True, value * 0.00000621371
        elif to_unit in self.yard:
            return True, value * 0.0109361
        elif to_unit in self.foot:
            return True, value * 0.0328084
        elif to_unit in self.inch:
            return True, value * 0.393701
        # SI
        elif to_unit in self.km:
            return True, value * 0.00001
        elif to_unit in self.m:
            return True, value * 0.01
        elif to_unit in self.cm:
            return True, value
        elif to_unit in self.mm:
            return True, value * 10.0

        else:
            return False, value

    def convert_from_mm(self, to_unit, value):
        """Convert from the mm unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.mile:
            return True, value * 0.000000621371
        elif to_unit in self.yard:
            return True, value * 0.00109361
        elif to_unit in self.foot:
            return True, value * 0.00328084
        elif to_unit in self.inch:
            return True, value * 0.0393701
        # SI
        elif to_unit in self.km:
            return True, value * 0.000001
        elif to_unit in self.m:
            return True, value * 0.001
        elif to_unit in self.cm:
            return True, value * 0.1
        elif to_unit in self.mm:
            return True, value

        else:
            return False, value
