"""Abstract Class that defines the base functionality for a View (delivery mechanism, GUI) plug-in."""
__copyright__ = "(C) Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules
from abc import ABC, abstractmethod
from pathlib import Path
from typing import Dict, List, Optional
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.view_model.dialog.dialog import Dialog
from xms.FhwaVariable.interface_adapters.view_model.main.about_dlg import AboutDlgData
from xms.FhwaVariable.interface_adapters.view_model.main.application import Application
from xms.FhwaVariable.interface_adapters.view_model.main.message_box import MessageBox
from xms.FhwaVariable.interface_adapters.view_model.main.select_file import SelectFile
from xms.FhwaVariable.interface_adapters.view_model.main.support_window import SupportWindow
from xms.FhwaVariable.interface_adapters.view_model.main.tool import ToolItem


windows = []


# def get_named_window(name: str) -> Optional[MainWindow]:
#     """Get the Qt main window with the given name."""
#     pass

class ViewMiscBase(ABC):
    """Abstract class that defines the miscellaneous view functions for the application."""

    # Miscellanea
    @abstractmethod
    def get_screens_info(self) -> dict:
        """Get information about the screens.

        Returns:
            dict: Information about the screens.
        """
        pass

    @abstractmethod
    def get_screen_resolution(self) -> tuple[int, int]:
        """Get the x and y resolution of the screen.

        Returns:
            tuple[int, int]: The x and y resolution of the screen.
        """
        pass

    @abstractmethod
    def get_screen_width_and_height(self) -> tuple[float, float]:
        """Get the width and height of the screen.

        Returns:
            tuple[float, float]: The width and height of the screen in millimeters.
        """
        pass

    @abstractmethod
    def get_screen_dpi(self) -> float:
        """Get the DPI of the screen.

        Returns:
            float: The DPI of the screen.
        """
        pass

    @abstractmethod
    def get_screen_scaling_factor(self) -> float:
        """Get the screen scaling factor.

        Returns:
            float: The screen scaling factor.
        """
        pass

    @abstractmethod
    def get_clipboard(self) -> str:
        """Get the clipboard."""
        return None

    @abstractmethod
    def message_box(self, message_box: MessageBox) -> str:
        """
        Send a message to the user.

        Args:
            message_box: The view model message box.

        Returns:
            str: The user response (button pressed).
        """
        pass

    @abstractmethod
    def select_file(self, select_file: SelectFile) -> List[Path]:
        """
        Select a file using QFileDialog.

        Args:
            select_file: The view model select_file.
        """
        pass

    @abstractmethod
    def display_image(self, image) -> None:
        """
        Display an image.

        Args:
            image: The image to display.
        """
        pass


class ViewDialogBase(ViewMiscBase):
    """Abstract class that defines the necessary view functions for the dialog."""
    # Dialog
    @abstractmethod
    def create_dialog(self, dialog: Dialog) -> None:
        """
        Create a dialog (or equivalent).

        Args:
            dialog (model.Dialog): The view model of the dialog.
        """
        pass

    @abstractmethod
    def update_dialog(self, dialog: Dialog) -> None:
        """
        Update a dialog (or equivalent).

        Args:
            dialog (model.Dialog): The view model of the dialog.
        """
        pass

    @abstractmethod
    def run_dialog(self) -> bool:
        """Run the dialog.

        Returns:
            bool: True if the dialog changes were accepted (True) or rejected (False).
        """
        pass

    # @abstractmethod
    # def select_color(self, dialog: SelectColor) -> None:
    #     """
    #     Select a file using QFileDialog.

    #     Args:
    #         dialog: The view model dialog.
    #     """
    #     pass


class ViewBase(ViewMiscBase):
    """Abstract class that defines the necessary view functions for the application."""
    # Main Interface
    @abstractmethod
    def create_application(self, application: Application) -> None:
        """
        Create the application view from a view model.

        Args:
            application (VmApplication): The view model of the application.
        """
        pass

    @abstractmethod
    def update_application(self, application: Application) -> None:
        """
        Update the application view from a view model.

        Args:
            application (model.Application): The view model of the application.
        """
        pass

    @abstractmethod
    def get_main_window_geometry(self) -> dict:
        """
        Get the geometry of the main window.

        Returns:
            dict: The geometry of the main window.
        """
        pass

    @abstractmethod
    def update_support_window(self, support_window_index: int, support_window: SupportWindow) -> None:
        """
        Update the Support Window from a view model.

        Args:
            support_window_index (int): The index of the support window.
            support_window (model.SupportWindow): The view model of the support window.
        """
        pass

    @abstractmethod
    def get_support_window_geometry(self, support_window_index: int) -> dict:
        """
        Get the geometry of the support window.

        Args:
            support_window_index (int): The index of the support window.

        Returns:
            dict: The geometry of the support window.
        """
        pass

    # @abstractmethod
    # def update_graphics_view(self, graphics_view: GraphicsView, window_name: str) -> None:
    #     """
    #     Update the graphics view.

    #     Args:
    #         graphics_view: The graphics view.
    #         window_name: The window to update.
    #     """
    #     pass

    @abstractmethod
    def show_context_menu(self, menu_dict: Dict[str, ToolItem], pos, item_uuid) -> None:
        """
        Show a context menu.

        Args:
            menu_dict: The context menu.
            pos: The position to show the menu.
            item_uuid (UUID): The uuid of the item.
        """
        pass

    @abstractmethod
    def run(self) -> bool:
        """
        Run the application using the supplied initial view model.
        """
        pass

    @abstractmethod
    def show_about_dialog(self, about_dlg: AboutDlgData) -> None:
        """Show the about dialog."""
        pass

    @abstractmethod
    def close(self) -> None:
        """Close the application."""
        pass

    @abstractmethod
    def create_dialog_view(self, dlg_uuid: Optional[uuid.UUID] = None) -> ViewDialogBase:
        """
        Create a dialog (or equivalent).

        Args:
            dlg_uuid: The uuid of the dialog.
            dialog (model.Dialog): The view model of the dialog.
        """
        pass
