"""CalcData for performing Pier Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.interface_adapters.presenter.dialog.plot_presenter import PlotPresenter

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scenario_calc import ScenarioCalc
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_calc import ScourBaseCalc
from xms.HydraulicToolboxCalc.util.list_utils import remove_nans


class BridgeScourCalc(ScourBaseCalc):
    """A class that defines a pier scour at a bridge contraction."""

    def update_bridge_geometry(self):
        """Updates the bridge geometry."""
        scenario = self.input_dict['calc_data']['Scenarios'][0]
        self.pier_data = None
        self.bh_cl = None
        self.abutment_data = None
        _, null_data = self.get_data('Null data')
        if 'Bridge geometry' in self.plot_dict:
            if 'Bridge cross-section' in self.plot_dict['Bridge geometry'] and \
                    0 in self.plot_dict['Bridge geometry']['Bridge cross-section']['series']:
                self.bcs_data = self.plot_dict['Bridge geometry']['Bridge cross-section']['series'][0]
            if 'Piers' in self.plot_dict['Bridge geometry']:
                first_time = True
                for pier_index in self.plot_dict['Bridge geometry']['Piers']:
                    geom_dict = self.plot_dict['Bridge geometry']['Piers'][pier_index]['Pier geometry']['series'][0]
                    if first_time:
                        self.pier_data = copy.deepcopy(geom_dict)
                        first_time = False
                        if 'x var' in geom_dict and len(geom_dict['x var'].get_val()) <= 1:
                            self.pier_data['x var'].set_val([])
                            self.pier_data['y var'].set_val([])
                    else:
                        if 'x var' in geom_dict and len(geom_dict['x var'].get_val()) > 1:
                            self.pier_data['x var'].value_options.append(null_data)
                            self.pier_data['y var'].value_options.append(null_data)

                            self.pier_data['x var'].value_options.extend(geom_dict['x var'].get_val())
                            self.pier_data['y var'].value_options.extend(geom_dict['y var'].get_val())
                            self.pier_data['x var'].value += len(geom_dict['x var'].get_val()) + 1
                            self.pier_data['y var'].value += len(geom_dict['y var'].get_val()) + 1
            if 'Borehole centerlines' in self.plot_dict['Bridge geometry'] and \
                    0 in self.plot_dict['Bridge geometry']['Borehole centerlines']['series']:
                self.bh_cl = self.plot_dict['Bridge geometry']['Borehole centerlines']['series'][0]
        if scenario['Name'] in self.plot_dict and 'lines' in self.plot_dict[scenario['Name']]:
            for line_index in self.plot_dict[scenario['Name']]['lines']:
                if self.plot_dict[scenario['Name']]['lines'][line_index]['Label'] == 'Abutment toe':
                    self.abutment_data = copy.deepcopy(self.plot_dict[scenario['Name']]['lines'][line_index])

        self.input_dict['calc_data']['Gradations']['Boreholes']['calculator'].bcs_data = copy.deepcopy(
            self.bcs_data)
        self.input_dict['calc_data']['Gradations']['Boreholes']['calculator'].pier_data = copy.deepcopy(
            self.pier_data)
        self.input_dict['calc_data']['Gradations']['Boreholes']['calculator'].abutment_data = copy.deepcopy(
            self.abutment_data)

        for bh_index in self.input_dict['calc_data']['Gradations']['Boreholes']['Boreholes']:
            bh = self.input_dict['calc_data']['Gradations']['Boreholes']['Boreholes'][bh_index]['calculator']
            bh.bcs_data = self.bcs_data
            bh.pier_data = self.pier_data
            bh.bh_cl = self.bh_cl
            bh.abutment_data = self.abutment_data
        self.input_dict['calc_data']['Gradations']['Single borehole']['calculator'].bcs_data = self.bcs_data
        self.input_dict['calc_data']['Gradations']['Single borehole']['calculator'].pier_data = self.pier_data
        self.input_dict['calc_data']['Gradations']['Single borehole']['calculator'].bh_cl = self.bh_cl
        self.input_dict['calc_data']['Gradations']['Single borehole']['calculator'].abutment_data = self.abutment_data

    @staticmethod
    def check_unique_pier_names(pier_names):
        """Check if pier names are unique.

        Args:
            pier_names (list of str): list of pier names

        Returns:
            duplicates (list of str): list of duplicate pier names
        """
        name_count = {}
        duplicates = []

        # Count occurrences of each pier name
        for name in pier_names:
            if name in name_count:
                name_count[name] += 1
            else:
                name_count[name] = 1

        # Identify duplicates
        for name, count in name_count.items():
            if count > 1:
                duplicates.append(name)

        return duplicates

    def update_gradations(self):
        """Update the gradation lists."""
        gradations = self.input_dict['calc_data']['Gradations']['calculator']
        gradations.bcs_data = self.bcs_data
        gradations.pier_data = self.pier_data
        gradations.bh_cl = self.bh_cl
        gradations.abutment_data = self.abutment_data

        gradations.compute_data_with_subdict(self.input_dict, self.input_dict['calc_data']['Gradations'],
                                             self.plot_dict)

        for scenario_index in self.input_dict['calc_data']['Scenarios']:
            if scenario_index in ['Selected item', ]:
                continue
            scenario = self.input_dict['calc_data']['Scenarios'][scenario_index]

            scenario['calculator'].gradations = gradations

            compute_shear_decay = scenario['Compute shear decay']

            # TODO: Ensure we are using these water and soil properties!!
            # Update water and soil properties
            item_list = [scenario['Long-term degradation'],
                         scenario['Contraction scour']['Main channel contraction scour'],
                         scenario['Contraction scour']['Left overbank contraction scour'],
                         scenario['Contraction scour']['Right overbank contraction scour'],
                         scenario['Abutment scour']['Left abutment scour'],
                         scenario['Abutment scour']['Right abutment scour']]
            for pier_index in scenario['Pier scour']:
                item_list.append(scenario['Pier scour'][pier_index])

            for item in item_list:
                item['calculator'].gradations = gradations
                item['calculator'].is_computing_shear_decay = compute_shear_decay
                item['calculator'].bcs_data = self.bcs_data
                item['calculator'].pier_data = self.pier_data
                item['calculator'].bh_cl = self.bh_cl
                item['calculator'].abutment_data = self.abutment_data

    def _get_can_compute(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        result = True

        return result

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.update_bridge_geometry()
        self.update_gradations()

        self.min_y = sys.float_info.max

        return self.compute_cross_section()

    def compute_cross_section(self):
        """Computes the cross section for the pier scour."""
        # Update the scenarios with basic data
        for scenario_index in self.input_dict['calc_data']['Scenarios']:
            if scenario_index in ['Selected item',]:
                continue
            scenario = self.input_dict['calc_data']['Scenarios'][scenario_index]['calculator']
            self.results[scenario.name] = {}
            if scenario.name not in self.plot_dict:
                self.plot_dict[scenario.name] = {}
            name = f'{scenario.name} profile'
            self.plot_dict[scenario.name]['Plot name'] = name
            self.plot_dict[scenario.name]['Legend'] = 'best'
            self.plot_dict[scenario.name]['index'] = 1

        # Update the selected scenario
        scenario_index = self.input_dict['calc_data']['Scenarios']['Selected item']
        sel_scenario = None
        if scenario_index is not None:
            sel_scenario = self.input_dict['calc_data']['Scenarios'][scenario_index]['calculator']
        name = sel_scenario.name

        bridge_geometry = self.input_dict['calc_data']['Bridge geometry']

        x_var = None

        add_min_scour_lateral = False
        add_left_abut_min_scour_lateral = False
        add_right_abut_min_scour_lateral = False

        # Get the approach cross section
        approach_valid = self.get_the_approach_bridge_cross_section(bridge_geometry, self.plot_dict[name])

        # Get the main bridge cross section
        _, null_data = self.get_data('Null data')
        thalweg_station = None
        cross_valid = self.get_the_main_bridge_cross_section(bridge_geometry, self.plot_dict[name])
        cs_subdict, _ = self.get_plot_subdict_and_key_by_name('Bridge cross-section', 'series', name)
        if cross_valid and cs_subdict is not None:
            x_var = copy.deepcopy(cs_subdict['x var'])
            x_data = copy.deepcopy(x_var.get_val())
            y_var = copy.deepcopy(cs_subdict['y var'])
            y_data = copy.deepcopy(y_var.get_val())
            x_data, y_data, _ = remove_nans(x_data, y_data, null_data=null_data)
            thalweg_elevation = min(y_data)
            thalweg_index = y_data.index(thalweg_elevation)
            thalweg_station = x_data[thalweg_index]
            left_bank_max_elevation = max(y_data[:thalweg_index])
            right_bank_max_elevation = max(y_data[thalweg_index:])
            if 'Geometry' not in self.results:
                self.results['Geometry'] = {}
            self.results['Geometry']['Thalweg station'] = thalweg_station
            self.results['Geometry']['Thalweg elevation'] = thalweg_elevation
            self.results['Geometry']['Left bank max elevation'] = left_bank_max_elevation
            self.results['Geometry']['Right bank max elevation'] = right_bank_max_elevation
            self.results[sel_scenario.name]['Thalweg station'] = thalweg_station
            self.results[sel_scenario.name]['Thalweg elevation'] = thalweg_elevation
            add_min_scour_lateral = True
        else:
            self.warnings['Cross-section data'] = 'Please enter the bridge cross-section data'

        if approach_valid and cross_valid and thalweg_station is not None:
            self.adjust_approach_thalweg_to_match_cross_section(self.plot_dict[name], thalweg_station)

        # Get the Bridge Deck and close the shape
        if self.get_the_bridge_deck(bridge_geometry, self.plot_dict[name]):
            bd_subdict, _ = self.get_plot_subdict_and_key_by_name('Bridge deck geometry', 'series', name)
            bridge_y = bd_subdict['y var'].get_val()
            bridge_low_chord = min(bridge_y)
            bridge_high_chord = max(bridge_y)
            if 'Geometry' not in self.results:
                self.results['Geometry'] = {}
            self.results['Geometry']['Bridge low-chord'] = bridge_low_chord
            self.results['Geometry']['Bridge high-chord'] = bridge_high_chord
        else:
            self.warnings['Bridge Deck'] = \
                'Enter bridge deck geometry data for visulaization and contraction pressure scour'

        # Get the Pier geometry -- This may already be done in plot_dict
        self.get_the_pier_geometry(name)

        # Plot the channel banks
        banks = []
        if ScenarioCalc.is_left_channel_bank_specified(bridge_geometry):
            banks.append(bridge_geometry['Channel left bank station'])
        if ScenarioCalc.is_right_channel_bank_specified(bridge_geometry):
            banks.append(bridge_geometry['Channel right bank station'])
        bank_subdict, _ = self.get_plot_subdict_and_key_by_name('Channel bank', 'lines', name)
        if bank_subdict is not None and 'Line intercepts' in bank_subdict:
            bank_subdict['Line intercepts'] = banks

        if ScenarioCalc.is_left_channel_bank_specified(bridge_geometry) and \
                ScenarioCalc.is_right_channel_bank_specified(bridge_geometry):
            pass
        else:
            self.warnings['Overbank stations'] = 'Specify channel bank stations to compute overbank contraction scour'

        # Plot the Abtment Toe Stations
        abutments = []
        if ScenarioCalc.is_left_abutments_specified(bridge_geometry):
            abutments.append(bridge_geometry['Left abutment toe station'])
        if ScenarioCalc.is_right_abutments_specified(bridge_geometry):
            abutments.append(bridge_geometry['Right abutment toe station'])
        abut_subdict, _ = self.get_plot_subdict_and_key_by_name('Abutment toe', 'lines', name)
        if abut_subdict is not None and 'Line intercepts' in abut_subdict:
            abut_subdict['Line intercepts'] = abutments

        if ScenarioCalc.is_left_abutments_specified(bridge_geometry) and ScenarioCalc.is_right_abutments_specified(
                bridge_geometry):
            pass
        else:
            self.warnings['Abutment toe stations'] = 'Specify abutment toe stations to compute abutment scour'

        # Plot Borehole Centerlines
        self.contracted_options = ['Single D50', 'Single borehole', 'Multiple boreholes']
        if self.input_dict['calc_data']['Gradations']['Contracted gradation input options'] == 'Multiple boreholes':
            borehole_names = []
            borehole_cl = []
            gradations = self.input_dict['calc_data']['Gradations']
            bh_manager = gradations['Boreholes']
            if bh_manager is not None:
                for bh in bh_manager['Boreholes']:
                    borehole_names.append(bh_manager['Boreholes'][bh]['Name'])
                    borehole_cl.append(bh_manager['Boreholes'][bh]['Centerline'])
            bh_subdict, bh_index = self.get_plot_subdict_and_key_by_name('Borehole centerlines', 'lines')
            if bh_subdict is not None:
                bh_subdict['Labels'] = borehole_names
                bh_subdict['Line intercepts'] = borehole_cl
        elif self.input_dict['calc_data']['Gradations']['Contracted gradation input options'] == 'Single borehole':
            borehole_names = []
            borehole_cl = []
            gradations = self.input_dict['calc_data']['Gradations']
            bh = gradations['Single borehole']
            borehole_names.append(bh['Name'])
            borehole_cl.append(bh['Centerline'])
            bh_subdict, bh_index = self.get_plot_subdict_and_key_by_name('Borehole centerlines', 'lines')
            if bh_subdict is not None:
                bh_subdict['Labels'] = borehole_names
                bh_subdict['Line intercepts'] = borehole_cl
        else:
            cl_subdict, _ = self.get_plot_subdict_and_key_by_name('Borehole centerlines', 'lines', name)
            if cl_subdict is not None and 'Line intercepts' in cl_subdict:
                cl_subdict['Line intercepts'] = []

        # Copy geometry data for all scenario plots
        for cur_scenario in self.input_dict['calc_data']['Scenarios']:
            if cur_scenario in ['Selected item',]:
                continue
            if cur_scenario != scenario_index:
                cur_name = self.input_dict['calc_data']['Scenarios'][cur_scenario]['calculator'].name
                self.plot_dict[cur_name] = copy.deepcopy(self.plot_dict[name])

        # Compute Scour Scenario Data
        if cross_valid:
            for cur_scenario_index in self.input_dict['calc_data']['Scenarios']:
                if cur_scenario_index in ['Selected item',]:
                    continue
                cur_scenario = self.input_dict['calc_data']['Scenarios'][cur_scenario_index]['calculator']
                cur_scenario.scenario_index = cur_scenario_index
                name = cur_scenario.name
                self.results[name]['Average WSE'] = null_data
                cur_scenario.input_dict = copy.deepcopy(self.input_dict)
                cur_scenario.input_dict['calc_data'] = self.input_dict['calc_data']['Scenarios'][cur_scenario_index]

                applied_ltd, applied_cs, left_cs, main_cs, right_cs, applied_labs, applied_rabs, applied_ps = \
                    cur_scenario.compute_cross_section(
                        self.plot_dict, bridge_geometry, x_var, y_var, x_data, y_data, thalweg_elevation,
                        null_data=null_data)
                for warning in cur_scenario.warnings:
                    self.warnings[f'Scenario warning - {cur_scenario.name}'] = f'{cur_scenario.name}: {warning}'
                cur_scenario.warnings.clear()

                if 'Average WSE' in cur_scenario.results and cur_scenario.results['Average WSE'] != [null_data]:
                    self.results[cur_scenario.name]['Average WSE'] = cur_scenario.results['Average WSE']
                else:
                    self.warnings['scenario WSE'] = \
                        'Scenario: {cur_scenario.name}; Specify WSE for visulaization and abutment scour'

                if applied_ltd:
                    self.results[cur_scenario.name]['Long-term degradation'] = \
                        cur_scenario.results['Long-term degradation']
                else:
                    self.warnings['ltd'] = (f'Scenario: {cur_scenario.name}; specify Longterm degradation parameters')

                if applied_cs:
                    self.results[cur_scenario.name]['Contraction scour'] = cur_scenario.results['Contraction scour']

                if applied_labs or applied_rabs:
                    self.results[cur_scenario.name]['Abutment scour'] = {}
                if applied_labs:
                    add_left_abut_min_scour_lateral = True
                    self.results[cur_scenario.name]['Abutment scour']['Left abutment'] = \
                        cur_scenario.results['Abutment scour']['Left abutment']
                else:
                    self.warnings['left abut'] = f'{cur_scenario.name}: Specify left abutment scour parameters'

                if applied_rabs:
                    add_right_abut_min_scour_lateral = True
                    self.results[cur_scenario.name]['Abutment scour']['Right abutment'] = \
                        cur_scenario.results['Abutment scour']['Right abutment']
                else:
                    self.warnings['right abut'] = f'{cur_scenario.name}: Specify right abutment scour parameters'

                if applied_ps:
                    pier_count = 0
                    for pier_index in cur_scenario.input_dict['calc_data']['Pier scour']:
                        dict_name = cur_scenario.input_dict['calc_data']['Pier scour'][pier_index]['Name']
                        if dict_name in cur_scenario.results['Pier scour']:
                            if 'Pier scour' not in self.results[cur_scenario.name]:
                                self.results[cur_scenario.name]['Pier scour'] = {}
                            self.results[cur_scenario.name]['Pier scour'][dict_name] = cur_scenario.results[
                                'Pier scour'][dict_name]
                        pier_count += 1
                else:
                    self.warnings['pier geometry'] = f'{cur_scenario.name}: Specify pier geometry and scour parameters'

                if applied_ltd or applied_cs or applied_labs or applied_rabs or applied_ps:
                    self.results[cur_scenario.name]['Lowest scour elevation'] = \
                        cur_scenario.results['Lowest scour elevation']

                if applied_ltd or applied_cs or applied_labs or applied_rabs or applied_ps:
                    success_msg = f'{cur_scenario.name} computed successfully including:'
                    if applied_ltd:
                        success_msg += ' LTD,'
                    if left_cs:
                        success_msg += ' left overbank CS,'
                    if main_cs:
                        success_msg += ' main channel CS, '
                    if left_cs:
                        success_msg += ' right overbank CS, '
                    if applied_ps:
                        success_msg += ' pier scour, '
                    if applied_labs:
                        success_msg += ' left abutment, '
                    if applied_rabs:
                        success_msg += ' right abutment, '

                    self.warnings['success'] = success_msg
                self.apply_lateral_limits(cur_scenario, self.plot_dict[name], x_data, bridge_geometry,
                                          add_min_scour_lateral, add_left_abut_min_scour_lateral,
                                          add_right_abut_min_scour_lateral)
        return True

    def adjust_approach_thalweg_to_match_cross_section(self, plot_options, thalweg_station):
        """Adjust the approach thalweg to match the cross section.

        Args:
            plot_options (PlotOptions): The plot options.
            thalweg_station (float): The station of the thalweg of the bridge cross-section.
        """
        approach_series = self.get_series_by_name(plot_options, 'Approach cross-section')
        x_data = approach_series['x var'].get_val()
        y_data = approach_series['y var'].get_val()
        app_thalweg_elevation = min(y_data)
        app_thalweg_station = x_data[y_data.index(app_thalweg_elevation)]
        thalweg_diff = thalweg_station - app_thalweg_station

        if thalweg_diff != 0.0:
            approach_series['x var'].set_val([x + thalweg_diff for x in x_data])

    def apply_lateral_limits(self, cur_scenario, plot_dict, x_data, bridge_geometry,
                             add_min_scour_lateral, add_left_abut_min_scour_lateral,
                             add_right_abut_min_scour_lateral):
        """Apply the lateral limits.

        Args:
            cur_scenario (Scenario): The current scenario.
            plot_dict (PlotOptions): The plot options.
            x_data (list of float): The x data.
            bridge_geometry (BridgeGeometry): The bridge geometry.
            add_min_scour_lateral (bool): True if the minimum scour lateral is added.
            add_left_abut_min_scour_lateral (bool): True if the left abutment minimum scour lateral is added.
            add_right_abut_min_scour_lateral (bool): True if the right abutment minimum scour lateral is added.
        """
        lateral_subdict, _ = self.get_plot_subdict_and_key_by_name('Lateral potential limits of contraction and '
                                                                   'abutment scour', 'series', plot_dict=plot_dict)
        if not lateral_subdict:
            return

        lateral_subdict['x var'].set_val([])
        lateral_subdict['y var'].set_val([])

        min_scour = None
        add_min_scour_lateral = False
        if 'Lowest scour elevation' in cur_scenario.results:
            min_scour = cur_scenario.results['Lowest scour elevation']
            add_min_scour_lateral = True

        if add_min_scour_lateral:
            lateral_x = [min(x_data), max(x_data)]
            lateral_y = [min_scour, min_scour]
            lateral_subdict['x var'].set_val(lateral_x)
            lateral_subdict['y var'].set_val(lateral_y)
        if add_left_abut_min_scour_lateral:
            l_abut_min_scour = cur_scenario.results['Abutment scour']['Left abutment']['Total scour at abutment']
            lateral_x = [min(x_data), bridge_geometry['Left abutment toe station']]
            lateral_y = [l_abut_min_scour, l_abut_min_scour]
            lateral_subdict['x var'].set_val(lateral_x)
            lateral_subdict['y var'].set_val(lateral_y)
        if add_right_abut_min_scour_lateral:
            r_abut_min_scour = cur_scenario.results['Abutment scour']['Right abutment']['Total scour at abutment']
            lateral_x = [bridge_geometry['Right abutment toe station'], max(x_data)]
            lateral_y = [r_abut_min_scour, r_abut_min_scour]
            lateral_subdict['x var'].set_val(lateral_x)
            lateral_subdict['y var'].set_val(lateral_y)

    def get_series_by_name(self, plot_dict, name):
        """Get the series by name.

        Args:
            plot_dict (dict): The plot dictionary.
            name (str): The name of the series.

        Returns:
            PlotOptions: The plot options for the series.
        """
        if 'series' not in plot_dict:
            return None
        for index in plot_dict['series']:
            if plot_dict['series'][index]['Name'] == name:
                return plot_dict['series'][index]
        return None

    def get_the_approach_bridge_cross_section(self, bridge_geometry, plot_options):
        """Get the main bridge cross section.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.
            plot_options (PlotOptions): The plot options.

        Returns:
            True if the main bridge cross section is valid.
        """
        acs_plot_options = self.get_geometry(bridge_geometry, plot_options, 'Approach cross-section')

        if all(value == 0 for value in acs_plot_options['x var'].get_val()) and \
                all(value == 0 for value in acs_plot_options['y var'].get_val()):
            return False

        return True

    def get_the_main_bridge_cross_section(self, bridge_geometry, plot_options):
        """Get the main bridge cross section.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.
            plot_options (PlotOptions): The plot options.

        Returns:
            True if the main bridge cross section is valid.
        """
        bcs_plot_options = self.get_geometry(bridge_geometry, plot_options, 'Bridge cross-section')

        if all(value == 0 for value in bcs_plot_options['x var'].get_val()) and \
                all(value == 0 for value in bcs_plot_options['y var'].get_val()):
            return False

        return True

    def get_the_bridge_deck(self, bridge_geometry, plot_dict):
        """Get the bridge deck.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.
            plot_options (PlotOptions): The plot options.

        Returns:
            True if the bridge deck is valid.
        """
        bd_plot_options = self.get_geometry(bridge_geometry, plot_dict, 'Bridge deck geometry')

        if 'x var' not in bd_plot_options or 'y var' not in bd_plot_options or bd_plot_options['x var'] is None or \
                bd_plot_options['y var'] is None:
            return False

        if all(value == 0 for value in bd_plot_options['x var'].get_val()) and \
                all(value == 0 for value in bd_plot_options['y var'].get_val()):
            return None

        return True

    def get_geometry(self, bridge_geometry, original_plot_dict, name):
        """Get the geometry.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.
            original_plot_dict (PlotOptions): The original plot options.
            name (str): The name of the geometry.

        Returns:
            PlotOptions: The plot options.
        """
        if name not in bridge_geometry:
            return None
        bg_geom, _ = self.get_plot_subdict_and_key_by_name(name, 'series', plot_dict=self.plot_dict['Bridge geometry'])
        _, orig_key = self.get_plot_subdict_and_key_by_name(name, 'series', plot_dict=original_plot_dict)
        if 'series' not in original_plot_dict:
            original_plot_dict['series'] = {}
        original_plot_dict['series'][orig_key] = bg_geom

        return original_plot_dict['series'][orig_key]

    def get_the_pier_geometry(self, scenario_name):
        """Get the pier geometry.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            dict: The plot data.
        """
        result = True
        first_pier = True
        _, null_data = self.get_data('Null data')

        if 'Piers' not in self.plot_dict['Bridge geometry']:
            return False
        name = 'Pier geometry'
        for pier_index in self.plot_dict['Bridge geometry']['Piers']:
            bg_p_geom, _ = self.get_plot_subdict_and_key_by_name(
                name, 'series', plot_dict=self.plot_dict['Bridge geometry']['Piers'][pier_index])
            _, orig_key = self.get_plot_subdict_and_key_by_name(
                name, 'series', plot_dict=self.plot_dict, plot_name=scenario_name)
            if first_pier:
                self.plot_dict[scenario_name]['series'][orig_key] = copy.deepcopy(bg_p_geom)
                first_pier = False
                if len(bg_p_geom['x var'].get_val()) <= 1:
                    self.plot_dict[scenario_name]['series'][orig_key]['x var'].set_val([])
                    self.plot_dict[scenario_name]['series'][orig_key]['y var'].set_val([])
            else:
                if len(bg_p_geom['x var'].get_val()) > 1:
                    self.plot_dict[scenario_name]['series'][orig_key]['x var'].append(null_data)
                    self.plot_dict[scenario_name]['series'][orig_key]['y var'].append(null_data)
                    self.plot_dict[scenario_name]['series'][orig_key]['x var'].value_options.extend(
                        bg_p_geom['x var'].get_val())
                    self.plot_dict[scenario_name]['series'][orig_key]['x var'].value += len(
                        bg_p_geom['x var'].get_val()) + 1
                    self.plot_dict[scenario_name]['series'][orig_key]['y var'].value_options.extend(
                        bg_p_geom['y var'].get_val())
                    self.plot_dict[scenario_name]['series'][orig_key]['y var'].value += len(
                        bg_p_geom['y var'].get_val()) + 1

            if all(value == 0 for value in bg_p_geom['x var'].get_val()) and \
                    all(value == 0 for value in bg_p_geom['y var'].get_val()):
                result = False

        if orig_key in self.plot_dict[scenario_name]['series']:
            is_closed = PlotPresenter.check_if_geometry_is_closed(
                self.plot_dict[scenario_name]['series'][orig_key]['x var'].get_val(),
                self.plot_dict[scenario_name]['series'][orig_key]['y var'].get_val(),
                null_data)
            self.plot_dict[scenario_name]['series'][orig_key]['is_closed'] = is_closed

        return result
