"""BoreHoleManagerData Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.calculator_list import CalcOrVarlist
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.gradations.gradation_layer_data import GradationLayerData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.gradations.bore_hole_data import BoreHoleData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.gradations.bore_hole_manager_calc import BoreHoleManagerCalc


class BoreHoleManagerData(CalcData):
    """Provides a class that will define a class that manages a list of gradation sets."""

    locations = BoreHoleManagerCalc.locations

    def __init__(self, location='Contraction cross-section', app_data=None, model_name=None, project_uuid=None):
        """Initializes the Gradation Layer.

        Args:
            location (str): The location of the boreholes.
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Boreholes'
        self.type = 'BoreHoleManagerData'

        self.calculator = BoreHoleManagerCalc()

        # self.calc_support_plot = False

        self.input['Location'] = Variable('Location', "list", 1, BoreHoleManagerData.locations)
        self.input['Location'].set_val(location)

        self.input['Boreholes'] = \
            Variable('Boreholes', 'calc_list', CalcOrVarlist(BoreHoleData, default_name='borehole',
                                                             default_plural_name='boreholes',
                                                             select_one=False,
                                                             min_number_of_items=1, app_data=app_data,
                                                             model_name=model_name, project_uuid=project_uuid))

        approach_locations = ['Single gradation for entire cross-section', 'Channel and overbanks']
        self.input['Approach locations'] = Variable('Approach locations', "list", 0, approach_locations)
        self.input['Approach layer'] = Variable('Approach layer', 'calc', GradationLayerData(
            app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input['Left overbank layer'] = Variable('Left overbank layer', 'calc', GradationLayerData(
            app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input['Right overbank layer'] = Variable('Right overbank layer', 'calc', GradationLayerData(
            app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        # Intermediate --
        # self.compute_prep_functions = []
        self.intermediate_to_copy.extend(['x_data', 'y_data', 'min_y', 'bcs_plot_dict', 'pier_data', 'abutment_data'])
        self.x_data = [0.0]
        self.y_data = [0.0]
        self.min_y = 0.0

        self.bcs_plot_dict = None
        self.pier_data = None
        self.abutment_data = None

        self.warnings = []
        self.results = {}
        # self.results['result'] = Variable('result', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)
        self.results['Approach gradations'] = {}
        self.results['Boreholes'] = {}
        self.results['Number of layers'] = Variable('Number of layers', 'int', 0, [])
        self.results['Number of boreholes'] = Variable('Number of boreholes', 'int', 0, [])
        self.results['Data complete'] = Variable('Data Complete', 'bool', 0.0, [])

        # Plot options
        self.theme = self.get_theme()
        soil_color = self.theme['Plot soil color']

        self.plot_names = ['Borehole centerlines']
        self.plots = {}
        for name in self.plot_names:
            self.plots[name] = {}
            self.plots[name]['Plot name'] = name
            self.plots[name]['Legend'] = "best"

        station_var = Variable('Station', 'float_list', 0, [0.0], precision=2,
                               limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                               native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        elevation_var = Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                 limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                 native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        station_vs_elevation = {}
        station_vs_elevation['Station'] = station_var
        station_vs_elevation['Elevation'] = elevation_var
        input_dict = {}
        input_dict['Bridge cross-section'] = station_vs_elevation

        self.input['Plot options'] = {}
        self.input['Plot options'][self.plot_names[0]] = Variable(
            'Plot options', 'class', PlotOptions(self.plot_names[0],
                                                 input_dict=input_dict,
                                                 show_series=True,
                                                 app_data=self.app_data,
                                                 model_name=self.model_name,
                                                 project_uuid=self.project_uuid),
            complexity=1)

        # Lines
        self.input['Plot options'][self.plot_names[0]].get_val().set_plot_line_options(
            0, 'Borehole centerlines',
            line_intercept=None, line_alignment='vertical', line_color=soil_color, linetype='dash-dot',
            line_width=1.5, text_color=soil_color, text_alignment='bottom', multiple_labels=True)
        abutment_toe_color = self.theme['Plot abutment toe line color']
        self.input['Plot options'][self.plot_names[0]].get_val().set_plot_line_options(
            1, 'Abutment toe',
            line_intercept=None, line_alignment='vertical', line_color=abutment_toe_color, linetype='dashed',
            line_width=1.5, text_color=abutment_toe_color, text_alignment='top')

        # Bridge Geometry
        self.input['Plot options'][self.plot_names[0]].get_val().set_plot_series_options(
            related_index=0, index=0, x_axis='Station', y_axis='Elevation', name='Bridge cross-section')
        self.input['Plot options'][self.plot_names[0]].get_val().set_plot_series_options(
            related_index=0, index=1, x_axis='Station', y_axis='Elevation', name='Piers')

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns:
            input_group (dict): dictionary of input variables
        """
        input_group = {}
        # input_group['Location'] = self.input['Location']
        if self.input['Location'].get_val() == 'Approach cross-section':
            self.calc_support_plot = False
            self.plot_names = {}
            self.plots = {}
            input_group['Approach locations'] = self.input['Approach locations']
            if self.input['Approach locations'].get_val() == 'Single gradation for entire cross-section':
                self.input['Approach layer'].name = 'Approach layer'
                input_group['Approach layer'] = self.input['Approach layer']
            else:
                self.input['Approach layer'].name = 'Main channel layer'
                input_group['Left overbank layer'] = self.input['Left overbank layer']
                input_group['Approach layer'] = self.input['Approach layer']
                input_group['Right overbank layer'] = self.input['Right overbank layer']
        else:
            self.calc_support_plot = True
            self.plot_names = ['Borehole centerlines']
            for name in self.plot_names:
                self.plots[name] = {}
                self.plots[name]['Plot name'] = name
                self.plots[name]['Legend'] = "best"
            input_group['Boreholes'] = self.input['Boreholes']

        return input_group

    def get_gradation_dict(self, location):
        """Get the list of gradation sets."""
        # self.update_elevation_data()
        gradation_dict = {}
        for set in self.input['Boreholes'].value.item_list:
            if set.input['Location'].get_val() == location:
                gradation_dict[set.uuid] = set

        return gradation_dict

    def get_results_tab_group(self, unknown=None):
        """Get the results tab group.

        Args:
            unknown (variable): unknown variable

        Returns:
            results_vars (list of variables): results group of variables
        """
        results_vars = {}

        if self.input['Location'].get_val() == 'Approach cross-section':
            results_vars['Approach gradations'] = self.results['Approach gradations']
        else:
            results_vars['Boreholes'] = self.results['Boreholes']

        return results_vars
