"""CalcData for performing Contraction Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.contraction_scour_calc import ContractionScourCalc
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_data import ScourBaseData


class ContractionScourData(ScourBaseData):
    """A class that defines a contraction scour at a bridge contraction."""

    # # Data to interpolate from figure 6.8 in HEC 18 April 2012, page 6.11 (pdf page 147)
    # t0_w = [
    #     0.001, 0.002, 0.003, 0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.01,
    #     0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.1, 0.15, 0.2, 0.3,
    #     0.4, 0.5
    # ]
    # t0_d = [
    #     0.04, 0.056, 0.069, 0.079, 0.09, 0.1, 0.109, 0.118, 0.126, 0.135, 0.21,
    #     0.29, 0.365, 0.45, 0.535, 0.62, 0.7, 0.78, 0.88, 1.45, 2.17, 4.3, 7.6,
    #     14
    # ]

    # t20_w = [
    #     0.001, 0.002, 0.003, 0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.01,
    #     0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.1, 0.15, 0.2, 0.3,
    #     0.4, 0.5
    # ]
    # t20_d = [
    #     0.03, 0.043, 0.053, 0.062, 0.069, 0.077, 0.083, 0.092, 0.099, 0.106,
    #     0.169, 0.228, 0.29, 0.36, 0.43, 0.495, 0.57, 0.66, 0.75, 1.27, 2.0,
    #     4.0, 7.4, 14
    # ]

    # t40_w = [
    #     0.001, 0.002, 0.003, 0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.01,
    #     0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.1, 0.15, 0.2, 0.3,
    #     0.4, 0.5
    # ]
    # t40_d = [
    #     0.025, 0.035, 0.043, 0.05, 0.056, 0.063, 0.069, 0.076, 0.08, 0.088,
    #     0.14, 0.19, 0.25, 0.31, 0.37, 0.43, 0.51, 0.59, 0.67, 1.2, 1.85, 3.85,
    #     7.15, 14
    # ]

    def __init__(self, clear_my_own_results=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the GVF calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        _, null_data = self.get_setting('Null data', model_name=model_name, project_uuid=project_uuid)
        _, zero_tol = self.get_setting('Zero tolerance', model_name=model_name, project_uuid=project_uuid)

        self.calculator = ContractionScourCalc(null_data=null_data, zero_tol=zero_tol,)

        # _, null_data = self.get_setting('Null data')
        # _, zero_tol = self.get_setting('Zero tolerance')
        # self.t0_interpolate = Interpolation(ContractionScourCalc.t0_d, ContractionScourCalc.t0_w,
        #                                     app_data=self.app_data,
        #                                     null_data=null_data, zero_tol=zero_tol)
        # self.t20_interpolate = Interpolation(ContractionScourCalc.t20_d, ContractionScourCalc.t20_w,
        #                                      app_data=self.app_data,
        #                                      null_data=null_data, zero_tol=zero_tol)
        # self.t40_interpolate = Interpolation(ContractionScourCalc.t40_d, ContractionScourCalc.t40_w,
        #                                      app_data=self.app_data,
        #                                      null_data=null_data, zero_tol=zero_tol)

        self.name = "Contraction Scour Calculator"
        self.type = "ContractionScourCalc"
        self.class_name = 'Contraction Scour Calculator'

        self.calc_support_dual_input = False

        k_options = ['calculate k exponent', 'specify k exponent']

        # Input
        self.input['Compute pressure flow'] = Variable('Compute pressure flow', 'bool', False)

        # Cohesive
        self.input['Average velocity in contracted section (v2)'] = Variable(
            'Average velocity in contracted section (v2)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['Velocity'],
            native_unit='ft/s',
            us_units=self.us_velocity,
            si_units=self.si_velocity)

        _, rho_w = self.get_setting('Water density', 1.94)
        self.input['Water density (ρw)'] = Variable(
            'Water density (ρw)',
            'float',
            rho_w, [],
            precision=2,
            unit_type=['specific_weight'],
            native_unit='slug/ft^3',
            us_units=[['slug/ft^3']],
            si_units=[['kN/m^3', 'N/m^3']])

        self.input["Manning's n (n)"] = Variable(
            "Manning's n (n)", 'float', 0.0, [], precision=4, unit_type=['coefficient'])

        self.input['Scour from single event'] = {}
        self.input['Scour from single event']['Initial rate of scour'] = Variable(
            'Initial rate of scour (żi)',
            'float',
            0.0, [],
            precision=2,
            native_unit='ft/hr',
            us_units=[['ft/hr']],
            si_units=[['m/hr']])

        self.input['Scour from single event']['Duration of flow event (t)'] = Variable(
            'Duration of Flow (t)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['time'],
            native_unit='hr',
            us_units=[['day', 'hr', 'min']],
            si_units=[['day', 'hr', 'min']])

        # Determine clear-water or live-bed
        self.input['Average depth upstream (y1)'] = Variable(
            'Average depth upstream of contraction (y1)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='(hu) in pressure scour')

        self.input['Average velocity upstream (v1)'] = Variable(
            'Average velocity upstream of contraction (v1)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['Velocity'],
            native_unit='ft/s',
            us_units=self.us_velocity,
            si_units=self.si_velocity)

        # Clear Water & Live Bed
        self.input['Temperature of water'] = Variable(
            'Temperature of water',
            'float',
            60.0, [],
            precision=1,
            unit_type=['Temperature'],
            native_unit='°F',
            us_units=self.us_temperature,
            si_units=self.si_temperature,
            note='Used to determine the fall velocity')

        self.input['Slope of energy grade line (S1)'] = Variable(
            'Slope of energy grade line at approach section (S1)',
            'float',
            0.0, [],
            precision=6,
            unit_type=['slope'],
            native_unit='ft/ft',
            us_units=self.us_slope,
            si_units=self.si_slope)

        self.input['Discharge in approach section (Q1)'] = Variable(
            'Discharge in approach section that is transporting sediment (Q1)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['flow'],
            native_unit='cfs',
            us_units=self.us_flow,
            si_units=self.si_flow)

        self.input['Discharge in contracted section (Q2)'] = Variable(
            'Discharge in contracted section (Q2)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['flow'],
            native_unit='cfs',
            us_units=self.us_flow,
            si_units=self.si_flow)

        self.input['Bottom width in approach section (W1)'] = Variable(
            'Bottom width in approach section (W1)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.input['Bottom width in contracted section (W2)'] = Variable(
            'Bottom width in contracted section (W2)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='Width should exclude pier widths')

        self.input[
            'Depth prior to scour in contracted section (y0)'] = Variable(
                'Depth prior to scour in contracted section (y0)',
                'float',
                0.0, [],
                precision=2,
                unit_type=['length'],
                native_unit='ft',
                us_units=self.us_mid_length,
                si_units=self.si_mid_length)

        self.input['k exponent preference'] = Variable('k exponent preference',
                                                       "list",
                                                       0,
                                                       k_options,
                                                       complexity=2)
        self.input['k exponent'] = Variable('k exponent',
                                            'float',
                                            0.0,
                                            precision=2,
                                            limits=(0.5, 0.69),
                                            unit_type='coefficient')

        self.input['Pressure scour parameters'] = {}
        self.input['Pressure scour parameters']['Depth prior to scour at bridge (hb + ht or hu)'] = Variable(
            'Average depth approaching bridge (hb + ht or hu)', 'float', 0.0, [], precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
            note='To be measured between WSE and channel close to the upstream face of the bridge, not within '
            'pressure zone')
        self.input['Pressure scour parameters']['Vertical size of bridge opening (hb)'] = Variable(
            'Average, vertical bridge opening prior to scour (hb)', 'float', 0.0, [], precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
            note='To be measured between low chord of bridge and average channel bed elevations'
        )
        self.input['Pressure scour parameters']['Deck thickness (t)'] = Variable(
            'Deck thickness (T)', 'float', 0.0, [], precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
            note=''
        )

        self.input['k exponent'] = Variable('k exponent',
                                            'float',
                                            0.0,
                                            precision=2,
                                            limits=(0.5, 0.69),
                                            unit_type='coefficient')

        _, gamma_w = self.get_setting('Unit weight of water (γw)')
        _, gamma_s = self.get_setting('Unit weight of soil (γs)')

        self.input['Unit weight of water (γw)'] = Variable(
            'Unit weight of water (γw)',
            'float',
            gamma_w,
            precision=4,
            unit_type=["specific_weight"],
            native_unit="lb/ft^3",
            us_units=[["lb/ft^3"]],
            si_units=[["kN/m^3", "N/m^3"]])

        self.input['Unit weight of soil (γs)'] = Variable(
            'Unit weight of soil (γs)',
            'float',
            gamma_s,
            precision=4,
            unit_type=["specific_weight"],
            native_unit="lb/ft^3",
            us_units=[["lb/ft^3"]],
            si_units=[["kN/m^3", "N/m^3"]])

        self.unknown = None

        # Intermediate
        # self.compute_prep_functions = []
        self.intermediate_to_copy.extend([
            'wse_x', 'wse_y', 'scour_depth', 'critical_velocity', 'is_upstream_transporting_sediment',
            'live_bed_scour_depth', 'clear_water_scour_depth', 'cohesive_scour_depth', 'applied_shear',
            'critical_shear', 'longterm_scour_depth', 'ht', 'hw', 'hue', 'que', 'q2c', 'con_decay_x', 'con_decay_y',
            'con_decay_x_markers', 'con_decay_y_markers', 'can_compute_transporting_sediment', 'can_compute_live_bed',
            'can_compute_clear_water', 'can_compute_pressure', 'is_computing_shear_decay', 'centerline',
            'centerline_streambed', 'channel_location', 'scour_reference_point', 'bh_dict', 'bh_uuid', 'bh',
            'bh_layers', 'layer_index', 'surface_d50', ])
        # self.unit_converter = ConversionCalc(app_data=app_data)

        self.critical_velocity = 0.0
        self.is_upstream_transporting_sediment = False

        self.live_bed_scour_depth = 0.0
        self.clear_water_scour_depth = 0.0
        self.cohesive_scour_depth = 0.0

        self.applied_shear = 0.0
        self.critical_shear = 0.0

        self.longterm_scour_depth = 0.0

        self.ht = 0.0
        self.hw = 0.0
        self.hue = 0.0
        self.que = 0.0
        self.q2c = 0.0

        self.can_compute_transporting_sediment = False
        self.can_compute_live_bed = False
        self.can_compute_clear_water = False
        self.can_compute_pressure = False

        # Results
        self.clear_my_own_results = clear_my_own_results

        # Approach Gradation
        self.results['Average scour depth (y2)'] = Variable(
            'Average scour depth (y2)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Governing scour method'] = Variable(
            'Governing scour method',
            "string",
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        # Cohesive
        self.results['Initial shear stress (τ)'] = Variable(
            'Initial shear stress (τ)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['force'],
            native_unit='lb/ft^2',
            us_units=[['lb/ft^2']],
            si_units=[['kN/m^2', 'N/m^2']])

        self.results['Partial scour depth'] = Variable(
            'Scour depth from flow event (y(t))',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Cohesive scour depth (ys)'] = Variable(
            'Cohesive scour depth (ys)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Scour depth (ys)'] = Variable('Estimated scour depth (ys)',
                                                    'float',
                                                    0.0, [],
                                                    precision=2,
                                                    unit_type=['length'],
                                                    native_unit='ft',
                                                    us_units=self.us_mid_length,
                                                    si_units=self.si_mid_length)

        self.results['Average live-bed scour depth (y2)'] = Variable(
            'Average live-bed scour depth (y2)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Live-bed scour depth (ys)'] = Variable(
            'Live-bed scour depth (ys)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Average clear-water scour depth (y2)'] = Variable(
            'Average live-bed scour depth (y2)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Clear-water scour depth (ys)'] = Variable(
            'Live-bed scour depth (ys)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Critical velocity (vc)'] = Variable(
            'critical velocity of upstream sediment (vc)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Upstream transporting sediment'] = Variable(
            'Is upstream transporting sediment', "bool", False)

        self.results['Applied shear (ϑ0)'] = Variable(
            'Average bed shear stess, contracted section (ϑ0)',
            'float',
            0.0, [],
            precision=4,
            unit_type=['stress'],
            native_unit='psf',
            us_units=[['psf']],
            si_units=[['pa']])

        self.results['Critical shear (ϑc)'] = Variable(
            'critical bed shear stress at incipient motion (ϑc)',
            'float',
            0.0, [],
            precision=4,
            unit_type=['stress'],
            native_unit='psf',
            us_units=[['psf']],
            si_units=[['pa']])

        self.results['Shear velocity (v*)'] = Variable(
            'shear velocity upstream (v*)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['Velocity'],
            native_unit='ft/s',
            us_units=self.us_velocity,
            si_units=self.si_velocity)

        self.results['Fall velocity (t)'] = Variable(
            'fall velocity of upstream D50 (t)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['Velocity'],
            native_unit='ft/s',
            us_units=self.us_velocity,
            si_units=self.si_velocity)

        self.results['k exponent'] = Variable('k exponent',
                                              'float',
                                              0.0,
                                              precision=2,
                                              limits=(0.5, 0.69),
                                              unit_type='coefficient')
        self.results['k exponent description'] = Variable(
            'k exponent', 'string', '')

        self.results['Distance from the WSE to the low chord of the bridge (ht)'] = Variable(
            'Distance from the WSE to the low chord of the bridge (ht)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='A zero value indicates the WSE is below the low chord of the bridge')

        self.results['Weir flow height (hw)'] = Variable(
            'Weir flow height (hw)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Effective upstream channel flow depth (hue)'] = Variable(
            'Effective upstream channel flow depth (hue)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Effective upstream channel flow (que)'] = Variable(
            'Effective upstream channel flow (que)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Flow separation thickness (t)'] = Variable(
            'Flow separation thickness (t)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Pressure scour depth (y2)'] = Variable(
            'Pressure scour depth (y2)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Unit conversion constant (Ku)'] = Variable(
            'Unit conversion constant (Ku)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['coefficient'])

        self.warnings = []

        # # Reference Manuals
        # self.reference_pdfs['HEC-18'] = (
        #     'HEC-18, 5th Edition - Evaluating Scour at Bridges.pdf', 137)

        shear_var = Variable('Shear stress', 'float_list', 0, [0.0], precision=2,
                             limits=(0.0, sys.float_info.max), unit_type=['stress'],
                             native_unit='psf', us_units=[['psf']], si_units=[['pa']],)

        elevation_var = Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                 limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                 native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        shear_vs_elevation = {}
        shear_vs_elevation['Shear stress'] = shear_var
        shear_vs_elevation['Elevation'] = elevation_var

        input_dict = {}
        input_dict['Shear decay'] = shear_vs_elevation

        self.input['Plot options'] = {}
        self.input['Plot options']['Shear decay'] = Variable('Shear decay', 'class',
                                                             PlotOptions('Shear decay',
                                                                         input_dict=input_dict,
                                                                         show_series=True, app_data=app_data,
                                                                         model_name=model_name,
                                                                         project_uuid=project_uuid),
                                                             complexity=1)

        name = 'Shear decay'
        shear_decay_color = (128, 0, 128)  # Deep Violet
        # Other options: Deep Violet: (128, 0, 128), Amber/Gold: (255, 170, 0), Orange-Red: (255, 85, 0),
        # Magenta: (220, 20, 140)
        self.input['Plot options'][name].get_val().set_plot_series_options(
            related_index=0, index=0, name=name, x_axis='Shear stress', y_axis='Elevation',
            line_color=shear_decay_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=shear_decay_color)  # need x_var - it is None

        self.input['Plot options'][name].get_val().set_plot_point_options(
            related_index=0, index=0, name=name, point_color=shear_decay_color,)
        #    point_marker=point_marker, marker_size=marker_size)

        grey_rgb = (127, 127, 127)
        self.input['Plot options'][name].get_val().set_plot_line_options(
            index=0, name='Y max', line_alignment='horizontal',
            line_color=grey_rgb, linetype='dotted', line_width=1.5, text_color=grey_rgb,
            labels='Y max')

        self.plot_names = ['Shear decay']
        self.plots = {}
        for name in self.plot_names:
            self.plots[name] = {}
            self.plots[name]['Plot name'] = name
            self.plots[name]['Legend'] = "best"

    def check_for_needed_data(self):
        """Checks for needed data for the calculator to run.

        Else, show popup with warnings and do not run dialog.
        """
        self.warnings = []

        if self.gradations is None:
            return True

        # _, zero_tol = self.get_setting('Zero tolerance')
        # Check for at least one contracted gradation with a D50
        self.update_gradation_lists()

        # self.contracted_bhs = None
        # self.contracted_bh = None
        if self.gradations.input['Contracted gradation input options'].get_val() == 'Single borehole':
            if self.calculator.contracted_bh is None:
                self.warnings.append("Please define a borehole before defining the contraction scour.")
                return False
        elif self.gradations.input['Contracted gradation input options'].get_val() == 'Multiple boreholes':
            if self.calculator.contracted_bhs is None or len(self.calculator.contracted_bhs.input[
                    'Boreholes'].value.item_list) <= 0:
                self.warnings.append("Please define a borehole before defining the contraction scour.")
                return False
        else:  # if self.gradations.input['Contracted gradation input options'].get_val() == 'Single D50':
            if self.d50 <= 0.0:
                self.warnings.append("Please define a D50 before defining the contraction scour.")
                return False

        if self.is_computing_shear_decay:
            self.calc_support_plot = True
        else:
            self.calc_support_plot = False

        # if self.contracted_gradation_dict is None or len(self.contracted_gradation_dict) <= 0:
        #     self.warnings.append("Please enter a gradation before defining the contraction scour.")
        #     return False
        # d50_or_tc_above_zero = False
        # for gradation in self.contracted_gradation_dict:
        #     for borehole in self.contracted_gradation_dict[gradation].input['Boreholes'].get_val().item_list:
        #         if borehole.get_d50() > zero_tol:
        #             d50_or_tc_above_zero = True
        #             break
        #         if borehole.get_critical_shear_stress() > zero_tol:
        #             d50_or_tc_above_zero = True
        #             break
        # if not d50_or_tc_above_zero:
        #     self.warnings.append('Please enter a D50 for a contracted gradation before defining the '
        #                          'contraction scour.')
        #     return False

        return True

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        if self.gradations:
            self.update_gradation_lists()
        else:
            self.d50 = self.input['Contracted D50'].get_val()
            self.upstream_d50 = self.input['Approach D50'].get_val()

        # if self.gradations:
        #     if len(self.approach_gradation_dict) <= 0:
        #         self.update_gradation_lists()
        # else:
        #     # Update D50s for gradation or user-specified
        #

        # if self.gradations:
        #     approach_gradation_dict = self.gradations.get_gradation_dict(GradationManagerData.locations[0])
        #     self.contracted_gradation_dict = self.gradations.get_gradation_dict(GradationManagerData.locations[1])

        #     # Allow the user to select None on the approach gradation
        #     none_item = {"None": None}
        #     self.approach_gradation_dict = {**none_item, **approach_gradation_dict}

        #     approach_dict = []
        #     contracted_dict = []
        #     for gradation in self.approach_gradation_dict:
        #         approach_dict[gradation] = self.approach_gradation_dict[gradation].name
        #     for gradation in self.contracted_gradation_dict:
        #         contracted_dict[gradation] = self.contracted_gradation_dict[gradation].name

        #     self.input['Approach gradation'].value_options = approach_dict
        #     self.input['Contracted gradation'].value_options = contracted_dict

        #     if len(contracted_dict) > 0:
        #         if self.input['Approach gradation'].get_val() not in approach_dict:
        #             self.input['Approach gradation'].set_val('None')

        #         if self.input['Contracted gradation'].get_val() not in contracted_dict:
        #             self.input['Contracted gradation'].set_val(next(iter(contracted_dict)))

        #         self.d50 = self.approach_gradation_dict[self.input['Approach gradation'].get_val()].get_d50
        #         if self.input['Approach gradation'].get_val() != 'None':
        #             self.upstream_d50 = self.approach_gradation_dict[self.input['Approach gradation'].get_val()].
        # get_d50

        input_vars = copy.deepcopy(self.input)

        # We use this internally; user does not need to see it
        input_vars.pop('Scour hole geometry')

        if self.clear_my_own_results:
            # This means we are a stand alone calculator and we pop the gradation selections
            # input_vars.pop('Approach gradation')
            input_vars.pop('Selected approach location')
            # input_vars.pop('Contracted gradation')
            input_vars.pop('Selected borehole')
        else:
            # This means we are part of a bridge project and we pop the D50s
            input_vars.pop('Approach D50')
            input_vars.pop('Contracted D50')
            input_vars.pop('Critical shear stress (τc)')
            # sel_approach_uuid = self.input['Approach gradation'].get_val()
            # if sel_approach_uuid is not None and sel_approach_uuid != 'None':
            #     approach_locations = self.approach_gradation_dict[sel_approach_uuid].input[
            #         'Approach locations'].get_val()
            #     if approach_locations == 'Single gradation for entire cross-section':
            #         input_vars.pop('Selected approach location')
            # else:
            #     input_vars.pop('Selected approach location')
            # input_vars.pop('Unit weight of soil (γs)')  # Defined in gradation
            if self.gradations.input['Approach gradation input options'].get_val() == 'Single D50' or \
                    self.gradations.input['Approach gradations'].get_val().input['Approach locations'].get_val() == \
                    'Single gradation for entire cross-section':
                input_vars.pop('Selected approach location')
            if self.gradations.input['Contracted gradation input options'].get_val() == 'Single D50' or \
                    self.gradations.input['Contracted gradation input options'].get_val() == 'Single borehole':
                input_vars.pop('Selected borehole')

        # Check if the user is specifying the k exponent
        if self.sys_complexity < self.input['k exponent preference'].complexity or \
                self.input['k exponent preference'].get_val() != 'specify k exponent':
            if 'k exponent' in input_vars:
                input_vars.pop('k exponent')

        # Check if it is cohesive
        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            input_vars.pop('Average velocity upstream (v1)')
            input_vars.pop('Temperature of water')
            input_vars.pop('Slope of energy grade line (S1)')
            input_vars.pop('Discharge in approach section (Q1)')
            input_vars.pop('Discharge in contracted section (Q2)')
            input_vars.pop('Bottom width in approach section (W1)')
            input_vars.pop('Depth prior to scour in contracted section (y0)')
        else:
            input_vars.pop('Average velocity in contracted section (v2)')
            input_vars.pop('Water density (ρw)')
            input_vars.pop("Manning's n (n)")
            input_vars.pop('Scour from single event')

        if not self.input['Compute pressure flow'].get_val():
            input_vars.pop('Pressure scour parameters')

        self.pop_property_vars(input_vars)

        return input_vars

    def get_plot_options(self):
        """Get the plot options.

        Returns:
            dict: The plot options.
        """
        input_vars = copy.deepcopy(self.input['Plot options'])

        # Remove these items, because we get those plot options from options defined with the geometry
        plot_not_hide_list = ['Shear decay curve']
        for data_item in input_vars['Data series'].value.item_list:
            if data_item.name not in plot_not_hide_list:
                data_item.complexity = 6  # Hide these items
            for series_item in data_item.input['Plot series'].get_val().item_list:
                if series_item.name in plot_not_hide_list:
                    series_item.complexity = 6  # Hide these items

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        results_vars = {}

        basic_results = {}
        complete_results = {}

        if self.gradations:
            # Cohesive
            if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
                basic_results['Critical shear stress (τc)'] = self.results['Results']['Critical shear stress (τc)']
                complete_results['Critical shear stress (τc)'] = self.results['Results']['Critical shear stress (τc)']
            else:
                basic_results['Approach D50'] = self.results['Results']['Approach D50']
                complete_results['Approach D50'] = self.results['Results']['Approach D50']
                basic_results['Contracted D50'] = self.results['Results']['Contracted D50']
                complete_results['Contracted D50'] = self.results['Results']['Contracted D50']

        if not self.can_compute:
            if not (self.d50 <= 0.0 and self.critical_shear_stress > 0.0):
                if self.can_compute_transporting_sediment:
                    self.is_upstream_transporting_sediment

                    basic_results['Critical velocity (vc)'] = self.results[
                        'Critical velocity (vc)']
                    basic_results['Upstream transporting sediment'] = self.results[
                        'Upstream transporting sediment']
                    complete_results = basic_results
                    results_vars['Basic Results'] = basic_results
                    results_vars['Complete Results'] = complete_results

            results_vars['Basic Results'] = basic_results
            results_vars['Complete Results'] = complete_results
            return results_vars

        # Cohesive
        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            complete_results['Cohesive scour depth (ys)'] = self.results['Cohesive scour depth (ys)']
            basic_results['Initial shear stress (τ)'] = self.results['Initial shear stress (τ)']
            complete_results['Initial shear stress (τ)'] = self.results['Initial shear stress (τ)']
            if self.input['Scour from single event']['Duration of flow event (t)'].get_val() > 0.0 and \
                    self.cohesive_scour_depth > 0.0:
                basic_results['Partial scour depth'] = self.results['Partial scour depth']
                complete_results['Partial scour depth'] = self.results['Partial scour depth']

        else:
            basic_results['Critical velocity (vc)'] = self.results[
                'Critical velocity (vc)']
            complete_results['Critical velocity (vc)'] = self.results[
                'Critical velocity (vc)']
            basic_results['Upstream transporting sediment'] = self.results[
                'Upstream transporting sediment']
            complete_results['Upstream transporting sediment'] = self.results[
                'Upstream transporting sediment']
            if self.can_compute_live_bed:
                complete_results['Shear velocity (v*)'] = self.results[
                    'Shear velocity (v*)']
                complete_results['Fall velocity (t)'] = self.results[
                    'Fall velocity (t)']
                complete_results['k exponent'] = self.results['k exponent']
                complete_results[
                    'Average live-bed scour depth (y2)'] = self.results[
                        'Average live-bed scour depth (y2)']
                complete_results['Applied shear (ϑ0)'] = self.results[
                    'Applied shear (ϑ0)']
                complete_results['Critical shear (ϑc)'] = self.results[
                    'Critical shear (ϑc)']
                complete_results['Live-bed scour depth (ys)'] = self.results[
                    'Live-bed scour depth (ys)']
            if self.can_compute_clear_water:
                complete_results['Average clear-water scour depth (y2)'] = \
                    self.results['Average clear-water scour depth (y2)']
                complete_results['Clear-water scour depth (ys)'] = self.results[
                    'Clear-water scour depth (ys)']

        if self.input['Compute pressure flow'].get_val() and self.can_compute_pressure:
            basic_results['Pressure scour'] = {}
            complete_results['Pressure scour'] = {}
            complete_results['Pressure scour']['Weir flow height (hw)'] = self.results['Pressure scour'][
                'Weir flow height (hw)']
            complete_results['Pressure scour']['Effective upstream channel flow depth (hue)'] = self.results[
                'Pressure scour']['Effective upstream channel flow depth (hue)']
            complete_results['Pressure scour']['Effective upstream channel flow (que)'] = self.results[
                'Pressure scour']['Effective upstream channel flow (que)']
            if 'Flow separation thickness (t)' in self.results['Pressure scour']:
                complete_results['Pressure scour']['Flow separation thickness (t)'] = self.results['Pressure scour'][
                    'Flow separation thickness (t)']
                basic_results['Pressure scour']['Flow separation thickness (t)'] = self.results['Pressure scour'][
                    'Flow separation thickness (t)']
            if 'Pressure scour depth (y2)' in self.results['Pressure scour']:
                complete_results['Pressure scour']['Pressure scour depth (y2)'] = self.results['Pressure scour'][
                    'Pressure scour depth (y2)']
                basic_results['Pressure scour']['Pressure scour depth (y2)'] = self.results['Pressure scour'][
                    'Pressure scour depth (y2)']

        basic_results['Governing scour method'] = self.results[
            'Governing scour method']
        complete_results['Governing scour method'] = self.results[
            'Governing scour method']
        basic_results['Scour depth (ys)'] = self.results['Scour depth (ys)']
        complete_results['Scour depth (ys)'] = self.results['Scour depth (ys)']

        if self.is_computing_shear_decay:
            basic_results['Shear decay results'] = copy.deepcopy(self.results['Shear decay results'])
            basic_results['Shear decay results'].pop('Shear decay curve shear')
            basic_results['Shear decay results'].pop('Shear decay curve elevation')
            basic_results['Shear decay results'].pop('Shear decay curve shear markers')
            basic_results['Shear decay results'].pop('Shear decay curve elevation markers')

            basic_results['Shear decay results'].pop('Computed Hager number')
            basic_results['Shear decay results'].pop('Max depth computed from Hager number')
            basic_results['Shear decay results'].pop('Max depth computed from CSU equation')
            basic_results['Shear decay results'].pop('Unadjusted computed local shear')
            complete_results['Shear decay results'] = copy.deepcopy(basic_results['Shear decay results'])

        results_vars['Basic Results'] = basic_results
        results_vars['Complete Results'] = complete_results

        if self.is_computing_shear_decay:
            results_vars['Shear decay table'] = {}
            results_vars['Shear decay table']['Shear decay curve shear'] = \
                copy.deepcopy(self.results['Shear decay results']['Shear decay curve shear'])
            results_vars['Shear decay table']['Shear decay curve elevation'] = \
                copy.deepcopy(self.results['Shear decay results']['Shear decay curve elevation'])

        return results_vars

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        return self.warnings
