"""CalcData for performing Contraction Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.units.unit_conversion import ConversionCalc
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.long_term_calc import LongTermDegradationCalc
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_data import ScourBaseData


class LongTermDegradationData(ScourBaseData):
    """A class that defines the long-term degradation at a bridge contraction."""

    def __init__(self, clear_my_own_results=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the long-term calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            app_data (AppData): The application data object.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Long-term Degradation CalcData"
        self.type = "LongTermDegradationCalc"

        self.calc_support_dual_input = False

        self.calculator = LongTermDegradationCalc()

        # Input
        calc_options = ['Specify LTD depth',]
        self.input['Calculation method'] = Variable('Calculation method', "list", 0, calc_options)

        self.input['Specified depth of long-term degradation'] = \
            Variable('Specified depth of long-term degradation', 'float', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.unknown = None

        # Intermediate
        # self.compute_prep_functions = []  # Functions to call before compute_data
        self.intermediate_to_copy.extend(['unit_converter'])  # Define variables to copy to the calculator

        # Results
        self.clear_my_own_results = clear_my_own_results

        self.results['Results']['Long-term degradation'] = \
            Variable('Long-term degradation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
                     us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.warnings = []

        self.unit_converter = ConversionCalc(app_data)

    def check_for_needed_data(self):
        """Checks for needed data for the calculator to run.

        Else, show popup with warnings and do not run dialog.
        """
        if self.is_computing_shear_decay:
            self.calc_support_plot = True
        else:
            self.calc_support_plot = False

        return True

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        if self.gradations:
            self.update_gradation_lists()

        if self.input['Calculation method'].get_val() == 'Specify LTD depth':
            input_vars['Calculation method'] = self.input['Calculation method']
            input_vars['Specified depth of long-term degradation'] = \
                self.input['Specified depth of long-term degradation']
        # else:
        #     input_vars = copy.deepcopy(self.input)

        #     # We use this internally; user does not need to see it
        #     input_vars.pop('Scour hole geometry')

        #     if self.clear_my_own_results:
        #         # This means we are a stand alone calculator and we pop the gradation selections
        #         input_vars.pop('Selected approach location')
        #         input_vars.pop('Selected borehole')
        #     else:
        #         # This means we are part of a bridge project and we pop the D50s
        #         input_vars.pop('Approach D50')
        #         input_vars.pop('Contracted D50')

        #         if self.gradations.input['Approach gradation input options'].get_val() == 'Single D50' or \
        #                 self.gradations.input['Approach gradations'].get_val().input[
        #                     'Approach locations'].get_val() == 'Single gradation for entire cross-section':
        #             input_vars.pop('Selected approach location')
        #         if self.gradations.input['Contracted gradation input options'].get_val() == 'Single D50' or \
        #                 self.gradations.input['Contracted gradation input options'].get_val() == 'Single borehole':
        #             input_vars.pop('Selected borehole')

        self.pop_property_vars(input_vars)

        return input_vars

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        results_vars = {}

        basic_results = {}
        complete_results = {}

        basic_results['Scour depth (ys)'] = self.results['Results']['Long-term degradation']
        complete_results['Scour depth (ys)'] = self.results['Results']['Long-term degradation']

        results_vars['Basic Results'] = basic_results
        results_vars['Complete Results'] = complete_results

        return results_vars

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        return self.warnings

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation."""
        if self.clear_my_own_results:
            self.warnings = []
