"""Pile group Data."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.pier.Complex.pile_group_calc import PileGroupCalc


class PileGroup(VariableGroup):
    """Provides a class that will Handle input for complex pier pile caps."""
    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Specify Pile Caps.

        Args:
            app_data (AppData): The application data (settings).
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.calculator = PileGroupCalc()

        self.input = {}

        self.input['Compute pile group'] = Variable('Compute pile group scour component', "bool", True)

        pile_group_shape_options = ['Square Nose', 'Round Nose']
        self.input['Pile nose shape (K1)'] = Variable('Pile nose shape (K1)', "list", 0, pile_group_shape_options)
        self.input['Pile width (a)'] = Variable('Pile width (a)', 'float', 0.0, [], precision=2,
                                                unit_type=['length'], native_unit='ft',
                                                us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Pile group number of rows (m)'] = Variable('Pile group number of rows (m)', 'int', 0, [],
                                                               note='rows travel parallel to the bridge')
        self.input['Pile group number of columns (n)'] = Variable('Pile group number of columns (n)', 'int', 0, [],
                                                                  note='columns travel perpendicular to the bridge')
        self.input['Pile group spacing between rows (sm)'] = Variable('Pile group spacing between rows (sm)',
                                                                      'float', 0.0, [], precision=2,
                                                                      unit_type=['length'], native_unit='ft',
                                                                      us_units=self.us_mid_length,
                                                                      si_units=self.si_mid_length,
                                                                      note='Spacing between center of pile to '
                                                                      'Center of pile')
        self.input['Pile group spacing between columns (sn)'] = Variable('Pile group spacing between columns (sn)',
                                                                         'float', 0.0, [], precision=2,
                                                                         unit_type=['length'], native_unit='ft',
                                                                         us_units=self.us_mid_length,
                                                                         si_units=self.si_mid_length,
                                                                         note='Spacing between center of pile to '
                                                                         'Center of pile')
        self.input['Staggered pile rows'] = Variable('Staggered pile rows', "bool", False)

        # Intermediate
        # self.compute_prep_functions = []  # Functions to call before compute_data
        # Define variables to copy to the calculator
        self.intermediate_to_copy.extend(['is_fdot_complex'])

        self.is_fdot_complex = False

    def get_input_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = {}

        if self.is_fdot_complex and not self.input['Compute pile group'].get_val():
            input_vars['Compute pile group'] = self.input['Compute pile group']
            return input_vars

        input_vars = copy.deepcopy(self.input)

        if not self.is_fdot_complex:
            input_vars.pop('Compute pile group')

        return input_vars
