"""Provides a class that will Handle user-selected for boundary conditions."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
# from pathlib import Path
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.user_array import UserArray
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.channel_geometry_data import \
    all_shapes, ChannelGeometryData, closed_shapes, open_shapes
from xms.HydraulicToolboxCalc.hydraulics.culvert.inlet_control.inlet_control_calc import InletControlCalc
from xms.HydraulicToolboxCalc.hydraulics.culvert.inlet_control.select_size import SelectSizeTable
from xms.HydraulicToolboxCalc.hydraulics.culvert.inlet_control.shape_db import ShapeDatabase
from xms.HydraulicToolboxCalc.hydraulics.manning_n.composite_manning_n_data import CompositeManningNData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.manning_n_data import ManningNData


class InletControlData(VariableGroup):
    """Provides a class that will Handle user-selected for boundary conditions.

    Boundary Conditions: Normal depth, critical depth, constant depth, rating curve, specified depths.
    """
    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Inlet Control Class.

        Args:
            app_data (AppData): The application data
            model_name (str): The name of the model
            project_uuid (str): The project uuid
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Inlet Control CalcData'
        self.type = 'InletControlCalc'

        self.calculator = InletControlCalc()

        self.update_inpute_after_compute = True

        # Input
        self.inlet_type_list = [
            'best available',
            'interpolation: circular or elliptical', 'interpolation: arch or embedded',
            'manual polynomial',
        ]
        self.input['Inlet type'] = Variable(
            'Inlet type', "list", 0, self.inlet_type_list, complexity=1,
            note='"best available" uses polynomial if available, otherwise interpolation, if shape is user-defined, '
            'arch/embedded interpolation is used')

        # self.input['Inlet configuration'] = Variable('Inlet configuration', "list", 0, self.inlet_configuration_list)

        # Computation variables
        self.input['Flows'] = Variable(
            'Flow(s)', 'UserArray', UserArray(2, ['flow'], 'cfs', us_units=self.us_flow, si_units=self.si_flow,
                                              select_name='specify flow(s) as:', name_append='flow'))

        allowed_shapes = 'closed'
        self.open_shapes = open_shapes
        self.closed_shapes = closed_shapes
        self.all_shapes = all_shapes
        self.shapes = self.all_shapes

        # if allowed_shapes == 'open':
        #     self.shapes = self.open_shapes
        if allowed_shapes == 'closed':
            self.shapes = self.closed_shapes
        self.input['Shape'] = Variable('Shape', "list", 0, self.shapes)

        self.input['Rise'] = Variable(
            'Rise', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.input['Span'] = Variable(
            'Span', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.input['Full flow area'] = Variable(
            'Full flow area', 'float', 0.0, [], precision=2, unit_type=['area'], native_unit='ft^2',
            us_units=[['yd^2', 'ft^2', 'in^2']], si_units=[['m^2', 'mm^2']])
        self.input['Slope'] = Variable(
            'Slope', 'float', 0.0, [], limits=[-10.0, 10.0], precision=6, unit_type=['slope'], native_unit='ft/ft',
            us_units=[['ft/ft']], si_units=[['m/m']])

        self.input['Culvert shape'] = Variable('Shape', "list", 0, [])
        self.input['Culvert material'] = Variable('Material', "list", 0, [])
        # self.input['Culvert type'] = Variable('Culvert type', "list", 0, [])
        self.input['Inlet configuration'] = Variable('Inlet configuration', "list", 0, [])

        self.input['Specify dimensions'] = Variable(
            'Specify dimensions (do not select geometry from a list)', "bool", False, [], complexity=1,
            note='Specified geometry will use interpolation coefficients instead of more accurate polynomial '
            'coefficients')

        self.input['Compute geometry'] = Variable(
            'Compute geometry (do not use geometry from the shapes database)', "bool", False, [], complexity=2,
            note='Computed geometry may be more or less accurate than geometry in shapes database. Shapes database '
            'uses a maximum of 19 points (x, y_top, y_bottom). Computed geometry max points is specified in settings.')

        # self.ih_hw_list = []
        # self.flow_list = []
        self.flow_unk_sorted = None
        self.flow_interp = None

        self.flow_at_half_rise = 0.0
        self.flow_at_triple_rise = 0.0

        self.mannings_n_calc = None

        self.shape_db = ShapeDatabase(app_data)

        max_value = sys.float_info.max

        # dimensions
        self.input['Geometry'] = Variable(
            'Geometry', 'class',
            ChannelGeometryData(stand_alone_calc=False, hide_depth=True, allowed_shapes='closed', app_data=app_data,
                                model_name=model_name, project_uuid=project_uuid))
        self.input['Composite n'] = Variable(
            "Manning's n", 'class',
            CompositeManningNData(shape='circle', open_shape=False, embedment_present=False, gradation_present=False,
                                  total_embedment=0.0, stations=[], elevations=[], app_data=app_data,
                                  model_name=model_name, project_uuid=project_uuid))
        self.input['Composite n'].value.input['Channel n entry'].set_val('specify material')
        self.input['Composite n'].value.input['n entry'].set_val('specify material')
        self.input['Composite n'].value.no_station_overlap = False
        self.input['Composite n'].value.in_culvert = True
        self.input['Composite n'].value.open_shape = False
        self.input['Composite n'].value.input['Material category'].set_val('storm drain conduits')

        # Polynomial
        self.input['A'] = Variable('A', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['BS'] = Variable('BS', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['C'] = Variable('C', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['DIP'] = Variable('DIP', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['EE'] = Variable('EE', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['F'] = Variable('F', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        # Both
        self.input['KE'] = Variable('KE', 'float', 0.0, limits=(-max_value, max_value))
        self.input['SR'] = Variable('SR', 'float', 0.0, limits=(-max_value, max_value))

        self.theme = self.get_theme()

        self.geometry = None

        self.dim_list = []
        self.show_select = False

        self.use_cir_ell_interp = False
        self.polynomial_available = False
        self.use_qad_poly = False

        self.x = []
        self.y_top = []
        self.y_bottom = []

        self.cs_x = []
        self.cs_y = []

        self.channel_n = None

        name = 'Select culvert size'
        culv_input = {
            'Station': self.input['Geometry'].value.input['Cross-section'].value.input['Data input'].value.input[
                'Station'],
            'Elevation': self.input['Geometry'].value.input['Cross-section'].value.input['Data input'].value.input[
                'Elevation'],
        }
        cur_shape = self.input['Culvert shape'].get_val()
        cur_mat = self.input['Culvert material'].get_val()
        geometry = self.input['Geometry']

        self.input[name] = Variable(name, 'table', SelectSizeTable(
            self.theme, name=name, plot_names=['Culvert cross-section'], input=culv_input, min_items=1,
            app_data=app_data, model_name=model_name, project_uuid=project_uuid, cur_shape=cur_shape, cur_mat=cur_mat,
            geometry=geometry))

        manning_n_data = ManningNData(standalone_calc=False, app_data=app_data, model_name=model_name,
                                      project_uuid=project_uuid)
        manning_n_data.prepare_for_compute()
        self.manning_n_calc = manning_n_data.calculator
        self.manning_n_calc.input_dict, self.manning_n_calc.plot_dict = manning_n_data.prepare_input_dict()

        # Intermediate
        self.compute_prep_functions.extend([self.update_lists])
        self.compute_finalize_functions.extend([])
        self.intermediate_to_copy.extend([
            'inlet_type_list', 'inlet_configuration_list', 'open_shapes', 'closed_shapes', 'all_shapes',
            'shapes', 'flow_at_half_rise', 'flow_at_triple_rise',
            'flow_unk_sorted', 'flow_interp', 'use_cir_ell_interp', 'polynomial_available',
            'use_qad_poly', 'manning_n_calc', 'geometry', 'channel_n',
            'shape_db',
        ])

        self.warnings = []
        self.results = {}
        # self.results['result'] = Variable('result', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)

        self.hw = []

        self.update_lists()

    def update_lists(self):
        """Update lists based on other variable values."""
        if self.shape_db.shape_database is None:
            self.warnings.append('Shapes database not loaded; only manual polynomial '
                                 'calculations available')
            return

        self.input['Select culvert size'].value.set_shape_mat(self)

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}
        input_vars['Culvert shape'] = self.input['Culvert shape']
        input_vars['Culvert material'] = self.input['Culvert material']

        cur_shape = self.input['Culvert shape'].get_val()
        specify_dim = self.input['Specify dimensions'].get_val()
        for dim in self.dim_list:
            input_vars[dim] = self.dim_list[dim]
            if not specify_dim and self.show_select or cur_shape == 'User Defined':
                input_vars[dim].read_only = True
            else:
                input_vars[dim].read_only = False
        if self.show_select and not specify_dim:
            input_vars['Select culvert size'] = self.input['Select culvert size']

        if input_vars['Culvert shape'].get_val() == 'User Defined':
            input_vars['Geometry'] = self.input['Geometry'].value.input['Cross-section']

        input_vars['Embedment'] = self.input['Geometry'].value.input['Embedment']
        input_vars['Composite n'] = self.input['Composite n']

        input_vars['Inlet type'] = copy.deepcopy(self.input['Inlet type'])
        sys_complexity = self.get_setting_var('Complexity')[1].value
        if sys_complexity < 2:
            if self.input['Inlet type'].get_val() == 'manual polynomial':
                self.input['Inlet type'].set_val(0)
                input_vars['Inlet type'].set_val(0)
            input_vars['Inlet type'].value_options.remove('manual polynomial')

        if self.input['Inlet type'].get_val() == 'manual polynomial':
            input_vars['KE'] = self.input['KE']
            input_vars['SR'] = self.input['SR']
            input_vars['A'] = self.input['A']
            input_vars['BS'] = self.input['BS']
            input_vars['C'] = self.input['C']
            input_vars['DIP'] = self.input['DIP']
            input_vars['EE'] = self.input['EE']
            input_vars['F'] = self.input['F']
        else:
            input_vars['Inlet configuration'] = self.input['Inlet configuration']

        return input_vars

    def set_ke(self):
        """Sets the KE value, if not specified by the user."""
        self.prepare_for_compute()
        self.calculator.input_dict, self.calculator.plot_dict = self.prepare_input_dict()

        return self.calculator.set_ke()

    def initialize_inlet_data_for_culvert(self, shape, span, rise, fullflow_area, slope, mannings_n_data):
        """Compute and set coefficient and data for the culvert to perform inlet computations.

        Args:
            rise (float): the rise of the culvert
            critical_depth_with_flow_at_half_rise (float): critical depth for the flow at half-rise
            critical_flow_area_with_flow_at_half_rise (float): flow area for the flow at half-rise

        Returns:
            bool: True if successful

        """
        self.input['Shape'].set_val(shape)
        self.input['Rise'].set_val(rise)
        self.input['Span'].set_val(span)
        self.input['Full flow area'].set_val(fullflow_area)
        self.input['Slope'].set_val(slope)

        mannings_n_data.prepare_for_compute()
        mannings_n_calc = mannings_n_data.calculator
        mannings_n_calc.input_dict, mannings_n_calc.plot_dict = mannings_n_data.prepare_input_dict()

        self.get_can_compute()
        result = False
        if self.can_compute:
            result = self.calculator.initialize_inlet_data_for_culvert(shape, span, rise, fullflow_area)
        return result
