"""Provides a class that will perform tailwater calculations and provide tailwater options."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.manning_n.manning_n_calc import ManningNCalc
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class TailwaterCalc(ManningNCalc):
    """Provides a class that will perform tailwater calculations and provide tailwater options."""

    def __init__(self):  # , index_of_culvert=1):
        """Initializes the Tailwater Class.

        Args:
            allowed_shapes (string): sets which channel shapes the user can select: 'open', 'closed', 'both'
            hide_depth (bool): Is this class being used for tailwater calculations?
            app_data (AppData): application data (settings, etc.).
            model_name (string): name of the model
            project_uuid (string): project uuid
        """
        super().__init__()

    def _get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        if self.input_dict['calc_data']['Tailwater type'] == 'rating curve':
            rating_curve_result = True
            if len(self.input_dict['calc_data']['Rating curve']['Flow']) == 0:
                rating_curve_result = False
            if self.input_dict['calc_data']['Head'] == 'Elevation':
                if len(self.input_dict['calc_data']['Rating curve']['Elevation']) == 0:
                    rating_curve_result = False
            else:
                if len(self.input_dict['calc_data']['Rating curve']['Depth']) == 0:
                    rating_curve_result = False
            if len(self.input_dict['calc_data']['Rating curve']['Velocity']) == 0:
                rating_curve_result = False

            if rating_curve_result:
                return True
            else:
                self.warnings['Rating curve'] = 'Please enter a rating curve.'
                return False
        elif self.input_dict['calc_data']['Tailwater type'] == 'constant tailwater elevation':
            return True
        else:
            channel_calc = self.input_dict['calc_data']['Geometry']['calculator']
            channel_calc.input_dict = copy.copy(self.input_dict)
            channel_calc.input_dict['calc_data'] = copy.copy(self.input_dict['calc_data']['Geometry'])
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Shape'] = self.input_dict[
                'calc_data']['Tailwater type']
            self.input_dict['calc_data']['Calculate'] = 'Head'
            self.unknown = 'Head'
            self.input_dict['calc_data']['Head'] = 'Elevation'
            return super()._get_can_compute()

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        flows = self.input_dict['calc_data']['Flows']
        constant_tw_elev = self.input_dict['calc_data']['Constant tailwater elevation']
        channel_invert_elev = self.input_dict['calc_data']['Channel invert elevation']
        rating_flows = None
        rating_depths = None
        rating_elevations = None
        rating_velocities = None
        if self.input_dict['calc_data']['Tailwater type'] == 'rating curve':
            rating_flows = self.input_dict['calc_data']['Rating curve']['Flow']
            rating_depths = self.input_dict['calc_data']['Rating curve']['Depth']
            rating_elevations = self.input_dict['calc_data']['Rating curve']['Elevation']
            rating_velocities = self.input_dict['calc_data']['Rating curve']['Velocity']
            if self.input_dict['calc_data']['Head'] == 'Elevation':
                rating_depths = []
                for elevation in rating_elevations:
                    rating_depths.append(elevation - channel_invert_elev)
            else:
                rating_elevation = []
                for depth in rating_depths:
                    rating_elevation.append(depth + channel_invert_elev)
        self.hw = []
        elev = 0.0
        depth = 0.0
        vel = 0.0
        tailwater_elevations = []
        tailwater_depths = []
        tailwater_velocities = []
        for flow in flows:
            if self.input_dict['calc_data']['Tailwater type'] == 'rating curve':
                _, null_data = self.get_data('Null data')
                _, zero_tol = self.get_data('Zero tolerance')
                depth_interpolation = Interpolation(
                    rating_flows, rating_depths, null_data=null_data, zero_tol=zero_tol)
                velocity_interpolation = Interpolation(
                    rating_flows, rating_velocities, null_data=null_data, zero_tol=zero_tol)
                depth, _ = depth_interpolation.interpolate_y(flow, True)
                vel, _ = velocity_interpolation.interpolate_y(flow, True)
                if depth < 0.0:
                    depth = 0.0
                    vel = 0.0
                if vel < 0:
                    vel = 0.0
                elev = depth + channel_invert_elev
            elif self.input_dict['calc_data']['Tailwater type'] == 'constant tailwater elevation':
                elev = constant_tw_elev
                depth = constant_tw_elev - channel_invert_elev
                vel = 0.0
            else:
                self.input_dict['calc_data']['Flows'] = [flow]
                self.input_dict['calc_data']['Tailwater type'] = self.input_dict['calc_data']['Tailwater type']
                super().clear_results()
                super()._compute_data()
                depth = self.results['Depths'][-1]
                elev = depth + channel_invert_elev
                vel = self.results['Average velocity'][-1]

            tailwater_elevations.append(elev)
            tailwater_depths.append(depth)
            tailwater_velocities.append(vel)

        self.results['Tailwater elevations'] = tailwater_elevations
        self.results['Tailwater depths'] = tailwater_depths
        self.results['Tailwater velocities'] = tailwater_velocities
        return True
