"""SimData class."""

# 1. Standard Python modules
import os

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem as io_util

# 4. Local modules
from xms.adcirc.__version__ import version
from xms.adcirc.data.adcirc_data import fix_datetime_format

SIM_COMP_MAIN_FILE = 'sim_comp.nc'

OUTPUT_FILENAMES = {
    'NOUTE': ['fort.61'],
    'NOUTV': ['fort.62'],
    'NOUTW': ['fort.71', 'fort.72'],
    'NOUTGE': ['fort.63'],
    'NOUTGV': ['fort.64'],
    'NOUTGW': ['fort.73', 'fort.74'],
}

REG_KEY_SUPPRESS_HOTSTART = 'suppress_missing_hotstart_errors'
REG_KEY_SUPPRESS_WIND_FILE = 'suppress_missing_wind_file_errors'
REG_KEY_SUPPRESS_NONPERIODIC = 'suppress_missing_nonperiodic_file_errors'


class SimData(XarrayBase):
    """Manages data file for the hidden simulation component."""
    def __init__(self, data_file):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        self._filename = data_file
        self._info = None
        self._general = None
        self._formulation = None
        self._timing = None
        self._output = None
        self._wind = None  # Has data_vars. Need to delete existing in file to overwrite dataset.
        self._nws10_files = None
        self._nws11_files = None
        self._nws12_files = None
        self._nws15_files = None
        self._nws16_files = None
        self._nodal_atts = None
        self._harmonics = None  # Has data_vars. Need to delete existing in file to overwrite dataset.
        self._advanced = None
        self._get_default_datasets(data_file)
        super().__init__(data_file)

    def _get_default_datasets(self, data_file):
        """Create default datasets if needed.

        Args:
            data_file (:obj:`str`): Name of the data file. If it doesn't exist, it will be created.
        """
        if not os.path.exists(data_file) or not os.path.isfile(data_file):
            info = {
                'FILE_TYPE': 'ADCIRC_SIMULATION',
                'VERSION': version,
                'proj_dir': '',  # Location of the saved project, if it exists.
                # Unused. Get links from project explorer tree now.
                'domain_uuid': '',
                'wind_grid_uuid': '',
                'wind_cov_uuid': '',
                'bound_uuid': '',
                'station_uuid': '',
                'tidal_uuid': '',
            }

            general = {
                'RUNDES': '',
                'RUNID': '',
                'IHOT': 0,  # 0, 17, 67, 68, 367, 368, 567, 568
                'IHOT_file': '',
                'run_padcirc': 0,
                'num_comp_proc': 1,  # Commandline arg for PADCIRC
                'num_io_proc': 0,  # Commandline arg for PADCIRC
                'NOLIBF': 1,  # 0, 1, 2, 3, 4, 5, or 6 (3-6 are 1 with nodal attribute)
                'CF': 0.0025,  # AKA TAU for linear friction
                'HBREAK': 1.0,
                'FTHETA': 10.0,
                'FGAMMA': 0.3333,
                'NCOR': 0,  # 0 or 1
                'CORI': 0.0,  # If NCOR = 0
                'SLAM0': 0.0,
                'SFEA0': 0.0,
                'calc_center_coords': 1,  # Bool option that will calculate SLAM0 and SFEA0 on export if enabled.
                'ANGINN': 110.0,
                'NFOVER': 1,
                'NABOUT': 0,
                'NSCREEN': 1000,
            }
            formulation = {
                'IM': 0,  # 0, 1, 21, 111112, or 611112
                'IDEN': -4,  # -4, -3, -2, -1, 4, 3, 2, or 1
                'ESLM': 3.0,
                'ESLC': 0.0,  # Only included if IM=10, not currently in the interface
                'NOLIFA': 0,  # 0, 1, 2
                'H0': 0.05,
                'VELMIN': 0.05,
                'NOLICA': 0,  # 0, 1
                'NOLICAT': 0,  # 0, 1
                'TAU0': 0,  # 0, 1 , -1, -2, -3, -5
                'TAU0_specified': 0.0,  # TAU0=1
                'Tau0FullDomainMin': 0.005,  # TAU0=-5
                'Tau0FullDomainMax': 0.2,  # TAU0=-5
                'A00': 0.35,
                'B00': 0.3,
                'C00': 0.35,
                'ITITER': 1,  # -1 or 1
                'ISLDIA': 1,  # 0, 1, 2, 3, 4, 5
                'CONVCR': 0.0000000001,
                'ITMAX': 50,
            }
            timing = {
                'ref_date': '',  # Reference date used for interpolation (not usually exported)
                'DTDP': 1.0,
                'RUNDAY': 5.0,
                'NRAMP': 0,  # 0, 1, 2, 3, 4, 5, 6, 7, or 8
                'DRAMP': 0.0,
                'DRAMPExtFlux': 0.0,
                'FluxSettlingTime': 0.0,
                'DRAMPIntFlux': 0.0,
                'DRAMPElev': 0.0,
                'DRAMPTip': 0.0,
                'DRAMPMete': 0.0,
                'DRAMPWRad': 0.0,
                'DUnRampMete': 0.0,
            }
            output = {
                # Hot start output
                'NHSTAR': 0,  # 0, 1, or 3  (5 = NetCDF4 not currently available on Windows)
                'NHSINC': 1.0,
                # NetCDF options
                'NCPROJ': '',
                'NCINST': '',
                'NCSOUR': '',
                'NCHIST': '',
                'NCREF': '',
                'NCCOM': '',
                'NCHOST': '',
                'NCCONV': '',
                'NCCONT': '',
                'NCDATE': '',  # Unused, here for backwards compatibility
            }
            output_data_vars = {
                'Name':
                    (
                        'ParamName', [
                            'Default', 'Global elevation', 'Global velocity', 'Global wind', 'Station elevation',
                            'Station velocity', 'Station wind'
                        ]
                    ),
                'Override __new_line__ Default': ('ParamName', [0, 0, 0, 0, 0, 0, 0]),
                'Output': ('ParamName', [3, 3, 3, 3, 3, 3, 3]),  # 0, 1, 2, 3, -1, -2, or -3 (>0 if appending)
                'Start __new_line__ (days)': ('ParamName', [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
                'End __new_line__ (days)': ('ParamName', [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
                'Increment __new_line__ (min)': ('ParamName', [30.0, 30.0, 30.0, 30.0, 30.0, 30.0, 30.0]),
                'File': ('ParamName', [1, 1, 1, 1, 1, 1, 1]),
                # These need to be declared as DataArray objects so the strings are variable length.
                'Hot Start': ('ParamName', np.array(['', '', '', '', '', '', ''], dtype=object)),
                'Hot Start (Wind Only)': ('ParamName', np.array(['', '', '', '', '', '', ''], dtype=object))
            }
            output_coords = {'ParamName': ['Default', 'NOUTGE', 'NOUTGV', 'NOUTGW', 'NOUTE', 'NOUTV', 'NOUTW']}
            wind = {
                'NWS': 0,
                'use_existing': 0,
                'existing_file': '',
                'first_matches_hot_start': 0,
                'WTIMINC': 1.0,
                'mesh_pressure': '',
                'mesh_wind': '',
                'grid_pressure': '',
                'grid_wind': '',
                'NWBS': 0.0,
                'DWM': 0.0,
                'max_extrap': 0.0,
                'wind_pressure': 'specifiedPc',  # 'dvorak', 'knaffzehr', 'specifiedPc', or 'background'
                'storm_start': '',  # Only if using an existing storm track coverage file
                'bladj': 0.9,
                'geofactor': 1,  # 0 or 1
                'wave_radiation': 0,  # 0, 100, 300, or 400
                'swan_hot_start': -1,  # 1 or -1  # TODO: How is this different than other wind hotstart toggle?
                'fort23_uuid': '',
                'RSTIMINC': 3600.0,
                'use_ice': 0,
                'CICE_TIMINC': 3600.0,
                'NWLAT': 1,
                'NWLON': 1,
                'WLATMAX': 0.0,
                'WLONMIN': 0.0,
                'WLATINC': 0.1,
                'WLONINC': 0.1
            }
            nws10_files = {
                'AVN File': xr.DataArray(data=np.array([], dtype=object)),
            }
            nws11_files = {
                'ETA File': xr.DataArray(data=np.array([], dtype=object)),
            }
            nws12_files = {
                'Pressure File': xr.DataArray(data=np.array([], dtype=object)),
                'Wind File': xr.DataArray(data=np.array([], dtype=object)),
            }
            nws15_files = {
                'Hours': xr.DataArray(data=np.array([], dtype=np.float64)),
                'Central Pressure': xr.DataArray(data=np.array([], dtype=np.float64)),
                'Ramp Multiplier': xr.DataArray(data=np.array([], dtype=np.float64)),
                'HWND File': xr.DataArray(data=np.array([], dtype=object)),
            }
            nws16_files = {
                'Hours': xr.DataArray(data=np.array([], dtype=np.float64)),
                'Ramp Multiplier': xr.DataArray(data=np.array([], dtype=np.float64)),
                'GFDL File': xr.DataArray(data=np.array([], dtype=object)),
            }
            nodal_atts = {
                'surface_submergence_state_on': 0,
                'surface_submergence_state': '',
                'surface_directional_effective_roughness_length_on': 0,  # 12 datasets
                'z0land_000': '',
                'z0land_030': '',
                'z0land_060': '',
                'z0land_090': '',
                'z0land_120': '',
                'z0land_150': '',
                'z0land_180': '',
                'z0land_210': '',
                'z0land_240': '',
                'z0land_270': '',
                'z0land_300': '',
                'z0land_330': '',
                'surface_canopy_coefficient_on': 0,
                'surface_canopy_coefficient': '',
                'bottom_roughness_length': '',
                'sea_surface_height_above_geoid_on': 0,
                'sea_surface_height_above_geoid': 0.0,
                'wave_refraction_in_swan_on': 0,
                'wave_refraction_in_swan': '',
                'average_horizontal_eddy_viscosity_in_sea_water_wrt_depth_on': 0,
                'average_horizontal_eddy_viscosity_in_sea_water_wrt_depth': '',
                'primitive_weighting_in_continuity_equation': '',
                'quadratic_friction_coefficient_at_sea_floor': '',
                'bridge_pilings_friction_paramenters_on': 0,
                'BK': '',  # bridge_pilings_friction_paramenter
                'BAlpha': '',  # bridge_pilings_friction_paramenter
                'BDelX': '',  # bridge_pilings_friction_paramenter
                'POAN': '',  # bridge_pilings_friction_paramenter
                'mannings_n_at_sea_floor': '',
                'chezy_friction_coefficient_at_sea_floor': '',
                'elemental_slope_limiter_on': 0,
                'elemental_slope_limiter': '',
                'advection_state_on': 0,
                'advection_state': '',
                'initial_river_elevation_on': 0,
                'initial_river_elevation': '',
            }
            harmonics_data = {
                'Constituent': xr.DataArray(data=np.array([], dtype=object)),
                'Constituent Name': xr.DataArray(data=np.array([], dtype=object)),
                'Frequency (HAREQ)': xr.DataArray(data=np.array([], dtype=np.float64)),
                'Nodal Factor (HAFF)': xr.DataArray(data=np.array([], dtype=np.float64)),
                'Equilibrium (HAFACE)': xr.DataArray(data=np.array([], dtype=np.float64)),
            }
            harmonics_atts = {
                'THAS': 0.0,
                'THAF': 0.0,
                'NHAINC': 100,
                'FMV': 0.0,
                'NHASE': 0,  # 0 or 1
                'NHASV': 0,  # 0 or 1
                'NHAGE': 0,  # 0 or 1
                'NHAGV': 0,  # 0 or 1
            }
            advanced = {  # Mostly for future
                'NDDT': 0,  # 0, 1, or 2
                'NDDT_hot_start': 1,  # -1 or 1
                'BTIMINC': 3600,
                'BCHGTIMINC': 3600,
            }
            self._info = xr.Dataset(attrs=info)
            self._general = xr.Dataset(attrs=general)
            self._formulation = xr.Dataset(attrs=formulation)
            self._timing = xr.Dataset(attrs=timing)
            self._output = xr.Dataset(data_vars=output_data_vars, coords=output_coords, attrs=output)
            self._wind = xr.Dataset(attrs=wind)
            self._nws10_files = xr.Dataset(data_vars=nws10_files)
            self._nws11_files = xr.Dataset(data_vars=nws11_files)
            self._nws12_files = xr.Dataset(data_vars=nws12_files)
            self._nws15_files = xr.Dataset(data_vars=nws15_files)
            self._nws16_files = xr.Dataset(data_vars=nws16_files)
            self._nodal_atts = xr.Dataset(attrs=nodal_atts)
            self._harmonics = xr.Dataset(data_vars=harmonics_data, attrs=harmonics_atts)
            self._advanced = xr.Dataset(attrs=advanced)
            self.commit()
        else:
            self._check_for_simple_migration()

    def _check_for_simple_migration(self):
        """Check for simple migration."""
        if 'fort23_uuid' not in self.wind.attrs:
            self.wind.attrs['fort23_uuid'] = ''
        fix_datetime_format(self.timing.attrs, 'ref_date')
        fix_datetime_format(self.wind.attrs, 'storm_start')

        if 'ESLC' not in self.formulation.attrs:
            self.formulation.attrs['ELSC'] = 0.0

        if 'NWLAT' not in self.wind.attrs:
            self.wind.attrs['NWLAT'] = 0
            self.wind.attrs['NWLON'] = 0
            self.wind.attrs['WLATMAX'] = 0.0
            self.wind.attrs['WLONMIN'] = 0.0
            self.wind.attrs['WLATINC'] = 0.1
            self.wind.attrs['WLONINC'] = 0.1

    @property
    def general(self):
        """Load the general parameters dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the general parameters in the main file
        """
        if self._general is None:
            self._general = self.get_dataset('general', False)
        return self._general

    @property
    def formulation(self):
        """Load the formulation parameters dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the formulation parameters in the main file
        """
        if self._formulation is None:
            self._formulation = self.get_dataset('formulation', False)
        return self._formulation

    @property
    def timing(self):
        """Load the timing parameters dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the timing parameters in the main file
        """
        if self._timing is None:
            self._timing = self.get_dataset('timing', False)
        return self._timing

    @property
    def output(self):
        """Load the output parameters dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the output parameters in the main file
        """
        if self._output is None:
            self._output = self.get_dataset('output', False)
        return self._output

    @output.setter
    def output(self, dset):
        """Setter for the output dataset attribute."""
        if dset:
            self._output = dset

    @property
    def wind(self):
        """Load the wind parameters dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the wind parameters in the main file
        """
        if self._wind is None:
            self._wind = self.get_dataset('wind', False)
        return self._wind

    @wind.setter
    def wind(self, dset):
        """Setter for the wind dataset attribute."""
        if dset:
            self._wind = dset

    @property
    def nws10_files(self):
        """Load the NWS=10 files dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the NWS=10 dataset in the main file
        """
        if self._nws10_files is None:
            self._nws10_files = self.get_dataset('nws10_files', False)
        return self._nws10_files

    @nws10_files.setter
    def nws10_files(self, dset):
        """Setter for the NWS=10 dataset attribute."""
        if dset is not None:
            self._nws10_files = dset

    @property
    def nws11_files(self):
        """Load the NWS=11 files dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the NWS=11 dataset in the main file
        """
        if self._nws11_files is None:
            self._nws11_files = self.get_dataset('nws11_files', False)
        return self._nws11_files

    @nws11_files.setter
    def nws11_files(self, dset):
        """Setter for the NWS=11 dataset attribute."""
        if dset is not None:
            self._nws11_files = dset

    @property
    def nws12_files(self):
        """Load the NWS=12 files dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the NWS=12 dataset in the main file
        """
        if self._nws12_files is None:
            self._nws12_files = self.get_dataset('nws12_files', False)
        return self._nws12_files

    @nws12_files.setter
    def nws12_files(self, dset):
        """Setter for the NWS=12 dataset attribute."""
        if dset:
            self._nws12_files = dset

    @property
    def nws15_files(self):
        """Load the NWS=15 files dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the NWS=15 dataset in the main file
        """
        if self._nws15_files is None:
            self._nws15_files = self.get_dataset('nws15_files', False)
        return self._nws15_files

    @nws15_files.setter
    def nws15_files(self, dset):
        """Setter for the NWS=15 dataset attribute."""
        if dset is not None:
            self._nws15_files = dset

    @property
    def nws16_files(self):
        """Load the NWS=16 files dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the NWS=16 dataset in the main file
        """
        if self._nws16_files is None:
            self._nws16_files = self.get_dataset('nws16_files', False)
        return self._nws16_files

    @nws16_files.setter
    def nws16_files(self, dset):
        """Setter for the NWS=16 dataset attribute."""
        if dset is not None:
            self._nws16_files = dset

    @property
    def nodal_atts(self):
        """Load the nodal attribute parameters dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the nodal attribute parameters in the main file
        """
        if self._nodal_atts is None:
            self._nodal_atts = self.get_dataset('nodal_atts', False)
        return self._nodal_atts

    @property
    def harmonics(self):
        """Load the harmonics analysis parameters dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the harmonic analysis parameters in the main file
        """
        if self._harmonics is None:
            self._harmonics = self.get_dataset('harmonics', False)
        return self._harmonics

    @harmonics.setter
    def harmonics(self, dset):
        """Setter for the harmonic analysis dataset attribute."""
        if dset:
            self._harmonics = dset

    @property
    def advanced(self):
        """Load the advanced parameters dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the advanced parameters in the main file
        """
        if self._advanced is None:
            self._advanced = self.get_dataset('advanced', False)
        return self._advanced

    def get_enabled_nodal_atts(self):
        """Get the ADCIRC card values for all enabled nodal attributes."""
        nodal_atts = []
        # Attributes enabled explicitly with toggles.
        have_wind = self.wind.attrs['NWS'] != 0
        if self.nodal_atts.attrs['surface_submergence_state_on']:
            nodal_atts.append('surface_submergence_state')
        # Next two are only applicable if wind is on (NWS != 0)
        if have_wind and self.nodal_atts.attrs['surface_directional_effective_roughness_length_on']:
            nodal_atts.append('surface_directional_effective_roughness_length')
        if have_wind and self.nodal_atts.attrs['surface_canopy_coefficient_on']:
            nodal_atts.append('surface_canopy_coefficient')
        if self.nodal_atts.attrs['sea_surface_height_above_geoid_on']:
            nodal_atts.append('sea_surface_height_above_geoid')
        if self.nodal_atts.attrs['wave_refraction_in_swan_on']:
            nodal_atts.append('wave_refraction_in_swan')
        if self.nodal_atts.attrs['average_horizontal_eddy_viscosity_in_sea_water_wrt_depth_on']:
            nodal_atts.append('average_horizontal_eddy_viscosity_in_sea_water_wrt_depth')
        if self.nodal_atts.attrs['bridge_pilings_friction_paramenters_on']:
            nodal_atts.append('bridge_pilings_friction_paramenters')
        if self.nodal_atts.attrs['elemental_slope_limiter_on']:
            nodal_atts.append('elemental_slope_limiter')
        if self.nodal_atts.attrs['advection_state_on']:
            nodal_atts.append('advection_state')
        if self.nodal_atts.attrs['initial_river_elevation_on']:
            nodal_atts.append('initial_river_elevation')
        # Check for nodal TAU0 option
        if self.formulation.attrs['TAU0'] == -3:
            nodal_atts.append('primitive_weighting_in_continuity_equation')
        # Check for nodal friction options
        nolibf = self.general.attrs['NOLIBF']
        if nolibf > 2:  # Friction from nodal attribute
            if nolibf == 3:
                nodal_atts.append('quadratic_friction_coefficient_at_sea_floor')
            elif nolibf == 4:
                nodal_atts.append('mannings_n_at_sea_floor')
            elif nolibf == 5:
                nodal_atts.append('chezy_friction_coefficient_at_sea_floor')
            elif nolibf == 6:
                nodal_atts.append('bottom_roughness_length')
        return nodal_atts

    def get_nws_type(self):
        """Get the non-simplified NWS value to export to the fort.15."""
        nws_type = self.wind.attrs['NWS']
        if nws_type == 0:  # Do not adjust NWS based on options that may be set if wind is disabled.
            return nws_type
        nws_type += self.wind.attrs['wave_radiation']

        if nws_type == 312:  # Check for SWAN hot start option if SWAN coupling enabled
            nws_type *= self.wind.attrs['swan_hot_start']
        if self.wind.attrs['use_ice']:
            nws_type += 12000
        if self.wind.attrs['first_matches_hot_start'] and nws_type in [2, 4, 5, 12, 15, 16]:
            nws_type *= -1
        return nws_type

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming
        if self._general is not None:
            self._general.close()
            self._general.to_netcdf(self._filename, group='general', mode='a')
        if self._formulation is not None:
            self._formulation.close()
            self._formulation.to_netcdf(self._filename, group='formulation', mode='a')
        if self._timing is not None:
            self._timing.close()
            self._timing.to_netcdf(self._filename, group='timing', mode='a')
        if self._output is not None:
            self._output.close()
            self._drop_h5_groups(['output'])
            self._output.to_netcdf(self._filename, group='output', mode='a')
        if self._wind is not None:
            self._wind.close()
            self._wind.to_netcdf(self._filename, group='wind', mode='a')
        if self._nws10_files is not None:  # Has data_vars. Need to delete existing in file to overwrite dataset.
            self._nws10_files.close()
            self._drop_h5_groups(['nws10_files'])
            self._nws10_files.to_netcdf(self._filename, group='nws10_files', mode='a')
        if self._nws11_files is not None:  # Has data_vars. Need to delete existing in file to overwrite dataset.
            self._nws11_files.close()
            self._drop_h5_groups(['nws11_files'])
            self._nws11_files.to_netcdf(self._filename, group='nws11_files', mode='a')
        if self._nws12_files is not None:  # Has data_vars. Need to delete existing in file to overwrite dataset.
            self._nws12_files.close()
            self._drop_h5_groups(['nws12_files'])
            self._nws12_files.to_netcdf(self._filename, group='nws12_files', mode='a')
        if self._nws15_files is not None:  # Has data_vars. Need to delete existing in file to overwrite dataset.
            self._nws15_files.close()
            self._drop_h5_groups(['nws15_files'])
            self._nws15_files.to_netcdf(self._filename, group='nws15_files', mode='a')
        if self._nws16_files is not None:  # Has data_vars. Need to delete existing in file to overwrite dataset.
            self._nws16_files.close()
            self._drop_h5_groups(['nws16_files'])
            self._nws16_files.to_netcdf(self._filename, group='nws16_files', mode='a')
        if self._nodal_atts is not None:
            self._nodal_atts.close()
            self._nodal_atts.to_netcdf(self._filename, group='nodal_atts', mode='a')
        if self._harmonics is not None:  # Has data_vars. Need to delete existing in file to overwrite dataset.
            self._harmonics.close()
            self._drop_h5_groups(['harmonics'])
            self._harmonics.to_netcdf(self._filename, group='harmonics', mode='a')
        if self._advanced is not None:
            self._advanced.close()
            self._advanced.to_netcdf(self._filename, group='advanced', mode='a')

    def vacuum(self):
        """Rewrite all SimData to a new/wiped file to reclaim disk space."""
        # Ensure all datasets are loaded into memory.
        if self._info is None:
            self._info = self.get_dataset('info', False)
        if self._general is None:
            self._general = self.get_dataset('general', False)
        if self._formulation is None:
            self._formulation = self.get_dataset('formulation', False)
        if self._timing is None:
            self._timing = self.get_dataset('timing', False)
        if self._output is None:
            self._output = self.get_dataset('output', False)
        if self._wind is None:
            self._wind = self.get_dataset('wind', False)
        if self._nws10_files is None:
            self._nws10_files = self.get_dataset('nws10_files', False)
        if self._nws11_files is None:
            self._nws11_files = self.get_dataset('nws11_files', False)
        if self._nws12_files is None:
            self._nws12_files = self.get_dataset('nws12_files', False)
        if self._nws15_files is None:
            self._nws15_files = self.get_dataset('nws15_files', False)
        if self._nws16_files is None:
            self._nws16_files = self.get_dataset('nws16_files', False)
        if self._nodal_atts is None:
            self._nodal_atts = self.get_dataset('nodal_atts', False)
        if self._harmonics is None:
            self._harmonics = self.get_dataset('harmonics', False)
        if self._advanced is None:
            self._advanced = self.get_dataset('advanced', False)
        io_util.removefile(self._filename)  # Delete the existing NetCDF file
        self.commit()  # Rewrite all datasets

    def migrate_relative_paths(self):
        """
        Migrate data that uses relative paths.

        We used to store relative paths, but they were a huge headache for no clear gain, so now we just make everything
        absolute to keep it simple.
        """
        proj_dir: str | None = self.info.attrs.pop('proj_dir', None)
        if not proj_dir:
            return

        _make_abs(proj_dir, self.general.attrs, 'IHOT_file')
        _make_abs(proj_dir, self.wind.attrs, 'existing_file')
        _make_array_abs(proj_dir, self.output.variables['Hot Start'])
        _make_array_abs(proj_dir, self.output.variables['Hot Start (Wind Only)'])
        _make_array_abs(proj_dir, self.nws10_files['AVN File'])
        _make_array_abs(proj_dir, self.nws11_files['ETA File'])
        _make_array_abs(proj_dir, self.nws12_files.variables['Pressure File'])
        _make_array_abs(proj_dir, self.nws12_files.variables['Wind File'])
        _make_array_abs(proj_dir, self.nws15_files['HWND File'])
        _make_array_abs(proj_dir, self.nws16_files['GFDL File'])
        self.commit()

    def copy_external_files(self):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Returns:
            (:obj:`str`): Message on failure, empty string on success
        """
        old_proj_dir = self.info.attrs['proj_dir']
        # Target folder is the <project name> folder next to the SMS project file.
        target_dir = os.path.normpath(os.path.join(os.path.dirname(self._filename), '../../..'))

        # Copy the single file references
        self._copy_attr_file(self.general.attrs, 'IHOT_file', old_proj_dir, target_dir)
        self._copy_attr_file(self.wind.attrs, 'existing_file', old_proj_dir, target_dir)

        # Copy the files referenced in tables
        self._copy_table_files(self.output.variables['Hot Start'], old_proj_dir, target_dir)
        self._copy_table_files(self.output.variables['Hot Start (Wind Only)'], old_proj_dir, target_dir)
        self._copy_table_files(self.nws10_files.variables['AVN File'], old_proj_dir, target_dir)
        self._copy_table_files(self.nws11_files.variables['ETA File'], old_proj_dir, target_dir)
        self._copy_table_files(self.nws12_files.variables['Pressure File'], old_proj_dir, target_dir)
        self._copy_table_files(self.nws12_files.variables['Wind File'], old_proj_dir, target_dir)
        self._copy_table_files(self.nws15_files.variables['HWND File'], old_proj_dir, target_dir)
        self._copy_table_files(self.nws16_files.variables['GFDL File'], old_proj_dir, target_dir)
        # Wipe the stored project directory. Paths will be absolute in GUI until resave.
        self.info.attrs['proj_dir'] = ''

        return ''


def _make_abs(base, attrs, key):
    if not attrs.get(key, None):
        return

    attrs[key] = os.path.normpath(os.path.join(base, attrs[key]))


def _make_array_abs(base, array):
    for i in range(len(array)):
        if not array[i]:
            continue
        old_value = array[i]
        if not isinstance(old_value, str):
            old_value = old_value.item()
        array[i] = os.path.normpath(os.path.join(base, old_value))
