"""Module to perform xms.api Query operations for the ADCIRC Boundary Conditions coverage component."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file
from xms.interp.interpolate.interp_linear import InterpLinear

# 4. Local modules
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.mapping.mapping_util import grid_projection_mismatches_display


def query_for_constituent_datasets(query, amp_dset, phase_dset):
    """Get the dataset and geometry dumps of the selected user constituent amplitudes and phases.

    Args:
        query (:obj:`Query`): The Query for communicating with XMS
        amp_dset (:obj:`xarray.DataArray`): Dataset containing the amplitude dataset UUIDS
        phase_dset (:obj:`xarray.DataArray`): Dataset containing the phase dataset UUIDS

    Returns:
        (:obj:`list, list, dict`): The amplitude datasets, the phase datasets, and dict of geometries
        keyed by geometry UUID
    """
    amp_dsets = []
    phase_dsets = []
    geoms = {}
    for i in range(amp_dset.size):
        # Get the dataset dumps from XMS.
        # The label index of the source tidal Dataset is one-based (comes from the GUI table).
        amp_uuid = amp_dset.loc[i + 1].item()
        phase_uuid = phase_dset.loc[i + 1].item()
        do_amp_dset = query.item_with_uuid(amp_uuid)
        if do_amp_dset is None:
            XmLog().instance.error('Error getting flow amplitude dataset.')
            raise RuntimeError()
        do_phase_dset = query.item_with_uuid(phase_uuid)
        if do_phase_dset is None:
            XmLog().instance.error('ERROR', 'Error getting flow phase dataset.')
            raise RuntimeError()
        amp_dsets.append(do_amp_dset)
        phase_dsets.append(do_phase_dset)

        # Get the dataset geometry dumps from XMS if we haven't already.
        amp_geom_uuid = amp_dsets[-1].geom_uuid
        if amp_geom_uuid not in geoms:
            amp_geom = query.item_with_uuid(amp_geom_uuid)
            if not amp_geom:
                XmLog().instance.error('Error getting flow amplitude dataset geometry.')
                raise RuntimeError()
            co_grid = read_grid_from_file(amp_geom.cogrid_file)
            ugrid = co_grid.ugrid

            if amp_dsets[-1].location == 'cells':
                pts = [ugrid.get_cell_centroid(cell)[1] for cell in range(ugrid.cell_count)]
            else:
                pts = ugrid.locations
            if len(pts) < 3:
                XmLog().instance.error('Flow amplitude dataset must contain at least three points.')
                raise RuntimeError()
            geoms[amp_geom_uuid] = InterpLinear(points=pts)

        phase_geom_uuid = phase_dsets[-1].geom_uuid
        if phase_geom_uuid not in geoms:
            phase_geom = query.item_with_uuid(phase_geom_uuid)
            if not phase_geom:
                XmLog().instance.error('Error getting flow phase dataset geometry.')
                raise RuntimeError()
            co_grid = read_grid_from_file(phase_geom.cogrid_file)
            ugrid = co_grid.ugrid

            if phase_dsets[-1].location == 'cells':
                pts = [ugrid.get_cell_centroid(cell)[1] for cell in range(ugrid.cell_count)]
            else:
                pts = ugrid.locations
            if len(pts) < 3:
                XmLog().instance.error('Flow amplitude dataset must contain at least three points.')
                raise RuntimeError()
            geoms[phase_geom_uuid] = InterpLinear(points=pts)
    return amp_dsets, phase_dsets, geoms


class BcComponentQueries:
    """Performs xms.api calls for the BC coverage component."""
    def __init__(self, bc_comp, query):
        """Initialize the API helper.

        Args:
            bc_comp (:obj:`BcComponent`): The component instance to perform API call for
            query (:obj:`xms.api.dmi.ActionRequest`): Object for communicating with XMS
        """
        self.bc_comp = bc_comp
        self.query = query

    def get_xms_data_for_mapping(self):
        """Query XMS for the data needed to apply a BC coverage.

        Data dict contains:

            cov = :obj:`xms.data_objects.parameters.Coverage`

            mesh = :obj:`xms.data_objects.parameters.UGrid`

            sim_uuid = :obj:`str`

            flow_geoms = :obj:`{str: InterpLinear}`  # Key is Periodic flow forcing geometry UUID, value is
                interpolator for geom

            flow_amps = :obj:`[xms.data_objects.parameters.Dataset]`  # Parallel with flow_phases. Periodic flow
                forcing datasets

            flow_phases = :obj:`[xms.data_objects.parameters.Dataset]`  # Parallel with flow_amps. Periodic flow
                forcing datasets

            old_mapped_flow_uuid = :obj:`str`

        Returns:
            (:obj:`dict`): The XMS data dictionary.
        """
        mapping_data = {
            'cov': None,
            'mesh': None,
            'sim_uuid': None,
            'flow_geoms': {},
            'flow_amps': [],
            'flow_phases': [],
            'old_mapped_flow_uuid': '',  # UUID of previously mapped flow component. Will be deleted if it exists.
        }
        # Get the coverage geometry
        XmLog().instance.info('Retrieving source Boundary Condition coverage geometry from SMS.')
        mapping_data['cov'] = self.query.item_with_uuid(self.bc_comp.cov_uuid)
        if not mapping_data['cov']:
            XmLog().instance.error('Error getting coverage geometry when applying boundary conditions.')
            self.unlink_bc_cov_on_error()
            return mapping_data

        # Find the simulation we are being linked to.
        sim_item = None
        ptr_items = []
        tree_util.find_linked_pointers_by_uuid(self.query.project_tree, self.bc_comp.cov_uuid, ptr_items)
        for ptr_item in ptr_items:  # Should only ever be one in the project explorer linked to an ADCIRC simulation.
            ptr_item_parent = ptr_item.parent
            if ptr_item_parent and ptr_item_parent.model_name == 'ADCIRC':
                sim_item = ptr_item_parent  # Store off the simulation UUID
                mapping_data['sim_uuid'] = sim_item.uuid
                break

        if not sim_item:
            XmLog().instance.error('Error finding ADCIRC simulation to apply boundary conditions.')
            self.unlink_bc_cov_on_error()
            return mapping_data

        # Find the linked mesh
        XmLog().instance.info("Retrieving simulation's ADCIRC domain mesh...")
        mesh_items = tree_util.descendants_of_type(
            sim_item, xms_types=['TI_MESH2D_PTR'], allow_pointers=True, recurse=False
        )
        if not mesh_items:  # No linked mesh
            XmLog().instance.error('Error getting mesh geometry when applying boundary conditions.')
            self.unlink_bc_cov_on_error()
            return mapping_data

        # Get the geometry dump
        mapping_data['mesh'] = self.query.item_with_uuid(mesh_items[0].uuid)
        if not mapping_data['mesh']:
            XmLog().instance.error('Error getting mesh geometry when applying boundary conditions.')
            self.unlink_bc_cov_on_error()
            return mapping_data

        # Ensure that the current display projection matches the ADCIRC mesh.
        native_proj = mapping_data['mesh'].native_projection
        display_proj = mapping_data['mesh'].projection
        if grid_projection_mismatches_display(native_proj, display_proj):
            XmLog().instance.error(
                'Display projection does not match the ADCIRC mesh. \n'
                'Cannot apply Boundary Condition data unless the projections match. \n'
                'Ensure mesh projection is the desired projection for ADCIRC and \n'
                'set display projection to match to allow application of boundary conditions.'
            )
            self.unlink_bc_cov_on_error()
            return mapping_data

        # Unlink existing mapped flow component, if one exists.
        linked_items = tree_util.descendants_of_type(sim_item, unique_name='Mapped_Flow_Component', recurse=False)
        if linked_items:
            mapping_data['old_mapped_flow_uuid'] = linked_items[0].uuid

        # Get river flow constituent amplitude and phase datasets
        amps, phases, geoms = self._query_for_flow_constituent_datasets(mapping_data)
        mapping_data['flow_geoms'] = geoms
        mapping_data['flow_amps'] = amps
        mapping_data['flow_phases'] = phases

        return mapping_data

    def _query_for_flow_constituent_datasets(self, xms_data):
        """Get the dataset and geometry dumps of the selected user constituent amplitudes and phases.

        Args:
            xms_data (:obj:`dict`): Dictionary containing the data retrieved from XMS prior to mapping

        Returns:
            (:obj:`list, list, dict`): The amplitude datasets, the phase datasets, dict of geometries
            keyed by geometry UUID.
        """
        amp_dsets = []
        phase_dsets = []
        geoms = {}
        try:
            flow_data = self.bc_comp.data.flow_cons
            if self.bc_comp.data.info.attrs['periodic_flow'].item() == 1 and flow_data.Amplitude.size > 0:
                # Grab datasets and geometries if periodic
                if flow_data.Amplitude.data.size != flow_data.Phase.data.size:
                    XmLog().instance.error('Number of amplitude datasets does not match the number of phase datasets.')
                    raise RuntimeError()

                amp_dsets, phase_dsets, geoms = query_for_constituent_datasets(
                    self.query, flow_data.Amplitude, flow_data.Phase
                )
        except Exception:
            XmLog().instance.error(
                'Could not retrieve user defined constituent amplitude and phase datasets for mapping flow '
                'constituents.'
            )
        return amp_dsets, phase_dsets, geoms

    def add_xms_data_for_mapping(self, xms_data, mapped_bc_comp, flow_comp, display_options, delete_uuids):
        """Send the new mapped BC component to XMS, unlink the source coverage, and ActionRequests.

        Args:
            xms_data (:obj:`dict`): Dictionary containing the data retrieved from XMS prior to mapping
            mapped_bc_comp (:obj:`xms.data_objects.parameters.Component`): The new mapped BC component
            flow_comp (:obj:`xms.data_objects.parameters.Component`): The new mapped flow component, if there were any
                river boundaries in the source BC coverage. Otherwise, should be None.
            display_options (:obj:`list[XmsDisplayMessage]`): The display options of the mapped BC component
            delete_uuids (list[str]): The UUIDs of MappedBcComponents to delete.
        """
        # Add the mapped component to the Context and send back to XMS.
        if mapped_bc_comp:  # Will be None if we logged an error during the mapping operation.
            # self.query.add_component(mapped_bc_comp, actions=action_requests)
            self.query.add_component(mapped_bc_comp, display_options=display_options)
            self.query.link_item(xms_data['sim_uuid'], mapped_bc_comp.uuid)
            if flow_comp:  # Will be None if no flow boundaries
                self.query.add_component(flow_comp)
                self.query.link_item(xms_data['sim_uuid'], flow_comp.uuid)

        # Unlink the unmapped BC coverage take
        self.query.unlink_item(xms_data['sim_uuid'], self.bc_comp.cov_uuid)
        # Delete previously mapped BC component, if it exists
        for delete_uuid in delete_uuids:
            self.query.delete_item(delete_uuid)
        if xms_data['old_mapped_flow_uuid']:
            self.query.delete_item(xms_data['old_mapped_flow_uuid'])

    def unlink_bc_cov_on_error(self):
        """Unlink the BC coverage from the sim when mapping fails.

        Query context should be at the BC coverage level if member is not initialized.
        """
        bc_item_ptrs = []
        tree_util.find_linked_pointers_by_uuid(self.query.project_tree, self.bc_comp.cov_uuid, bc_item_ptrs)
        # Should only ever be one source BC coverage linked to an ADCIRC simulation in the project tree at any given
        # time.
        for bc_item_ptr in bc_item_ptrs:
            bc_item_ptr_parent = bc_item_ptr.parent
            if bc_item_ptr_parent and bc_item_ptr_parent.model_name == 'ADCIRC':
                sim_uuid = bc_item_ptr_parent.uuid
                self.query.unlink_item(sim_uuid, self.bc_comp.cov_uuid)
                return
