"""Worker thread for the solution importer."""

# 1. Standard Python modules
import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackThread
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules
from xms.adcirc.components.sim_component import SimComponent
from xms.adcirc.data import sim_data as sid
from xms.adcirc.dmi.sim_component_queries import SimComponentQueries
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.file_io.solution_importer import ADCIRCSolutionReader


class SolutionLoadWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing an ADCIRC solution after a model run in SMS."""
    def __init__(self, filename, query, parent):
        """Construct the worker.

        Args:
            filename (:obj:`str`): Path to the simulation hidden component main file
            query (:obj:`Query`): Object for communicating with XMS
            parent (:obj:`QWidget`): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.filename = filename
        self.query = query

    def _do_work(self):
        """Thread runner that exports the fort.14."""
        # Look for a mapped BC coverage first.
        XmLog().instance.info('Loading ADCIRC simulation run solution files...')
        try:
            py_comp = SimComponent(self.filename)
            helper = SimComponentQueries(py_comp, self.query)
            data = py_comp.data
            # Don't try to read wind solutions if no wind. Option might be enabled by default in data file because the
            # GUI option is disabled but the format parameter defaults to NetCDF (3)
            has_wind = data.wind.attrs['NWS'] != 0
            filenames = []
            XmLog().instance.info('Gathering list of solution files to import...')
            # Read recording station output, if we have them
            if helper.find_sim_link('TI_COVER_PTR', coverage_type='Recording Stations'):
                for output_type in ['NOUTE', 'NOUTV', 'NOUTW']:
                    output_format = data.output.sel(ParamName=output_type).Output.item()
                    nc_ext = '.nc' if output_format in [3, 5] else ''  # Append .nc extension for NetCDF file formats
                    if output_format != 0:  # Read if enabled
                        if output_type != 'NOUTW':  # But not if it is a wind solution and we have no wind
                            filenames.append(f'{sid.OUTPUT_FILENAMES[output_type][0]}{nc_ext}')
                        elif has_wind:  # Two files for meteorological solutions
                            filenames.append(f'{sid.OUTPUT_FILENAMES[output_type][0]}{nc_ext}')
                            filenames.append(f'{sid.OUTPUT_FILENAMES[output_type][1]}{nc_ext}')
            for output_type in ['NOUTGE', 'NOUTGV', 'NOUTGW']:  # Read global output files
                output_format = data.output.sel(ParamName=output_type).Output.item()
                nc_ext = '.nc' if output_format in [3, 5] else ''  # Append .nc extension for NetCDF file formats
                if output_format != 0:  # Read if enabled
                    if output_type != 'NOUTGW':  # But not if it is a wind solution and we have no wind
                        filenames.append(f'{sid.OUTPUT_FILENAMES[output_type][0]}{nc_ext}')
                        if output_type == 'NOUTGE':
                            filenames.append(f'maxele.63{nc_ext}')
                        else:
                            filenames.append(f'maxvel.63{nc_ext}')
                    elif has_wind:  # Two files for meteorological solutions
                        filenames.append(f'{sid.OUTPUT_FILENAMES[output_type][0]}{nc_ext}')
                        filenames.append(f'{sid.OUTPUT_FILENAMES[output_type][1]}{nc_ext}')
                        filenames.append(f'maxwvel.63{nc_ext}')
                        filenames.append(f'minpr.63{nc_ext}')
                        if data.wind.attrs['wave_radiation'] > 0:
                            filenames.append(f'maxrs.63{nc_ext}')  # Only written if radiation stress enabled.

            # Look for harmonic analysis solutions.
            if data.harmonics.attrs['NHAGE']:
                filenames.append('fort.53')  # Global elevation harmonic analysis solution
            if data.harmonics.attrs['NHAGV']:
                filenames.append('fort.54')  # Global velocity harmonic analysis solution

            # Use the ADCIRC simulation reference time as reference time for transient solutions
            reftime = datetime.datetime.strptime(data.timing.attrs['ref_date'], ISO_DATETIME_FORMAT)

            # Read the solution file and send to XMS. Will be loaded on the mesh linked to the run simulation.
            domain_uuid = helper.find_sim_link('TI_MESH2D_PTR').uuid
            reader = ADCIRCSolutionReader(filenames, reftime, self.query.xms_temp_directory, domain_uuid)
            reader.read()
            reader.add_datasets(self.query)
        except Exception:
            XmLog().instance.exception('Error(s) encountered while loading ADCIRC solution files.')
