"""Table of file selectors for NWS file variants."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, QSortFilterProxyModel, Qt
from PySide2.QtWidgets import QAbstractItemView

# 3. Aquaveo modules
from xms.guipy import settings
from xms.guipy.delegates.file_selector_delegate import FileSelectorButtonDelegate
from xms.guipy.dialogs.file_selector_dialogs import get_open_filenames

# 4. Local modules
from xms.adcirc.gui.adcirc_table import AdcircTableWidget

NUM_NWS10_NWS11_COLS = 1
NUM_NWS15_COLS = 4
NUM_NWS16_COLS = 3
TBL_COL_HOURS = 0  # The first column in the table


class WindFileDisableEditModel(QSortFilterProxyModel):
    """A model to set disable editing of the file selector columns."""
    def __init__(self, disable_edit_col, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            disable_edit_col (:obj:`int`): This column and all columns after it will be disabled for editing. Assuming
                the last column is the file selector
        """
        super().__init__(parent)
        self._disable_edit_col = disable_edit_col

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (:obj:`QModelIndex`): The item's model index

        Returns:
            (:obj:`Qt.ItemFlags`): Updated flags for the item
        """
        ret_flags = super().flags(index)
        col = index.column()
        if col >= self._disable_edit_col:  # This is a file selector column
            ret_flags = ret_flags & (~Qt.ItemIsEditable)
        return ret_flags


class WindFileTableWidget(AdcircTableWidget):
    """The NWS file selector table widget (for NWS=15,16)."""
    def __init__(self, data_frame, data, default_values, parent=None):
        """Construct the widget.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The model data.
            data (:obj:`SimData`): SimData object. Needed for resolving relative paths
            default_values (:obj:`dict`): Default DataFrame values for added rows
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        if len(data_frame.index) > 0 and data_frame.index[0] == 0:
            data_frame.index += 1  # Make vertical header 1-based
        super().__init__(parent, data_frame, TBL_COL_HOURS, default_values)
        file_col = self.model.columnCount() - 1
        self._file_col_name = self.model.data_frame.columns[file_col]
        self._file_selector_delegate = FileSelectorButtonDelegate(
            data.info.attrs['proj_dir'], 'Select a wind file', None, self.parent()
        )
        self._filter_model = WindFileDisableEditModel(file_col, self)
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        # Assuming the last column is file selector. Change if need to support multiple columns of file selectors in
        # a single table.
        file_col = self.model.columnCount() - 1
        super()._setup_ui({file_col: self._file_selector_delegate}, True, False, self._filter_model)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)

    def on_btn_add_row(self):
        """Overload to display a multi-file open dialog."""
        start_dir = settings.get_file_browser_directory()
        filenames = get_open_filenames(self, 'Select wind files', start_dir=start_dir)
        if filenames:
            row_idx = self.model.rowCount()
            self.model.insertRows(row_idx, len(filenames))
            df_idx = row_idx if not row_idx else row_idx + 1
            self.model.data_frame.loc[df_idx:, self._file_col_name] = filenames
            new_index = self.model.index(row_idx, self.select_col)
            self.table_view.selectionModel().setCurrentIndex(
                new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
            )
