"""A class for lazily loading time series data."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from typing import Any, Optional

# 2. Third party modules
from adhparam.time_series import TimeSeries

# 3. Aquaveo modules

# 4. Local modules
from xms.adh.data.param_h5_io import read_params_recursive


class LazyTimeSeries:
    """A class for lazily loading time series data.

    This class defers the loading of time series data until it is accessed.
    It supports deep copying while preserving the lazy-loading behavior.

    Attributes:
        filename (str): The name of the file containing the time series data.
        int_id (int): The unique identifier for the time series within the file.
        _data (Optional[TimeSeries]): The actual time series data. It remains
            None until accessed.
    """

    def __init__(self, filename: str, int_id: int) -> None:
        """Initializes a LazyTimeSeries instance.

        Args:
            filename (str): The name of the file containing the time series data.
            int_id (int): The unique identifier for the time series within the file.
        """
        self.filename: str = filename
        self.int_id: int = int_id
        self._data: Optional[TimeSeries] = None  # Data will be loaded only when accessed

    def _load_data(self) -> None:
        """Loads the actual time series data.

        This method checks if the `_data` attribute is None. If so, it lazily
        initializes `_data` by reading the time series from the file.
        """
        if self._data is None:
            self._data = TimeSeries()
            read_params_recursive(self.filename, group_name=f"/time_series/{self.int_id}/", param_class=self._data)

    def __getattr__(self, name: str) -> Any:
        """Intercepts attribute access and triggers lazy loading.

        If the accessed attribute is not found, this method ensures that
        `_data` is loaded, and then attempts to retrieve the requested
        attribute from `_data`.

        Args:
            name (str): The name of the attribute being accessed.

        Returns:
            Any: The value of the requested attribute from `_data`.

        Raises:
            AttributeError: If the attribute does not exist in `_data`.
        """
        self._load_data()
        return getattr(self._data, name)

    def __deepcopy__(self, memo: dict) -> "LazyTimeSeries":
        """Creates a deep copy of the LazyTimeSeries instance.

        This method ensures that the deep copy preserves the state of lazy loading.
        If `_data` has not been loaded, it remains None in the copy.
        If `_data` has been loaded, it is deep-copied.

        Args:
            memo (dict): A dictionary used by `copy.deepcopy` to store
                objects already copied, to handle cyclic references.

        Returns:
            LazyTimeSeries: A new LazyTimeSeries instance that is a deep copy of the original.
        """
        new_obj = type(self)(self.filename, self.int_id)
        memo[id(self)] = new_obj

        new_obj._data = copy.deepcopy(self._data, memo) if self._data is not None else None

        return new_obj
