"""Class for managing interprocess communication with XMS."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional, TYPE_CHECKING

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util, TreeNode
from xms.constraint import Grid as CoGrid
from xms.data_objects.parameters import Coverage
from xms.datasets.dataset_reader import DatasetReader

# 4. Local modules
from xms.adh.data.adh_data import AdhData
from xms.adh.data.lazy_load import LazyField

if TYPE_CHECKING:
    from xms.adh.components.bc_conceptual_component import BcConceptualComponent
    from xms.adh.components.material_conceptual_component import MaterialConceptualComponent
    from xms.adh.components.sediment_constituents_component import SedimentConstituentsComponent
    from xms.adh.components.sediment_material_conceptual_component import SedimentMaterialConceptualComponent
    from xms.adh.components.sim_component import SimComponent
    from xms.adh.components.transport_constituents_component import TransportConstituentsComponent
    from xms.adh.components.vessel_component import VesselComponent


class XmsData:
    """Class for managing interprocess communication with XMS."""
    adh_data: AdhData | None = LazyField()
    project_tree: TreeNode | None = LazyField()
    xms_temp_directory: str | None = LazyField()
    sim_item: TreeNode | None = LazyField()
    sim_name: str | None = LazyField()
    sim_uuid: str | None = LazyField()
    sim_component: Optional['SimComponent'] = LazyField()
    bc_component: Optional['BcConceptualComponent'] = LazyField()
    material_component: Optional['MaterialConceptualComponent'] = LazyField()
    transport_component: Optional['TransportConstituentsComponent'] = LazyField()
    sediment_material_component: Optional['SedimentMaterialConceptualComponent'] = LazyField()
    sediment_constituents_component: Optional['SedimentConstituentsComponent'] = LazyField()
    vessel_components: Optional[list['VesselComponent']] = LazyField()
    co_grid: CoGrid | None = LazyField()
    geom_uuid: str | None = LazyField()
    bc_coverage: Coverage | None = LazyField()
    material_coverage: Coverage | None = LazyField()
    sediment_material_coverage: Coverage | None = LazyField()
    output_coverage: Coverage | None = LazyField()
    vessel_coverages: list[Coverage] | None = LazyField()
    scalar_datasets: list[TreeNode] | None = LazyField()
    vector_datasets: list[TreeNode] | None = LazyField()
    dataset_uuid_from_path: dict[str, str] | None = LazyField()
    dataset_path_from_uuid: dict[str, str] | None = LazyField()
    horizontal_units: str | None = LazyField()

    def __init__(self):
        """Initialise XmsData."""
        self.dataset_readers: dict[str, DatasetReader] = {}  # For testing and caching. Dataset UUID to reader.

    @property
    def domain_project_tree(self) -> TreeNode | None:
        """Get the domain project tree.

        Returns:
            The domain project tree root.
        """
        return tree_util.trim_project_explorer(self.project_tree, self.adh_data.model_control.domain_uuid)

    def get_dataset_from_uuid(self, uuid: str) -> DatasetReader | None:
        """Get the dataset reader for the given UUID.

        Args:
            uuid (str): The UUID of the dataset.

        Returns:
            The dataset reader for the given UUID.
        """
        dataset = self.dataset_readers.get(uuid, None)
        return dataset

    def send(self):
        """Send data to XMS."""
        pass

    def _load_adh_data(self) -> AdhData | None:
        return None

    def _load_project_tree(self) -> TreeNode | None:
        return None

    def _load_sim_item(self) -> TreeNode | None:
        return None

    def _load_sim_name(self) -> str | None:
        return None

    def _load_sim_uuid(self) -> str | None:
        return None

    def _load_sim_component(self) -> Optional['SimComponent']:
        return None

    def _load_bc_component(self) -> Optional['BcConceptualComponent']:
        return None

    def _load_material_component(self) -> Optional['MaterialConceptualComponent']:
        return None

    def _load_transport_component(self) -> Optional['TransportConstituentsComponent']:
        return None

    def _load_sediment_material_component(self) -> Optional['SedimentMaterialConceptualComponent']:
        return None

    def _load_sediment_constituents_component(self) -> Optional['SedimentConstituentsComponent']:
        return None

    def _load_vessel_components(self) -> Optional[list['VesselComponent']]:
        return None

    def _load_material_coverage(self) -> Coverage | None:
        return None

    def _load_co_grid(self) -> CoGrid | None:
        return None

    def _load_geom_uuid(self) -> str | None:
        return None

    def _load_bc_coverage(self) -> Coverage | None:
        return None

    def _load_sediment_material_coverage(self) -> Coverage | None:
        return None

    def _load_output_coverage(self) -> Coverage | None:
        return None

    def _load_vessel_coverages(self) -> list[Coverage] | None:
        return None

    def _load_scalar_datasets(self) -> list[TreeNode] | None:
        """
        Load a list of scalar datasets in the domain.

        Returns:
            The list of scalar datasets.
        """
        datasets = tree_util.descendants_of_type(self.domain_project_tree, xms_types=['TI_SFUNC'])
        return datasets

    def _load_vector_datasets(self) -> list[TreeNode] | None:
        """
        Load a list of vector datasets in the domain.

        Returns:
            The list of vector datasets.
        """
        datasets = tree_util.descendants_of_type(self.domain_project_tree, xms_types=['TI_VFUNC'])
        return datasets

    def _load_horizontal_units(self) -> str | None:
        return None
