"""This module handles aspects of running an AdH simulation."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import multiprocessing
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand, XmsEnvironment as XmEnv
from xms.components.bases.run_base import RunBase
from xms.guipy.dialogs import message_box
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.adh.data.xms_data import XmsData
from xms.adh.data.xms_query_data import XmsQueryData
from xms.adh.file_io.io_util import run_executable_with_tee


class SimRunner(RunBase):
    """Class that handles running AdH."""
    def __init__(self, _dummy_main_file=''):
        """Initializes the class.

        Args:
            _dummy_main_file (str): Unused, but here to keep constructor consistent with component classes.
        """
        super().__init__()
        self.sim_name = None
        self.file_location = ''
        self.geom_uuid = ''
        self.xms_temp_dir = ''
        self._logger = logging.getLogger('xms.adh')

    def get_executables(self, sim, query, file_location):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            file_location (str): The location of input files for the simulation.

        Returns:
            (:obj:`list` of :obj:`xms.api.dmi.ExecutableCommand`):
                The executable objects to run and the action requests that go with it.
        """
        xms_data = XmsQueryData(query)
        settings = SettingsManager(python_path=False)
        package = 'File_Location_Preferences'
        mpi_path = str(settings.get_setting(package, 'MsMPI', ''))

        script_path = os.path.normpath(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'AdH_RunModel.py'))

        # Set up command to run Pre-AdH
        pre_cmd = ExecutableCommand(
            executable=script_path,
            model='AdH',
            executable_order=0,
            display_name='Pre-AdH',
            run_weight=10,
            executable_is_script=True
        )
        exe_file = os.path.normpath(query.named_executable_path("PreAdHExec"))
        if not os.path.isfile(exe_file):
            message_box.message_with_ok(
                parent=None, message=f'Could not find the executable at {exe_file}.', app_name='SMS'
            )
            return []
        pre_cmd.add_commandline_arg(exe_file)
        pre_cmd.add_commandline_arg(xms_data.sim_name)

        # Set up command to run AdH
        cmd = ExecutableCommand(
            executable=script_path,
            model='AdH',
            executable_order=1,
            display_name='AdH',
            run_weight=90,
            progress_script='AdH_Progress.py',
            executable_is_script=True
        )

        model_control = xms_data.adh_data.model_control
        num_processors = model_control.param_control.operation_parameters.number_of_processors

        if num_processors > 1:  # Set up for a Parallel AdH run
            cpu_count = multiprocessing.cpu_count()
            cpu_count = '*' if num_processors > cpu_count else f'{num_processors}'
            cmd.add_commandline_arg(mpi_path)
            cmd.add_commandline_arg("-n")
            cmd.add_commandline_arg(str(cpu_count))
        exe_file = os.path.normpath(query.named_executable_path("AdHExec"))
        if not os.path.isfile(exe_file):
            message_box.message_with_ok(
                parent=None, message=f'Could not find the executable at {exe_file}.', app_name='SMS'
            )
            return []
        cmd.add_commandline_arg(exe_file)
        cmd.add_commandline_arg(xms_data.sim_name)
        cmd.add_solution_file(self._get_solution_load_actions(xms_data, file_location)[0])
        commands = [pre_cmd, cmd]
        return commands

    def get_solution_load_actions(self, sim, query, file_location):
        """Get the simulation load ActionRequests for any Simulation object given.

        This method is called when we are loading an existing solution from a previous model run. get_executables is
        called when running or rerunning a simulation.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with SMS.
            file_location (str): The location of input files for the simulation.

        Returns:
            (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): The solution load ActionRequests for the simulation
        """
        xms_data = XmsQueryData(query)
        return self._get_solution_load_actions(xms_data, file_location)

    def _get_solution_load_actions(self, xms_data: XmsData, file_location: str) -> list[ActionRequest]:
        """Get the simulation load ActionRequests for any Simulation object given.

        This method is called when we are loading an existing solution from a previous model run. get_executables is
        called when running or rerunning a simulation.

        Args:
            xms_data: The XMS data.
            file_location: The location of input files for the simulation.

        Returns:
            The solution load ActionRequests for the simulation
        """
        # Create an ActionRequest to load the solution located in file_location
        parameters = {
            'simulation_name': xms_data.sim_name,
            'file_location': file_location,
            'geom_uuid': xms_data.geom_uuid,
            'temp_dir': xms_data.xms_temp_directory
        }
        load_sol = ActionRequest(
            main_file='foo',
            modality='MODAL',
            class_name='SolutionReader',
            module_name='xms.adh.file_io.load_solution',
            method_name='read_solution',
            parameters=parameters
        )
        return [load_sol]


def launch_executable(args) -> int:
    """
    Launch a process to run a simulation executable.

    Args:
        args: The arguments used to run the executable.

    Returns:
        The return code of the executable.
    """
    child_proc_file = os.path.join(XmEnv.xms_environ_temp_directory(), f'{os.getpid()}_child_proc.txt')
    log_file = XmEnv.xms_environ_stdout_file()
    executable_path = Path(args[3] if 'mpiexec' in args[0] else args[0])
    model_log_file = f"{executable_path.stem}_output.txt"
    arguments = args.copy()
    return_code = run_executable_with_tee(arguments, log_file, model_log_file, child_proc_file)
    return return_code
