"""Dialog for populating a time series based on inputs."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from matplotlib.backends.backend_qt5agg import FigureCanvas
from matplotlib.figure import Figure
from PySide2.QtWidgets import QDialogButtonBox

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import CoverageItem, UGridItem
from xms.api.tree import tree_util
from xms.api.tree.tree_node import TreeNode
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.resources import resources_util

# 4. Local modules
from xms.adh.gui.populate_flow_dialog_ui import Ui_PopulateFlowDialog


class PopulateFlowDialog(XmsDlg):
    """Dialog for specifying a flow based on datasets inputs."""
    def __init__(self, xms_getter, parent=None):
        """Initializes the dialog.

        Args:
            xms_getter (PopulateFlowData): Handles getting data from xms.
            parent (QObject): The parent.
        """
        super().__init__(parent, 'xms.adh.gui.populate_flow_dialog')

        # Try to get the previously selected dataset first. If it no longer exists, do not restore
        # previously entered inputs.
        self.data = xms_getter

        self.ui = Ui_PopulateFlowDialog()
        self.ui.setupUi(self)
        self.ui.cbx_populate_method.addItems(self.data.population_methods)
        populate_index = self.data.population_methods.index(self.data.populate_method)
        self.ui.cbx_populate_method.setCurrentIndex(populate_index)
        self.ui.populate_method_stack.setCurrentIndex(populate_index)
        items = self.data.get_hydrograph_names()
        if items:
            self.ui.cbx_hydrographs.addItems(items)
            self.ui.cbx_hydrographs.setCurrentIndex(self.data.current_hydrograph)
        self.setMinimumSize(self.size())

        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Boundary_Condition_Assignment'

        # Update widget state
        self._update_widgets()

        # Map Plot Library
        self.figure = None  # matplotlib.figure Figure
        self.canvas = None  # matplotlib.backends.backend_qt5agg FigureCanvas
        self.axes = None  # matplotlib Axes
        self.initialize_plot()
        self.ui.plot_layout.addWidget(self.canvas)

        self.setup_connections()
        # format the table
        # self.compute_data()

    def setup_connections(self):
        """Setup signals and slots for the dialog's controls."""
        self.ui.button_box.helpRequested.connect(self.help_requested)
        self.ui.btn_geometry.pressed.connect(self.select_geometry)
        self.ui.btn_depth.pressed.connect(self.select_depth_dataset)
        self.ui.btn_velocity.pressed.connect(self.select_velocity_dataset)
        self.ui.btn_spatial_data.pressed.connect(self.select_spatial_data)
        self.ui.cbx_populate_method.currentIndexChanged[int].connect(self._set_population_method)
        self.ui.cbx_hydrographs.currentIndexChanged[int].connect(self._set_hydrograph)

    def select_geometry(self):
        """Launches the select geometry dialog and sets the cross section values."""
        dialog = TreeItemSelectorDlg(
            'Select Geometry',
            UGridItem,
            self.data.pe_tree,
            previous_selection=self.data.geometry_uuid,
            parent=self,
            selectable_xms_types=['TI_MESH2D']
        )
        if dialog.exec_():
            self.data.geometry_uuid = dialog.get_selected_item_uuid()
            self.setup_plot()
            self._update_widgets()

    def select_spatial_data(self):
        """Launches the select spatial data coverage dialog."""
        spatial_tree = TreeNode(other=self.data.pe_tree)  # Create a copy of the project explorer tree to filter.
        tree_util.filter_project_explorer(spatial_tree, lambda x: x.coverage_type in ['SPATIAL_DATA', ''])
        dialog = TreeItemSelectorDlg(
            'Select Spatial Data',
            CoverageItem,
            spatial_tree,
            previous_selection=self.data.spatial_data_uuid,
            parent=self
        )
        if dialog.exec_():
            self.data.spatial_data_uuid = dialog.get_selected_item_uuid()
            self.ui.cbx_hydrographs.clear()
            items = self.data.get_hydrograph_names()
            if items:
                self.ui.cbx_hydrographs.addItems(items)
            else:
                self.ui.cbx_hydrographs.setCurrentText('')
            self.data.current_hydrograph = 0
            self._update_widgets()

    def _set_population_method(self, method_index):
        """Sets the population method.

        Args:
            method_index (int): The current index of the populate method combobox.
        """
        self.ui.populate_method_stack.setCurrentIndex(method_index)
        self.data.populate_method = self.ui.cbx_populate_method.currentText()
        self.ui.grp_plot.setVisible(self.data.population_methods[method_index] == 'Flow')
        self._update_messages()
        self._update_ok_enabled()

    def _set_hydrograph(self, selection):
        """Sets the current hydrograph.

        Args:
            selection (int): The current selected index.
        """
        self.data.current_hydrograph = selection

    def _update_widgets(self):
        """Updates the enabled state and text of selector related widgets."""
        self.ui.txt_geometry_selected.setText(self.data.geometry_name)
        self.ui.txt_depth_selected.setText(self.data.depth_name)
        self.ui.txt_velocity_selected.setText(self.data.velocity_name)
        self.ui.txt_spatial_data_selected.setText(self.data.spatial_data_name)
        enable_depth = True if self.data.scalar_pe_tree else False
        self.ui.btn_depth.setEnabled(enable_depth)
        self.ui.txt_depth.setEnabled(enable_depth)
        self.ui.txt_depth_selected.setEnabled(enable_depth)
        enable_velocity = True if self.data.vector_pe_tree else False
        self.ui.btn_velocity.setEnabled(enable_velocity)
        self.ui.txt_velocity.setEnabled(enable_velocity)
        self.ui.txt_velocity_selected.setEnabled(enable_velocity)
        enable_hydrographs = True if self.data.spatial_data_uuid else False
        self.ui.cbx_hydrographs.setEnabled(enable_hydrographs)
        self.ui.txt_hydrograph.setEnabled(enable_hydrographs)
        self.ui.grp_plot.setVisible(self.data.population_methods[self.ui.cbx_populate_method.currentIndex()] == 'Flow')
        self._update_messages()
        self._update_ok_enabled()

    def _update_messages(self):
        """Updated the messages to the end user that are shown."""
        self.ui.txt_warnings.clear()
        messages = self.data.get_messages()
        for message in messages:
            self.ui.txt_warnings.append(message)

    def _update_ok_enabled(self):
        """Updated the enabled state of the ok button."""
        populate_method = self.ui.cbx_populate_method.currentIndex()
        enable = False
        if self.data.population_methods[populate_method] == 'Flow':
            if self.data.geometry_uuid and self.data.depth_uuid and self.data.velocity_uuid:
                enable = True
        elif self.data.population_methods[populate_method] == 'Hydrograph':
            names = self.data.get_hydrograph_names()
            if names:
                enable = True
        self.ui.button_box.button(QDialogButtonBox.Ok).setEnabled(enable)

    def select_depth_dataset(self):
        """Launches the select dataset dialog and sets the depth values."""
        storage_dict = {'uuid': self.data.depth_uuid}
        accepted = DatasetSelector.select_dataset(
            self, self.ui.txt_depth_selected, 'Select Depth', self.data.scalar_pe_tree,
            DatasetSelector.is_scalar_if_dset, storage_dict, 'uuid', resources_util.get_mesh_icon
        )
        if accepted:
            self.data.depth_uuid = storage_dict['uuid']
            self._update_messages()
            self._update_ok_enabled()

    def select_velocity_dataset(self):
        """Launches the select dataset dialog and sets the velocity values."""
        storage_dict = {'uuid': self.data.velocity_uuid}
        accepted = DatasetSelector.select_dataset(
            self, self.ui.txt_velocity_selected, 'Select Velocity', self.data.vector_pe_tree,
            DatasetSelector.is_vector_if_dset, storage_dict, 'uuid', resources_util.get_mesh_icon
        )
        if accepted:
            self.data.velocity_uuid = storage_dict['uuid']
            self._update_messages()
            self._update_ok_enabled()

    def setup_warning_table(self, warnings):
        """Sets up the warning table.

        Args:
            warnings (list of str): warnings from the calculations
        """
        self.ui.txt_warnings.clear()
        index = 0
        for warning in warnings:
            # add warning to the table
            self.ui.txt_warnings.append(warning)
            index += 1
        if index == 0:
            self.ui.txt_warnings.append("Computations completed with no warnings")

    def initialize_plot(self):
        """Sets up the plot."""
        self.figure = Figure()
        self.figure.set_layout_engine('tight')  # Frames the plots
        self.canvas = FigureCanvas(self.figure)
        self.ui.plt_channel = FigureCanvas(self.figure)
        self.setup_plot()

    def setup_plot(self):
        """Add a new plot to the data model and draw it."""
        if not self.axes:
            self.axes = self.figure.subplots()
        self.data.add_series(self.axes, 'Normalized Distance', 'Elevation')
        self.canvas.draw()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def get_dialog_data_dict(self):
        """Returns a dict of the dialog input field widgets."""
        return {}
