"""This dialog provides a way for the user specify a sediment diversion."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.number_corrector import NumberCorrector  # noqa: AQU103
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.gui.sediment_diversion_dialog_ui import Ui_SedimentDiversionDialog


class SedimentDiversionDialog(XmsDlg):
    """A dialog for specifying a sediment diversion."""
    def __init__(self, win_cont):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            win_cont (QWidget): Parent window
        """
        super().__init__(win_cont, 'xms.adh.gui.sediment_diversion_dialog')
        self.ui = Ui_SedimentDiversionDialog()
        self.ui.setupUi(self)
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Sediment_Transport_and_Bed_Layers'

        self.number_corrector = NumberCorrector(self)

        self.ui.button_box.helpRequested.connect(self.help_requested)

        self.setup_ui()

    def setup_ui(self):
        """Setup dialog widgets."""
        self.ui.snap_combo.addItems(['Edgestring', 'Midstring'])
        self.ui.top_withdrawal_edit.setValidator(QxDoubleValidator())
        self.ui.bottom_withdrawal_edit.setValidator(QxDoubleValidator())
        self.ui.bottom_main_edit.setValidator(QxDoubleValidator())
        self.ui.top_withdrawal_edit.installEventFilter(self.number_corrector)
        self.ui.bottom_withdrawal_edit.installEventFilter(self.number_corrector)
        self.ui.bottom_main_edit.installEventFilter(self.number_corrector)
        self.ui.is_sed_diversion.stateChanged[int].connect(self._is_sed_diversion_changed)

    def set_diversion(self, snap, top, bottom, bottom_main, is_diversion):
        """Sets the values for the dialog.

        Args:
            snap (str): Either 'Edgestring' or 'Midstring'.
            top (float): Value for top elevation of the zone of withdrawal.
            bottom (float): Value for bottom elevation of the zone of withdrawal.
            bottom_main (float): Value for bottom elevation of the main channel.
            is_diversion (bool): True if this is an existing sediment diversion.
        """
        self.ui.snap_combo.setCurrentText(snap)
        self.ui.top_withdrawal_edit.setText(str(top))
        self.ui.bottom_withdrawal_edit.setText(str(bottom))
        self.ui.bottom_main_edit.setText(str(bottom_main))
        self.ui.is_sed_diversion.setChecked(is_diversion)
        self._is_sed_diversion_changed(self.ui.is_sed_diversion.checkState())

    def get_diversion(self):
        """Gets the diversion values out of the widgets.

        Returns:
            A tuple consisting of: 'Edgestring' or 'Midstring', a float for the top elevation, a float for the
            bottom elevation, a float for the bottom elevation in the main channel, and True if this is a sediment
            diversion.
        """
        snap = self.ui.snap_combo.currentText()
        top = float(self.ui.top_withdrawal_edit.text())
        bottom = float(self.ui.bottom_withdrawal_edit.text())
        bottom_main = float(self.ui.bottom_main_edit.text())
        is_diversion = self.ui.is_sed_diversion.isChecked()
        return snap, top, bottom, bottom_main, is_diversion

    def _is_sed_diversion_changed(self, is_diversion):
        """Handles changes based on whether this is a sediment diversion or not.

        Args:
            is_diversion (int): 1 if this is a sediment diversion, 0 otherwise.
        """
        self.ui.snap_combo.setEnabled(is_diversion)
        self.ui.top_withdrawal_edit.setEnabled(is_diversion)
        self.ui.bottom_withdrawal_edit.setEnabled(is_diversion)
        self.ui.bottom_main_edit.setEnabled(is_diversion)
        self.ui.snap_label.setEnabled(is_diversion)
        self.ui.top_withdrawal_label.setEnabled(is_diversion)
        self.ui.bottom_withdrawal_label.setEnabled(is_diversion)
        self.ui.bottom_main_label.setEnabled(is_diversion)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Called when Ok is clicked."""
        return super().accept()

    def reject(self):
        """Called when Cancel is clicked."""
        return super().reject()
