"""This is a widget for displaying and editing sediment material names and display options."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, Qt, Signal
from PySide2.QtGui import QStandardItem, QStandardItemModel

# 3. Aquaveo modules
from xms.guipy.delegates.display_option import DisplayOptionDelegate
from xms.guipy.widgets.basic_table_widget import BasicTableWidget

# 4. Local modules
from xms.adh.gui.widgets.color_list import ColorList


class SedimentMaterialTableWidget(BasicTableWidget):
    """Sediment material table widget class."""
    MAT_DISPLAY_COL = 0
    MAT_NAME_COL = 1
    added = Signal(int)
    deleted = Signal(list)

    def __init__(self, parent, data):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data (SedimentMaterialsIO): The sediment material data.
        """
        super().__init__(parent)
        self.model = QStandardItemModel()
        self.data = data
        self.setup_table_model()
        self.model.setHorizontalHeaderLabels(['', 'Material Name'])
        super()._setup_ui({0: DisplayOptionDelegate(self)}, True, False)
        self.table_view.verticalHeader().hide()

    def setup_table_model(self):
        """Sets up the model from the material objects."""
        for mat_id in self.data.materials.keys():
            mat_name = QStandardItem()
            mat_display = QStandardItem()
            mat = self.data.materials[mat_id].name
            display = self.data.materials[mat_id].display
            mat_name.setData(mat, Qt.EditRole)
            mat_name.setData(mat, Qt.DisplayRole)
            mat_name.setData(mat_id, Qt.UserRole)
            mat_display.setData(display, Qt.UserRole)
            if mat_id == 0:
                mat_name.setEditable(False)
            self.model.appendRow([mat_display, mat_name])

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        mat_id = self.data.add_material()
        mat_name = QStandardItem()
        mat_display = QStandardItem()
        row_idx = self.model.rowCount()
        mat = self.data.materials[mat_id].name
        display = self.data.materials[mat_id].display
        ColorList.get_next_color_and_texture(mat_id, display)
        mat_name.setData(mat, Qt.EditRole)
        mat_name.setData(mat, Qt.DisplayRole)
        mat_name.setData(mat_id, Qt.UserRole)
        mat_display.setData(display, Qt.UserRole)
        self.model.appendRow([mat_display, mat_name])
        new_index = self.model.index(row_idx, self.MAT_DISPLAY_COL)
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )
        self.added.emit(mat_id)

    def on_btn_delete_row(self):
        """Called when a row is removed from the table."""
        indices = self.table_view.selectionModel().selectedIndexes()
        next_select_row = -1
        delete_rows = {index.row() if index.isValid() else -1 for index in indices}
        # sort from largest row id to smallest
        sorted(delete_rows, reverse=True)
        deleted_ids = []
        for row in delete_rows:
            if row <= 0:
                break
            next_select_row = row - 1
            mat_id = self.model.data(self.model.index(row, self.MAT_NAME_COL), Qt.UserRole)
            self.data.materials.pop(mat_id)
            deleted_ids.append(mat_id)
            self.model.removeRows(row, 1)
        if next_select_row >= 0:
            select_index = self.table_view.model().index(next_select_row, self.MAT_NAME_COL)
            self.table_view.selectionModel().setCurrentIndex(
                select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
            )
        self.deleted.emit(deleted_ids)

    def _enable_delete(self, selected, deselected):
        """Enable/disable delete row button.

        Args:
            selected (QItemSelection): Selected indices
            deselected (QItemSelection): Deselected indices, unused

        """
        indices = selected.indexes()
        enable_delete = False
        if len(indices) > 0 and indices[0].isValid():
            enable_delete = True
        # disable the delete button if the unassigned/OFF material row (0) is selected
        for idx in indices:
            if idx.row() == 0:
                enable_delete = False
                break
        self.toolbar.widgetForAction(self.btn_actions[self.delete_icon]).setEnabled(enable_delete)
