"""Module for get_model function."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.tool_core.table_definition import FloatColumnType, TableDefinition

# 4. Local modules


def get_model() -> GenericModel:
    """Get the model."""
    model = GenericModel()

    _add_global_parameters(model)
    _add_model_parameters(model)
    _add_arc_parameters(model)
    return model


def _add_global_parameters(model: GenericModel):
    """Add the global parameters to the model."""
    section = model.global_parameters

    group = section.add_group('timing', 'Timing', is_active=True)
    group.add_float('dt', 'dt', default=60.0, description='[s] Time step size')
    group.add_float('tstart', 'tstart', default=0.0, description='[s] Start time of simulation')
    group.add_float('tstop', 'tstop', default=3600.0, description='[s] End time of simulation')
    group.add_float('restart', 'restart', default=-1.0, description='[s] Interval for which to write restart files')
    group.add_text(
        'refdate', 'refdate', default='2020-01-01 00:00', description='[-] Reference datetime in netCDF output'
    )

    group = section.add_group('grid_files', 'Grid files', is_active=True)
    group.add_text(
        'xgrid_file',
        'xgrid_file',
        default='xgrid.txt',
        description='Filename of ASCII file with x-coordinates of grid cells'
    )
    group.add_text(
        'ygrid_file',
        'ygrid_file',
        default='ygrid.txt',
        description='Filename of ASCII file with y-coordinates of grid cells'
    )
    group.add_text(
        'bed_file',
        'bed_file',
        default='bed.txt',
        description='Filename of ASCII file with bed level heights of grid cells'
    )
    group.add_text('ne_file', 'ne_file', default='ne.txt', description='Filename of ASCII file with non-erodible layer')
    group.add_text(
        'veg_file', 'veg_file', default='veg.txt', description='Filename of ASCII file with initial vegetation density'
    )
    group.add_text(
        'threshold_file',
        'threshold_file',
        default='threshold.txt',
        description='Filename of ASCII file with shear velocity threshold'
    )

    group.add_text(
        'threshold_mask',
        'threshold_mask',
        default='threshold_mask.txt',
        description='Filename of ASCII file with mask for the shear velocity threshold'
    )

    group = section.add_group('time_series', 'Time series', is_active=True)

    group.add_text(
        'wind_file',
        'wind_file',
        default='wind.txt',
        description='Filename of ASCII file with time series of wind velocity and direction'
    )
    group.add_text(
        'bedcomp_file',
        'bedcomp_file',
        default='bedcomp.txt',
        description='Filename of ASCII file with initial bed composition'
    )
    group.add_text(
        'meteo_file',
        'meteo_file',
        default='meteo.txt',
        description='Filename of ASCII file with time series of meteorlogical conditions'
    )
    group.add_text(
        'runup_mask', 'runup_mask', default='runup_mask.txt', description='Filename of ASCII file with mask for run-up'
    )
    group.add_text(
        'gw_mask',
        'gw_mask',
        default='gw_mask.txt',
        description='Filename of ASCII file with mask for the groundwater level'
    )

    group = section.add_group('output', 'Output', is_active=True)
    group.add_float(
        'output_times', 'output_times', default=60.0, description='[s] Output interval in seconds of simulation time'
    )
    group.add_text('output_file', 'output_file', default='output.txt', description='Filename of netCDF4 output file')
    # TODO group.add_table('output_vars', 'output_vars',
    #                 default=['zb', 'zs', 'Ct', 'Cu', 'uw', 'udir', 'uth', 'masspickup', 'w'],
    #                 description='Names of spatial grids to be included in output')

    group = section.add_group('processes', 'Processes', is_active=True)
    group.add_boolean('process_wind', 'process_wind', default=True, description='Enable the process of wind')
    group.add_boolean('process_shear', 'process_shear', default=False, description='Enable the process of wind shear')

    group.add_boolean('process_runup', 'process_runup', default=False, description='Enable the process of wave runup')
    group.add_boolean('process_moist', 'process_moist', default=False, description='Enable the process of moist')
    group.add_boolean(
        'process_mixtoplayer', 'process_mixtoplayer', default=False, description='Enable the process of mixing '
    )
    group.add_boolean(
        'process_threshold', 'process_threshold', default=True, description='Enable the process of threshold'
    )
    group.add_boolean(
        'process_transport', 'process_transport', default=True, description='Enable the process of transport'
    )
    group.add_boolean(
        'process_bedupdate', 'process_bedupdate', default=True, description='Enable the process of bed updating'
    )
    group.add_boolean('process_meteo', 'process_meteo', default=False, description='Enable the process of meteo')
    group.add_boolean('process_salt', 'process_salt', default=False, description='Enable the process of salt')
    group.add_boolean(
        'process_humidity', 'process_humidity', default=False, description='Enable the process of humidity'
    )
    group.add_boolean(
        'process_avalanche', 'process_avalanche', default=False, description='Enable the process of avalanching'
    )
    group.add_boolean('process_inertia', 'process_inertia', default=False, description='NEW')
    group.add_boolean(
        'process_separation',
        'process_separation',
        default=False,
        description='Enable the including of separation bubble'
    )
    group.add_boolean(
        'process_vegetation', 'process_vegetation', default=False, description='Enable the process of vegetation'
    )

    group.add_boolean(
        'process_dune_erosion',
        'process_dune_erosion',
        default=False,
        description='Enable the process of wave-driven dune erosion'
    )
    group.add_boolean(
        'process_wet_bed_reset',
        'process_wet_bed_reset',
        default=False,
        description='Enable the process of bed-reset in the intertidal zone'
    )
    group.add_boolean(
        'process_groundwater', 'process_groundwater', default=False, description='Enable the process of groundwater'
    )
    group.add_boolean(
        'process_scanning', 'process_scanning', default=False, description='Enable the process of scanning curves'
    )
    group.add_boolean(
        'process_seepage_face',
        'process_seepage_face',
        default=False,
        description='Enable the process of groundwater seepage (NB. only applicable to positive beach slopes)'
    )

    group = section.add_group('threshold', 'Threshold', is_active=True)
    group.add_boolean(
        'th_grainsize', 'th_grainsize', default=True, description='Enable wind velocity threshold based on grainsize'
    )
    group.add_boolean(
        'th_bedslope', 'th_bedslope', default=False, description='Enable wind velocity threshold based on bedslope'
    )
    group.add_boolean(
        'th_moisture', 'th_moisture', default=False, description='Enable wind velocity threshold based on moisture'
    )
    group.add_boolean(
        'th_drylayer', 'th_drylayer', default=False, description='Enable threshold based on drying of layer'
    )
    group.add_boolean(
        'th_humidity', 'th_humidity', default=False, description='Enable wind velocity threshold based on humidity'
    )
    group.add_boolean('th_salt', 'th_salt', default=False, description='Enable wind velocity threshold based on salt')
    group.add_boolean(
        'th_nelayer',
        'th_nelayer',
        default=False,
        description='Enable wind velocity threshold based on a non-erodible layer'
    )
    group.add_boolean(
        'th_sheltering',
        'th_sheltering',
        default=False,
        description='Enable wind velocity threshold based on sheltering by roughness elements'
    )

    group = section.add_group('physics', 'Physics', is_active=True)
    group.add_float('g', 'g', default=9.81, description='[m/s^2] Gravitational constant')
    group.add_float('v', 'v', default=1.5e-05, description='[m^2/s] Air viscosity  ')
    group.add_float('rhoa', 'rhoa', default=1.225, description='[kg/m^3] Air density')
    group.add_float('rhog', 'rhog', default=2650.0, description='[kg/m^3] Grain density')
    group.add_float('rhow', 'rhow', default=1025.0, description='[kg/m^3] Water density')
    group.add_float('porosity', 'porosity', default=0.4, description='[-] Sediment porosity')
    group.add_float('cpair', 'cpair', default=0.0010035, description='[MJ/kg/oC] Specific heat capacity air')

    group = section.add_group('sediment', 'Sediment', is_active=True)
    group.add_integer('nlayers', 'nlayers', default=3, description='[-] Number of bed layers')
    group.add_float('layer_thickness', 'layer_thickness', default=0.01, description='[m] Thickness of bed layers')
    # TODO group.add_table('grain_dist', 'grain_dist', default=[1.0],
    #                 description='[-] Initial distribution of sediment fractions')
    # group.add_table('grain_size', 'grain_size', default=[0.000225],
    #                 description='[m] Average grain size of each sediment fraction')

    group = section.add_group('wind_shear', 'Wind and shear', is_active=True)
    group.add_text(
        'wind_convention',
        'wind_convention',
        default='nautical',
        description='Convention used for the wind direction in the input files (cartesian or nautical)'
    )
    group.add_integer(
        'alfa', 'alfa', default=0, description='[deg] Real-world grid cell orientation wrt the North (clockwise)'
    )
    group.add_float('k', 'k', default=0.001, description='[m] Bed roughness')
    group.add_float('z', 'z', default=10.0, description='[m] Measurement height of wind velocity')
    group.add_float('kappa', 'kappa', default=0.41, description='[-] Von Kármán constant')
    group.add_float('h', 'h', default=-999.9, description='[m] Representative height of saltation layer')
    group.add_float('L', 'L', default=100.0, description='[m] Typical length scale of dune feature (perturbation)')
    group.add_float('l', 'l', default=10.0, description='[m] Inner layer height (perturbation)')

    group = section.add_group('transport', 'Transport', is_active=True)
    group.add_float('bi', 'bi', default=1.0, description='[-] Bed interaction factor')
    group.add_text(
        'method_transport',
        'method_transport',
        default='bagnold',
        description='Name of method to compute equilibrium sediment transport rate'
    )
    group.add_text(
        'method_grainspeed',
        'method_grainspeed',
        default='windspeed',
        description='Name of method to assume/compute grainspeed (windspeed, duran, constant)'
    )
    group.add_float(
        'Aa',
        'Aa',
        default=0.085,
        description='[-] Constant in formulation for wind velocity threshold based on grain size'
    )
    group.add_float(
        'Cb',
        'Cb',
        default=1.5,
        description='[-] Constant in bagnold formulation for equilibrium sediment concentration'
    )
    group.add_float(
        'Ck',
        'Ck',
        default=2.78,
        description='[-] Constant in kawamura formulation for equilibrium sediment concentration'
    )
    group.add_float(
        'Cl',
        'Cl',
        default=6.7,
        description='[-] Constant in lettau formulation for equilibrium sediment concentration'
    )
    group.add_float(
        'Cdk', 'Cdk', default=5.0, description='[-] Constant in DK formulation for equilibrium sediment concentration'
    )

    group = section.add_group('solver', 'Solver', is_active=True)
    group.add_float('T', 'T', default=1.0, description='[s] Adaptation time scale in advection equation')
    group.add_text(
        'solver',
        'solver',
        default='trunk',
        description='Name of the solver (trunk, pieter, steadystate,steadystatepieter)'
    )
    group.add_float('CFL', 'CFL', default=1.0, description='[-] CFL number to determine time step in explicit scheme')
    group.add_float('accfac', 'accfac', default=1.0, description='[-] Numerical acceleration factor')
    group.add_text(
        'scheme',
        'scheme',
        default='euler_backward',
        description='Name of numerical scheme (euler_forward, euler_backward or crank_nicolson)'
    )
    group.add_float(
        'max_error',
        'max_error',
        default=1e-06,
        description='[-] Maximum error at which to quit iterative solution in implicit numerical schemes'
    )
    group.add_integer(
        'max_iter',
        'max_iter',
        default=1000,
        description='[-] Maximum number of iterations at which to quit iterative solution in implicit numerical schemes'
    )
    description = (
        '[m] Maximum bedlevel change after one timestep. Next timestep dt will be modified '
        '(use 999. if not used)'
    )
    group.add_float('max_bedlevel_change', 'max_bedlevel_change', default=999.0, description=description)

    group = section.add_group('boundary_conditions', 'Boundary conditions', is_active=True)
    group.add_text(
        'boundary_onshore',
        'boundary_onshore',
        default='gradient',
        description='Name of onshore boundary conditions (flux, constant, uniform, gradient)'
    )
    group.add_text(
        'boundary_lateral',
        'boundary_lateral',
        default='circular',
        description='Name of lateral boundary conditions (circular, constant ==noflux)'
    )
    group.add_text(
        'boundary_offshore',
        'boundary_offshore',
        default='constant',
        description='Name of offshore boundary conditions (flux, constant, uniform, gradient)'
    )
    description = (
        '[-] Factor to determine offshore boundary flux as a function of Q0 '
        '(= 1 for saturated flux , = 0 for noflux)'
    )
    group.add_float('offshore_flux', 'offshore_flux', default=0.0, description=description)
    group.add_float(
        'constant_offshore_flux',
        'constant_offshore_flux',
        default=0.0,
        description='[kg/m/s] Constant input flux at offshore boundary'
    )
    description = (
        '[-] Factor to determine onshore boundary flux as a function of Q0 '
        '(= 1 for saturated flux , = 0 for noflux)'
    )
    group.add_float('onshore_flux', 'onshore_flux', default=0.0, description=description)
    group.add_float(
        'constant_onshore_flux',
        'constant_onshore_flux',
        default=0.0,
        description='[kg/m/s] Constant input flux at offshore boundary'
    )
    description = (
        '[-] Factor to determine lateral boundary flux as a function of Q0 '
        '(= 1 for saturated flux , = 0 for noflux)'
    )
    group.add_float('lateral_flux', 'lateral_flux', default=0.0, description=description)
    group.add_float(
        'sedimentinput',
        'sedimentinput',
        default=0.0,
        description='[-] Constant boundary sediment influx (only used in solve_pieter)'
    )

    group = section.add_group('rotating_shear_grid', 'Rotating shear-grid', is_active=True)
    group.add_float('dx', 'dx', default=1.0, description='')
    group.add_float('dy', 'dy', default=1.0, description='')
    group.add_integer(
        'buffer_width',
        'buffer_width',
        default=10,
        description='[m] Width of the bufferzone around the rotational grid for wind perturbation'
    )

    group = section.add_group('vegetation', 'Vegetation', is_active=True)
    group.add_float(
        'sigma',
        'sigma',
        default=4.2,
        description='[-] Ratio between basal area and frontal area of roughness elements'
    )
    group.add_float(
        'beta',
        'beta',
        default=130.0,
        description='[-] Ratio between drag coefficient of roughness elements and bare surface'
    )
    group.add_float(
        'gamma_vegshear',
        'gamma_vegshear',
        default=16.0,
        description='[-] Roughness factor for the shear stress reduction by vegetation'
    )
    description = '[s] Indication of the time period over which the bed level change is averaged for vegetation growth'
    group.add_float('avg_time', 'avg_time', default=86400.0, description=description)
    group.add_integer(
        'dzb_interval',
        'dzb_interval',
        default=86400,
        description='[s] Interval used for calcuation of vegetation growth'
    )
    group.add_float('hveg_max', 'hveg_max', default=1.0, description='[m] Max height of vegetation')
    group.add_float('dzb_opt', 'dzb_opt', default=0.0, description='[m/year] Sediment burial for optimal growth')
    group.add_float('V_ver', 'V_ver', default=0.0, description='[m/year] Vertical growth')
    group.add_float('V_lat', 'V_lat', default=0.0, description='[m/year] Lateral growth')
    group.add_float('germinate', 'germinate', default=0.0, description='[1/year] Possibility of germination per year')
    group.add_float('lateral', 'lateral', default=0.0, description='[1/year] Posibility of lateral expension per year')
    group.add_float('veg_gamma', 'veg_gamma', default=1.0, description='[-] Constant on influence of sediment burial')
    group.add_float(
        'veg_sigma',
        'veg_sigma',
        default=0.8,
        description='[-] Sigma in gaussian distrubtion of vegetation cover filter'
    )
    group.add_text(
        'vegshear_type',
        'vegshear_type',
        default='raupach',
        description='Choose the Raupach grid based solver (1D or 2D) or the Okin approach (1D only)'
    )
    group.add_integer(
        'veg_min_elevation',
        'veg_min_elevation',
        default=3,
        description='Choose the minimum elevation where vegetation can grow'
    )
    group.add_float(
        'okin_c1_veg',
        'okin_c1_veg',
        default=0.48,
        description='x/h spatial reduction factor in Okin model for use with vegetation'
    )
    group.add_float(
        'okin_initialred_veg',
        'okin_initialred_veg',
        default=0.32,
        description='initial shear reduction factor in Okin model for use with vegetation'
    )

    group = section.add_group('fences', 'Fences', is_active=True)
    group.add_float(
        'okin_c1_fence',
        'okin_c1_fence',
        default=0.48,
        description='x/h spatial reduction factor in Okin model for use with sand fence module'
    )
    group.add_float(
        'okin_initialred_fence',
        'okin_initialred_fence',
        default=0.32,
        description='initial shear reduction factor in Okin model for use with sand fence module'
    )

    group = section.add_group('separation', 'Separation', is_active=True)
    description = (
        '[-] Slope at the leeside of the separation bubble c = 0.2 according to Durán 2010 '
        '(Sauermann 2001: c = 0.25 for 14 degrees)'
    )
    group.add_float('c_b', 'c_b', default=0.2, description=description)
    group.add_integer(
        'mu_b', 'mu_b', default=30, description='[deg] Minimum required slope for the start of flow separation'
    )
    group.add_integer(
        'sep_filter_iterations',
        'sep_filter_iterations',
        default=0,
        description='[-] Number of filtering iterations on the sep-bubble (0 = no filtering)'
    )
    group.add_boolean(
        'zsep_y_filter',
        'zsep_y_filter',
        default=False,
        description='[-] Boolean for turning on/off the filtering of the separation bubble in y-direction'
    )

    group = section.add_group('soil_moisture', 'Soil moisture', is_active=True)
    group.add_float('Tdry', 'Tdry', default=5400.0, description='[s] Adaptation time scale for soil drying')
    group.add_float('eps', 'eps', default=0.001, description='[m] Minimum water depth to consider a cell "flooded"')

    group = section.add_group('salt', 'Salt', is_active=True)
    group.add_float('Tsalt', 'Tsalt', default=2592000.0, description='[s] Adaptation time scale for salinitation')
    group.add_float('csalt', 'csalt', default=0.035, description='[-] Maximum salt concentration in bed surface layer')

    group = section.add_group('waves', 'Waves', is_active=True)
    group.add_float('Tbedreset', 'Tbedreset', default=86400.0, description='[s] ')
    group.add_float('xi', 'xi', default=0.3, description='[-] Surf similarity parameter')
    group.add_float(
        'facDOD', 'facDOD', default=0.1, description='[-] Ratio between depth of disturbance and local wave height'
    )

    group = section.add_group('avalanching', 'Avalanching', is_active=True)
    group.add_float(
        'theta_dyn',
        'theta_dyn',
        default=33.0,
        description='[degrees] Initial Dynamic angle of repose, critical dynamic slope for avalanching'
    )
    group.add_float(
        'theta_stat',
        'theta_stat',
        default=34.0,
        description='[degrees] Initial Static angle of repose, critical static slope for avalanching'
    )
    group.add_integer(
        'max_iter_ava',
        'max_iter_ava',
        default=1000,
        description='[-] Maximum number of iterations at which to quit iterative solution in avalanching calculation'
    )

    #
    # TODO break
    #
    group = section.add_group('other', 'Other', is_active=True)
    group.add_boolean(
        'visualization',
        'visualization',
        default=False,
        description='Boolean for visualization of model interpretation before and just after initialization'
    )
    group.add_integer('nx', 'nx', default=0, description='[-] Number of grid cells in x-dimension')
    group.add_integer('ny', 'ny', default=0, description='[-] Number of grid cells in y-dimension')
    # TODO: Implement these
    # group.add_table('output_types', 'output_types', default=[],
    #                description='Names of statistical parameters to be included in output (avg, sum, var, min or max)')
    # group.add_table('external_vars', 'external_vars', default=[],
    #               description='Names of variables that are overwritten by an external (coupling) model, i.e. CoCoNuT')
    group.add_float('gamma', 'gamma', default=0.5, description='[-] Maximum wave height over depth ratio')
    group.add_float('fc', 'fc', default=0.11, description='[-] Moisture content at field capacity (volumetric)')
    group.add_float('w1_5', 'w1_5', default=0.02, description='[-] Moisture content at wilting point (gravimetric)')
    group.add_float(
        'resw_moist', 'resw_moist', default=0.01, description='[-] Residual soil moisture content (volumetric) '
    )
    group.add_float(
        'satw_moist', 'satw_moist', default=0.35, description='[-] Satiated soil moisture content (volumetric)'
    )
    group.add_float(
        'resd_moist', 'resd_moist', default=0.01, description='[-] Residual soil moisture content (volumetric) '
    )
    group.add_float(
        'satd_moist', 'satd_moist', default=0.5, description='[-] Satiated soil moisture content (volumetric) '
    )
    group.add_float(
        'nw_moist',
        'nw_moist',
        default=2.3,
        description='[-] Pore-size distribution index in the soil water retention function'
    )
    group.add_float(
        'nd_moist',
        'nd_moist',
        default=4.5,
        description='[-] Pore-size distribution index in the soil water retention function '
    )
    group.add_float(
        'mw_moist', 'mw_moist', default=0.57, description='[-] m, van Genucthen param (can be approximated as 1-1/n)'
    )
    group.add_float(
        'md_moist', 'md_moist', default=0.42, description='[-] m, van Genucthen param (can be approximated as 1-1/n)'
    )
    description = (
        '[cm^-1] Inverse of the air-entry value for a wetting branch of the soil water retention function '
        '(Schmutz, 2014)'
    )
    group.add_float('alfaw_moist', 'alfaw_moist', default=-0.07, description=description)
    description = (
        '[cm^-1] Inverse of the air-entry value for a drying branch of the soil water retention function '
        '(Schmutz, 2014)'
    )
    group.add_float('alfad_moist', 'alfad_moist', default=-0.035, description=description)
    group.add_float(
        'thick_moist', 'thick_moist', default=0.002, description='[m] Thickness of surface moisture soil layer'
    )
    group.add_float('K_gw', 'K_gw', default=0.00078, description='[m/s] Hydraulic conductivity (Schmutz, 2014)')
    group.add_float('ne_gw', 'ne_gw', default=0.3, description='[-] Effective porosity')
    group.add_integer('D_gw', 'D_gw', default=12, description='[m] Aquifer depth')
    group.add_integer(
        'tfac_gw', 'tfac_gw', default=10, description='[-] Reduction factor for time step in ground water calculations'
    )
    group.add_float('Cl_gw', 'Cl_gw', default=0.7, description='[m] Groundwater overheight due to runup')
    group.add_integer('in_gw', 'in_gw', default=0, description='[m] Initial groundwater level')
    group.add_integer(
        'GW_stat',
        'GW_stat',
        default=1,
        description='[m] Landward static groundwater boundary (if static boundary is defined)'
    )
    group.add_text(
        'boundary_gw',
        'boundary_gw',
        default='no_flow',
        description="Landward groundwater boundary, dGw/dx = 0 (or 'static')"
    )
    group.add_text(
        'method_moist_threshold',
        'method_moist_threshold',
        default='belly_johnson',
        description='Name of method to compute wind velocity threshold based on soil moisture content'
    )
    group.add_text(
        'method_moist_process',
        'method_moist_process',
        default='infiltration',
        description='Name of method to compute soil moisture content(infiltration or surface_moisture)'
    )
    description = (
        'Name of method to compute the roughness height z0. '
        'Note that here the z0 = k, which does not follow the definition of Nikuradse where z0 = k/30.'
    )
    group.add_text('method_roughness', 'method_roughness', default='constant', description=description)
    group.add_text(
        'callback',
        'callback',
        default='',
        description="Reference to callback function (e.g. example/callback.py':callback)"
    )
    group.add_integer(
        'dune_toe_elevation',
        'dune_toe_elevation',
        default=3,
        description='Choose dune toe elevation, only used in the PH12 dune erosion solver'
    )
    group.add_float(
        'beach_slope',
        'beach_slope',
        default=0.1,
        description='Define the beach slope, only used in the PH12 dune erosion solver'
    )
    group.add_text('veggrowth_type', 'veggrowth_type', default='orig', description="'orig', 'duranmoore14'")
    group.add_float(
        'rhoveg_max',
        'rhoveg_max',
        default=0.5,
        description='maximum vegetation density, only used in duran and moore 14 formulation'
    )
    group.add_integer(
        't_veg',
        't_veg',
        default=3,
        description='time scale of vegetation growth (days), only used in duran and moore 14 formulation'
    )
    group.add_integer('v_gam', 'v_gam', default=1, description='only used in duran and moore 14 formulation')


def _add_model_parameters(model: GenericModel):
    """Add the model control parameters, which are used for the process dialog."""
    section = model.model_parameters

    # Tides
    group = section.add_group('process_tide', 'Tides')
    group.add_text(
        'tide_file',
        'tide_file',
        default='tide.txt',
        description='Filename of ASCII file with time series of water levels'
    )
    group.add_text(
        'tide_mask',
        'tide_mask',
        default='tide_mask.txt',
        description='Filename of ASCII file with mask for tidal elevation'
    )
    group.add_curve(
        'tide_series',
        'Tidal series',
        axis_titles=['Time (s)', 'Water Elevation (m)'],
        description='Information about the water elevation due to tides. Time is from simulation start.'
    )

    # Sand fencing
    group = section.add_group('process_fences', 'Sand fencing')
    group.add_text(
        'fence_file',
        'fence_file',
        default='fence.txt',
        description='Filename of ASCII file with sand fence location/height (above the bed)'
    )

    # Waves
    group = section.add_group('process_wave', 'Waves')
    group.add_text(
        'wave_file',
        'wave_file',
        default='wave.txt',
        description='Filename of ASCII file with time series of wave heights'
    )
    group.add_text(
        'wave_mask',
        'wave_mask',
        default='wave_mask.txt',
        description='Filename of ASCII file with mask for wave height'
    )

    definition = [
        FloatColumnType(header='Time (s)', default=0.0, tool_tip='Time'),
        FloatColumnType(header='Wave Height (m)', default=0.0, tool_tip='Wave height'),
        FloatColumnType(header='Peak Period (s)', default=0.0, tool_tip='Peak Period'),
    ]
    table_definition = TableDefinition(definition)
    group.add_table('wave_table', 'Wave table', default=[], table_definition=table_definition, display_table=False)


def _add_arc_parameters(model: GenericModel):
    """Add the arc parameters to the model."""
    section = model.arc_parameters
    section.add_group('sand_fence', 'Sand Fence')
    section.add_group('wave', 'Wave')
    section.add_group('tide', 'Water Level')
