"""Python wrapping for xms.api._xmsapi.dmi.FilterBy."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.api._xmsapi.dmi import FilterBy as CFilterBy, NeighborCriterion

NeighborCriterionStrings = {
    'SHARED_COMPONENT': NeighborCriterion.SHARED_COMPONENT,
    'COMPONENT_OF_SAME_ENTITY': NeighborCriterion.COMPONENT_OF_SAME_ENTITY,
}


class FilterBy:
    """The pure Python wrapper for C++ exposed xms.api._xmsapi.dmi.FilterBy objects."""
    def __init__(self, instance=None):
        """Construct the wrapper.

        Args:
            instance (xms.api._xmsapi.dmi.FilterBy): The C++ object to wrap
        """
        if instance is None:
            self._instance = CFilterBy()
        else:
            self._instance = instance

    @property
    def location(self):
        """Returns the xms.data_objects.parameters.FilterLocation mask used by *this as an int."""
        return self._instance.GetLocation()

    @property
    def neighbor_by(self):
        """Returns the xms.api.dmi.NeighborBy mask used by *this as an int."""
        return self._instance.GetNeighborBy()

    @property
    def neighbor_criterion(self):
        """Returns the Python string constant associated with the FilterBy object's C++ neighbor criterion enum."""
        # Convert the C++ enum to it's associated Python string constant.
        c_enum = self._instance.GetNeighborCriterion()
        return next(key for key, value in NeighborCriterionStrings.items() if value == c_enum)

    @neighbor_criterion.setter
    def neighbor_criterion(self, criterion):
        """Sets the Python string constant associated with the FilterBy object's C++ neighbor criterion enum.

        Args:
            criterion (str): One of the NeighborCriterionStrings keys
        """
        # Convert the Python string constant to the associated C++ enum DialogModalityStrings keys.
        criterion_enum = NeighborCriterionStrings.get(criterion.upper())
        if criterion_enum is None:
            raise ValueError(f'criterion must be one of the following options: {NeighborCriterionStrings.keys()}')
        self._instance.SetNeighborCriterion(criterion_enum)

    def clear_location(self):
        """Clear all locations affected by this FilterBy object."""
        self._instance.ClearLocation()

    def clear_neighbor_by(self):
        """Clear all neighbors affected by this FilterBy object."""
        self._instance.ClearNeighborBy()

    def add_location(self, location):
        """Append a location value to this filter.

        Args:
            location (xms.data_objects.parameters.FilterLocation): The additional location to retrieve output from
        """
        self._instance.AddLocation(location)

    def add_neighbor_by(self, neighbor):
        """Append a neighbor location value to this filter.

        Args:
            neighbor (xms.api.dmi.NeighborBy): The neighbor to additionally consider
        """
        self._instance.AddLocation(neighbor)
