"""Class for building a UGrid representation of a 3D Bridge."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from math import sqrt
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.coverage.grid.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.extractor.ugrid_2d_data_extractor import UGrid2dDataExtractor
from xms.grid.ugrid import UGrid
from xms.grid.ugrid import ugrid_utils
from xms.tool.algorithms.earcut.earcut import earcut

# 4. Local modules


class GridBuilder:
    """Builds a 3D grid representing the bridge."""
    def __init__(self):
        """Initializes the class."""
        self.arc_upstream = None  # The upstream bridge arc
        self.arc_downstream = None  # The downstream bridge arc

        self.upstream_line = None  # Upstream profile line in t,y coordinates
        self.downstream_line = None  # Downstream profile line in t,y coordinates
        self.top_line = None  # Top profile line in t,y coordinates
        self.top_line_dn = None  # down stream top (only used to create the bottom half of 3d culvert)

        self.arc_upstream_t_values = None  # Upstream arc parametric t values for points defining arc
        self.arc_downstream_t_values = None  # Downstream arc parametric t values for points defining arc
        self.upstream_line_idxs = None  # Upstream profile line in x,y,z coordinates
        self.downstream_line_idxs = None  # Downstream profile line in x,y,z coordinates
        self.top_line_up_idxs = None  # Top upstream profile line in x,y,z coordinates
        self.top_line_down_idxs = None  # Top downstream profile line in x,y,z coordinates
        self.end0_top = None
        self.end0_bot = None
        self.end1_top = None
        self.end1_bot = None

        self._df_upstream = None
        self._df_downstream = None
        self._df_top = None
        self._df_top_dn = None
        self._bridge_mesh = None
        self._pier_elev = 0.0
        self._bot_bridge_file = ''
        self._match_parametric_values = True

        self.points = []  # All the xyz points of the grid
        self.cell_stream = []  # The cell stream. cell type, num points, point numbers

    def get_parametric_line(self, data_frame):
        """Extract a parametric line definition from the XY Series DataFrame.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The XY Series editor table data

        Returns:
            (:obj:`list`): List of t,y tuples defining the parametric line

        """
        df = data_frame.sort_values(data_frame.columns[0])
        start_x = df.iloc[0, 0]
        row_count = df.shape[0]
        length = df.iloc[row_count - 1, 0] - start_x
        if length == 0:
            length = 1.0
        line = []
        for row in range(row_count):
            distance = df.iloc[row, 0] - start_x
            elevation = df.iloc[row, 1]
            line.append((distance / length, elevation))
        if len(line) > 0:
            if line[-1][0] != 1.0:
                line.append((1.0, line[-1][1]))
        return line

    def _create_parametric_lines(self):
        """Creates the parametric lines (t values) from the profile lines.

        The parametric lines are the distances (t values) between 0 and 1.
        """
        self.upstream_line = self.get_parametric_line(self._df_upstream)
        self.downstream_line = self.get_parametric_line(self._df_downstream)
        self.top_line = self.get_parametric_line(self._df_top)
        if self._df_top_dn is not None:
            self.top_line_dn = self.get_parametric_line(self._df_top_dn)
        if self.arc_upstream:
            self.arc_upstream_t_values = self._parametric_values_from_line(self.arc_upstream)
        if self.arc_downstream:
            self.arc_downstream_t_values = self._parametric_values_from_line(self.arc_downstream)

    def _get_set_of_t(self, line):
        """Returns a set of the parametric lengths (t values) for the line.

        Args:
            line (:obj:`list[tuples(x,y)]`): A profile line.

        Returns:
            (:obj:`set`): The unique parametric lengths in a line
        """
        return {pt[0] for pt in line}

    def _get_list_of_all_parametric_lengths(self):
        """Returns a set of all the parametric lengths (t values) for all lines.

        Returns:
            (:obj:`set`): t values.
        """
        set_t = self._get_set_of_t(self.upstream_line)
        set_t = set_t | self._get_set_of_t(self.downstream_line)
        set_t = set_t | self._get_set_of_t(self.top_line)
        if self.top_line_dn is not None:
            set_t = set_t | self._get_set_of_t(self.top_line_dn)
        if self.arc_upstream:
            set_t = set_t | set(self.arc_upstream_t_values)
        if self.arc_downstream:
            set_t = set_t | set(self.arc_downstream_t_values)
        return sorted(set_t)

    def add_parametric_lengths_to_line(self, all_t, line):
        """Add parametric lengths (t values) from all_t into line, interpolating y values.

        Args:
            line (:obj:`list[tuple(t,y)]`) : A parametric profile line. Gets modified.
            all_t (:obj:`list[t values]`): List of all t values
        """
        curr = 1
        new_line = [line[0]]
        for i in range(1, len(all_t) - 1):
            t_all = all_t[i]
            t_line = line[curr][0]
            if t_all == t_line:
                new_line.append(line[curr])
                continue

            while t_all > t_line:
                curr += 1
                t_line = line[curr][0]

            y1 = line[curr][1]
            y0 = line[curr - 1][1]
            t_prev = line[curr - 1][0]
            factor_t = (t_all - t_prev) / (t_line - t_prev)
            y_new = y0 + (y1 - y0) * factor_t
            new_point = (t_all, y_new)
            new_line.append(new_point)

        new_line.append(line[-1])
        line[:] = new_line

    def _add_parametric_lengths_to_lines(self, all_t):
        """Add parametric lengths (t values) from all_t into all lines, interpolating y values.

        Args:
            all_t (:obj:`list`): The parametric lengths to add to all lines
        """
        self.add_parametric_lengths_to_line(all_t, self.upstream_line)
        self.add_parametric_lengths_to_line(all_t, self.downstream_line)
        self.add_parametric_lengths_to_line(all_t, self.top_line)
        if self.top_line_dn is not None:
            self.add_parametric_lengths_to_line(all_t, self.top_line_dn)

    def _parametric_values_from_line(self, line):
        """Computes the parametric distance in (x, y) at each point along a line.

        Args:
            line (:obj:`list[tuple(t,y)]`)

        Returns:
            t_values (:obj:`list[float]`): List of parametric values
        """
        # calculate the cumulative distance along the line
        dist = 0
        line_dist = []
        for i in range(1, len(line)):
            p0 = line[i - 1]
            p1 = line[i]
            dist += sqrt((p1[0] - p0[0])**2 + (p1[1] - p0[1])**2)
            line_dist.append(dist)

        # calculate the parametric distances
        t_values = [0.0]
        for i in range(0, len(line_dist)):
            t = line_dist[i] / line_dist[-1]
            t_values.append(t)

        return t_values

    def _xy_locations_from_parametric_values(self, t_values, xy_line, xy_line_t_values):
        """Computes the (x, y) locations from parametric values given a line and parametric values along that line.

        Args:
            t_values (:obj:`list[float]`): parametric values 0 to 1
            xy_line (:obj:`list[tuple(t,y)]`):
            xy_line_t_values (:obj:`list[float]`): parametric values of the points that make up the line

        Returns:
            xy_locations (:obj:`list[tuple(x,y)]`): List of xy locations
        """
        xy_locations = []
        curr = 0
        for i in range(0, len(t_values)):
            if t_values[i] == xy_line_t_values[curr]:
                xy_locations.append(xy_line[curr])
                continue

            while xy_line_t_values[curr] < t_values[i]:
                curr += 1

            x0 = xy_line[curr - 1][0]
            x1 = xy_line[curr][0]
            y0 = xy_line[curr - 1][1]
            y1 = xy_line[curr][1]
            t0 = xy_line_t_values[curr - 1]
            t1 = xy_line_t_values[curr]
            local_t = (t_values[i] - t0) / (t1 - t0)
            x = x0 + (x1 - x0) * local_t
            y = y0 + (y1 - y0) * local_t
            xy_locations.append((x, y))

        return xy_locations

    def _transform_line_to_world_space(self, arc, line):
        """Transforms the line from parametric space to world space, adding xyz points to self.points.

        Args:
            line (:obj:`list[tuple(t,y)]`): The parametric line definition to transform

        Returns:
            point_idxs (:obj:`list[int]`): List of indices (0-based) into self.points for the line.
        """
        arc_t_values = self._parametric_values_from_line(arc)
        line_t_values = [t[0] for t in line]
        xy_locations = self._xy_locations_from_parametric_values(line_t_values, arc, arc_t_values)
        point_idxs = []
        for i in range(0, len(line)):
            x = xy_locations[i][0]
            y = xy_locations[i][1]
            z = line[i][1]
            self.points.append((x, y, z))
            point_idxs.append(len(self.points) - 1)

        return point_idxs

    def _transform_to_world_space(self):
        """Transforms all lines from parametric space to world space."""
        self.top_line_up_idxs = self._transform_line_to_world_space(self.arc_upstream, self.top_line)
        self.upstream_line_idxs = self._transform_line_to_world_space(self.arc_upstream, self.upstream_line)
        if self.top_line_dn is not None:
            self.top_line_down_idxs = self._transform_line_to_world_space(self.arc_downstream, self.top_line_dn)
        else:
            self.top_line_down_idxs = self._transform_line_to_world_space(self.arc_downstream, self.top_line)
        self.downstream_line_idxs = self._transform_line_to_world_space(self.arc_downstream, self.downstream_line)

    def _make_face(self, line0, line1):
        """Makes a face of the bridge by connecting the two lines with cells.

        Lines must have the same number of points. Cells are made like this: ::


                    0       1       2
            line 0  *-------*-------*
                    |0   2/2|0   2/2|
                    |   /   |   /   |
                    |1/0   1|1/0   1|
            line 1  *-------*-------*
                    0       1       2

        """
        for i in range(len(line0) - 1):
            self.cell_stream.append(UGrid.cell_type_enum.TRIANGLE)
            self.cell_stream.append(3)
            self.cell_stream.append(line0[i])
            self.cell_stream.append(line1[i])
            self.cell_stream.append(line0[i + 1])

            self.cell_stream.append(UGrid.cell_type_enum.TRIANGLE)
            self.cell_stream.append(3)
            self.cell_stream.append(line1[i])
            self.cell_stream.append(line1[i + 1])
            self.cell_stream.append(line0[i + 1])

    def _make_cells(self):
        """Makes the cell stream for the 3D grid. Faces will be oriented outwards."""
        self._make_face(self.top_line_up_idxs, self.upstream_line_idxs)  # upstream
        self._make_face(self.downstream_line_idxs, self.top_line_down_idxs)  # downstream
        self._make_face(self.top_line_down_idxs, self.top_line_up_idxs)  # top
        self._make_face(self.upstream_line_idxs, self.downstream_line_idxs)  # bottom

        self.end0_top = [self.top_line_down_idxs[0], self.top_line_up_idxs[0]]
        self.end0_bot = [self.downstream_line_idxs[0], self.upstream_line_idxs[0]]
        self._make_face(self.end0_top, self.end0_bot)

        self.end1_top = [self.top_line_down_idxs[-1], self.top_line_up_idxs[-1]]
        self.end1_bot = [self.downstream_line_idxs[-1], self.upstream_line_idxs[-1]]
        self._make_face(self.end1_bot, self.end1_top)

    def _make_top_cells(self):
        """Makes the cell stream for the top grid."""
        self._make_face(self.top_line_down_idxs, self.top_line_up_idxs)  # top

    def _make_bottom_cells(self):
        """Makes the cell stream for the bottom grid."""
        self._make_face(self.upstream_line_idxs, self.downstream_line_idxs)  # bottom

    def _make_prisms(self):
        """Creates prism cells for xmsgrid."""
        top_up = self.top_line_up_idxs
        top_down = self.top_line_down_idxs
        bottom_up = self.upstream_line_idxs
        bottom_down = self.downstream_line_idxs
        for i in range(len(top_up) - 1):
            self.cell_stream.append(UGrid.cell_type_enum.WEDGE)
            self.cell_stream.append(6)

            self.cell_stream.append(bottom_down[i])
            self.cell_stream.append(bottom_up[i])
            self.cell_stream.append(bottom_down[i + 1])

            self.cell_stream.append(top_down[i])
            self.cell_stream.append(top_up[i])
            self.cell_stream.append(top_down[i + 1])

            self.cell_stream.append(UGrid.cell_type_enum.WEDGE)
            self.cell_stream.append(6)

            self.cell_stream.append(bottom_up[i])
            self.cell_stream.append(bottom_up[i + 1])
            self.cell_stream.append(bottom_down[i + 1])

            self.cell_stream.append(top_up[i])
            self.cell_stream.append(top_up[i + 1])
            self.cell_stream.append(top_down[i + 1])

    def _dot_product(self, vector1, vector2):
        """Returns the dot product."""
        return (vector1[0] * vector2[0]) + (vector1[1] * vector2[1])

    def _create_vector(self, point1, point2):
        """Returns the XY vector from point1 to point2.

        Args:
            point1 (:obj:`tuple(x,y)`)
            point2 (:obj:`tuple(x,y)`)

        Returns:
            (:obj:`tuple(x,y)`): tuple containing:

                x (:obj:`float`): X value.

                y (:obj:`float`): Y value.

        """
        return point2[0] - point1[0], point2[1] - point1[1]

    def _match_arc_direction(self):
        """Reverses downstream arc if necessary to point in same general direction as upstream arc."""
        vec_up = self._create_vector(self.arc_upstream[0], self.arc_upstream[-1])
        vec_down = self._create_vector(self.arc_downstream[0], self.arc_downstream[-1])

        dot = self._dot_product(vec_up, vec_down)
        if dot < 0.0:
            self.arc_downstream.reverse()

    def _turn(self, point1, point2, point3, angle_tol=0.0017453):
        """Determines if angle made by three points is a left, right, or colinear turn.

        See gmTurn.

        Args:
            point1 (:obj:`tuple(x,y)`): 1st point.
            point2 (:obj:`tuple(x,y)`): 2nd point.
            point3 (:obj:`tuple(x,y)`): 3rd point.
            angle_tol (:obj:`float`): Angle tolerance.

        Returns:
            0 = left
            1 = right
            2 = colinear 180 (continue onward)
            3 = colinear 0 (turns back on same segment)
        """
        # compute sin T = (v3-v2)x(v1-v2)/(d12*d32)
        dx32 = point3[0] - point2[0]
        dy32 = point3[1] - point2[1]
        dx12 = point1[0] - point2[0]
        dy12 = point1[1] - point2[1]
        d32 = sqrt(dx32 * dx32 + dy32 * dy32)
        d12 = sqrt(dx12 * dx12 + dy12 * dy12)
        mag = d12 * d32
        sint = (dx32 * dy12 - dx12 * dy32) / mag

        if sint > angle_tol:
            return 0  # TURN_LEFT
        else:
            return 1  # TURN_RIGHT

    def _orient_arcs_for_outward_normals(self):
        """Swaps upstream and downstream arcs if necessary for face normals to point outwards."""
        rv = self._turn(self.arc_upstream[0], self.arc_upstream[-1], self.arc_downstream[-1])
        if rv == 0:  # downstream to right of upstream
            self.arc_upstream, self.arc_downstream = self.arc_downstream, self.arc_upstream
            self._df_upstream, self._df_downstream = self._df_downstream, self._df_upstream

    def _build_geometry(self):
        """Builds a 2D or 3D grid representing the bridge."""
        self.points = []
        self.cell_stream = []
        self._match_arc_direction()
        self._orient_arcs_for_outward_normals()
        self._create_parametric_lines()
        if self._match_parametric_values:
            all_t = self._get_list_of_all_parametric_lengths()
            self._add_parametric_lengths_to_lines(all_t)
        self._transform_to_world_space()

    def write_parametric_profiles_for_bridge_scour(self, bridge, arc_upstream, arc_downstream, filename):
        """Super custom function used by SMS when exporting a bridge scour coverage for Hydraulic Toolbox.

        Basically does everything we do when generating the grid except for building the UGrid. SMS does not care
        about anything other than the upstream and top profiles for this operation.

        Args:
            bridge (:obj:`Bridge`): The bridge.
            arc_upstream (:obj:`list[tuple]`): Points of the upstream arc.
            arc_downstream (:obj:`list[tuple]`): Points of the downstream arc.
            filename (:obj:`str`): Path to the output file to write.
        """
        self._df_upstream = bridge.df_upstream
        self._df_downstream = bridge.df_downstream
        self._df_top = bridge.df_top
        self.arc_upstream = arc_upstream
        self.arc_downstream = arc_downstream

        self._match_arc_direction()
        self._orient_arcs_for_outward_normals()
        self._create_parametric_lines()
        all_t = self._get_list_of_all_parametric_lengths()
        self._add_parametric_lengths_to_lines(all_t)
        with open(filename, 'w') as file:
            file.write('TOP_PROFILE\n')
            for top_val in self.top_line:
                file.write(f'{top_val[0]} {top_val[1]}\n')
            file.write('UPSTREAM_PROFILE\n')
            for up_val in self.upstream_line:
                file.write(f'{up_val[0]} {up_val[1]}\n')

    def build(self, bridge, arc_upstream, arc_downstream, wkt, filename):
        """Builds a 3D grid representing the bridge.

        Args:
            bridge (:obj:`Bridge`): The bridge.
            arc_upstream (:obj:`list[tuple]`): Points of the upstream arc.
            arc_downstream (:obj:`list[tuple]`): Points of the downstream arc.
            wkt (:obj:`str`): projection string
            filename (:obj:`str`): Path to the grid file to write.

        Returns:
            (:obj:`UGrid`): ugrid
        """
        data = {
            'up_stream': bridge.df_upstream,
            'down_stream': bridge.df_downstream,
            'top': bridge.df_top,
            'up_arc': arc_upstream,
            'down_arc': arc_downstream,
            'wkt': wkt,
            'ug_filename': filename
        }
        return self.build_from_dict(data)

    def build_from_dict(self, data_dict):
        """Builds a 3D grid representing the bridge.

        Args:
            data_dict (:obj:`dict`): data needed to create the 3d ugrid

        Returns:
            (:obj:`UGrid`): ugrid
        """
        self._df_upstream = data_dict['up_stream']
        self._df_downstream = data_dict['down_stream']
        self._df_top = data_dict['top']
        if 'top_dn' in data_dict:
            self._df_top_dn = data_dict['top_dn']
        self.arc_upstream = data_dict['up_arc']
        self.arc_downstream = data_dict['down_arc']
        wkt = data_dict['wkt']
        filename = data_dict['ug_filename']
        main_file = data_dict.get('main_file', '')
        if main_file:
            self._bot_bridge_file = os.path.join(os.path.dirname(main_file), 'bottom2d.xmugrid')
        if 'match_parametric_values' in data_dict:
            self._match_parametric_values = data_dict['match_parametric_values']

        self._build_geometry()
        self._make_prisms()

        bridge_mesh = data_dict.get('bridge_mesh', None)
        if 'culvert_bottom_up_profile' in data_dict:  # make the bottom half of the 3D culvert
            self._gen_culvert_ugrid3d(data_dict)
        elif bridge_mesh is not None:
            self._bridge_mesh = bridge_mesh
            self._pier_elev = data_dict['pier_elev']
            self._add_piers_to_ugrid3d()

        ugrid = UGrid(self.points, self.cell_stream)
        ugrid_utils.write_ugrid_to_ascii_file(ugrid, filename)
        if len(wkt) > 0:
            prj_filename = os.path.splitext(filename)[0] + '.prj'
            with open(prj_filename, 'w') as prj_file:
                prj_file.write(wkt)
        return ugrid

    def _gen_culvert_ugrid3d(self, data_dict):
        """Creates the culvert ugrid.

        Args:
            data_dict (:obj:`dict`): data needed to create the 3d ugrid
        """
        pts = self.points
        cs = self.cell_stream
        self.points = self.cell_stream = []
        elev = data_dict['culvert_ug_base_elev']
        self._df_top = data_dict['culvert_bottom_dn_profile']
        self._df_top_dn = data_dict['culvert_bottom_up_profile']
        df = self._df_top.copy()
        df['Elevation'] = elev
        self._df_upstream = df
        df = self._df_top_dn.copy()
        df['Elevation'] = elev
        self._df_downstream = df
        self._build_geometry()
        self._make_prisms()
        # merge the grids
        pt_idx = {p: idx for idx, p in enumerate(pts)}
        new_idxs = [-1] * len(self.points)
        for i, p in enumerate(self.points):
            if p in pt_idx:
                new_idxs[i] = pt_idx[p]
            else:
                new_idxs[i] = len(pts)
                pts.append(p)
        cnt = 0
        while cnt < len(self.cell_stream):
            cell_type = self.cell_stream[cnt]
            n_pts = self.cell_stream[cnt + 1]
            start = cnt + 2
            end = start + n_pts
            cell_pts = self.cell_stream[start:end]
            new_cell_pts = [new_idxs[p] for p in cell_pts]
            cs.extend([cell_type, n_pts] + new_cell_pts)
            cnt = end
        self.points = pts
        self.cell_stream = cs

    def _add_piers_to_ugrid3d(self):
        """Adds piers from a 2d bridge mesh to the 3d ugrid."""
        # get the polygons from the bridge mesh
        cell_materials = [1] * self._bridge_mesh.ugrid.cell_count
        cov_builder = GridCellToPolygonCoverageBuilder(
            self._bridge_mesh, cell_materials, projection=None, coverage_name=''
        )
        polys = cov_builder.find_polygons()
        plist = polys[1][0]
        if len(plist) > 1:
            bot_bridge = ugrid_utils.read_ugrid_from_ascii_file(self._bot_bridge_file)
            ug_pts = bot_bridge.locations
            ug_elev = [p[2] for p in ug_pts]
            ug_act = [1] * len(ug_elev)
            extractor = UGrid2dDataExtractor(bot_bridge)
            extractor.set_grid_point_scalars(ug_elev, ug_act, 'points')
            plist = plist[1:]
            ug_pts = self._bridge_mesh.ugrid.locations
            for poly in plist:
                poly.reverse()
                poly.pop()
                pts = [xy for p in poly for xy in ug_pts[p][:2]]
                new_pts = [(ug_pts[p][0], ug_pts[p][1], self._pier_elev) for p in poly]
                start = len(self.points)
                offset = len(new_pts)
                self.points.extend(new_pts)
                extractor.extract_locations = new_pts
                top_z = extractor.extract_data()
                new_pts = [(p[0], p[1], z) for p, z in zip(new_pts, top_z)]
                self.points.extend(new_pts)
                ear_tris = earcut(pts)
                it = iter(ear_tris)
                new_tris = zip(it, it, it)
                for t in new_tris:
                    t1 = [tt + start for tt in t]
                    t2 = [tt + offset for tt in t1]
                    cell = [UGrid.cell_type_enum.WEDGE, 6] + t1 + t2
                    self.cell_stream.extend(cell)

    def build_top_and_bottom(self, bridge, arc_upstream, arc_downstream, wkt, main_file):
        """Builds a 3D grid representing the bridge.

        Args:
            bridge (:obj:`Bridge`): The bridge.
            arc_upstream (:obj:`list[tuple]`): Points of the upstream arc.
            arc_downstream (:obj:`list[tuple]`): Points of the downstream arc.
            wkt (:obj:`str`): projection string
            main_file (:obj:`str`): component main filename
        """
        data = {
            'up_stream': bridge.df_upstream,
            'down_stream': bridge.df_downstream,
            'top': bridge.df_top,
            'up_arc': arc_upstream,
            'down_arc': arc_downstream,
            'wkt': wkt,
            'main_file': main_file
        }
        self.build_top_and_bottom_from_dict(data)

    def build_top_and_bottom_from_dict(self, data_dict):
        """Builds a 3D grid representing the bridge.

        Args:
            data_dict (:obj:`dict`): Data used by the method
        """
        self._df_upstream = data_dict['up_stream']
        self._df_downstream = data_dict['down_stream']
        self._df_top = data_dict['top']
        if 'top_dn' in data_dict:
            self._df_top_dn = data_dict['top_dn']
        self.arc_upstream = data_dict['up_arc']
        self.arc_downstream = data_dict['down_arc']
        wkt = data_dict['wkt']
        main_file = data_dict['main_file']
        if 'match_parametric_values' in data_dict:
            self._match_parametric_values = data_dict['match_parametric_values']

        self._build_geometry()
        self.cell_stream.clear()
        self._make_top_cells()
        ugrid_top = UGrid(self.points, self.cell_stream)
        main_path = os.path.dirname(main_file)
        tempfile_top = os.path.join(main_path, 'top2d.xmugrid')
        ugrid_utils.write_ugrid_to_ascii_file(ugrid_top, tempfile_top)
        self.cell_stream.clear()
        self._make_bottom_cells()
        ugrid_bottom = UGrid(self.points, self.cell_stream)
        tempfile_bottom = os.path.join(main_path, 'bottom2d.xmugrid')
        ugrid_utils.write_ugrid_to_ascii_file(ugrid_bottom, tempfile_bottom)
        if len(wkt) > 0:
            prj_filename = os.path.splitext(tempfile_top)[0] + '.prj'
            with open(prj_filename, 'w') as prj_file:
                prj_file.write(wkt)
            prj_filename = os.path.splitext(tempfile_bottom)[0] + '.prj'
            with open(prj_filename, 'w') as prj_file:
                prj_file.write(wkt)
