"""The Arc properties tab in the Structure properties dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSortFilterProxyModel, Qt

# 3. Aquaveo modules
from xms.tool_core.table_definition import FloatColumnType, IntColumnType, StringColumnType, TableDefinition

# 4. Local modules
from xms.bridge.gui.struct_dlg.arc_props_base import ArcPropsBase

ARC_TYPE_BRIDGE = 'Bridge'
ARC_TYPE_PG = 'Pier group (PG)'
ARC_TYPE_WP = 'Wall pier (WP)'
ARC_TYPE_ABUTMENT = 'Abutment'

ELEM_WRAP_WIDTH = 'Element width\naround piers'
PG_DIAMETER = 'PG\nDiameter'
PG_NUM_PIERS = 'PG\n# piers'
PG_SPACING = 'PG\nspacing'
WP_WIDTH = 'WP width'
WP_LENGTH = 'WP length'
WP_END_TYPE = 'WP end type'


class StructureArcProps(ArcPropsBase):
    """The arc properties tab in the structure dialog."""
    def _table_definition(self):
        return _table_definition()

    def _get_filter_model(self):
        model = _ArcPropFilterModel(self._parent)
        model.set_table_def(self._table_def)
        return model

    def _get_default_arc_data(self, arc_ids):
        """Get the default arc data."""
        narc = len(arc_ids)
        default_vals = {
            self._col_names[0]: arc_ids,  # Arc ID
            self._col_names[1]: ['Unassigned'] * narc,  # Type
            self._col_names[2]: [0.0] * narc,  # Element width
            self._col_names[3]: [0.0] * narc,  # Pier group diameter
            self._col_names[4]: [1] * narc,  # Pier group number of piers
            self._col_names[5]: [0.0] * narc,  # Pier group spacing
            self._col_names[6]: [0.0] * narc,  # Wall pier width
            self._col_names[7]: [0.0] * narc,  # Wall pier length
            self._col_names[8]: ['Sharp'] * narc,  # Wall pier end type
        }

        return default_vals

    def _get_arc_property_csv_str(self):
        """Get the arc property CSV string."""
        return self._parent.data.data_dict.get('arc_properties', '')


class _ArcPropFilterModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)
        self.type_col = -1
        self.table_def = None
        self.elem_width_col = -1
        self.wall_pier_cols = {}
        self.pier_group_cols = {}

    def set_table_def(self, table_def):
        """Set the table definition for this class."""
        self.table_def = table_def
        col_names = [col.header for col in table_def.column_types]
        self.type_col = col_names.index('Type')
        self.elem_width_col = col_names.index(ELEM_WRAP_WIDTH)
        self.pier_group_cols = {
            col_names.index(PG_DIAMETER),
            col_names.index(PG_NUM_PIERS),
            col_names.index(PG_SPACING),
        }
        self.wall_pier_cols = {
            col_names.index(WP_WIDTH),
            col_names.index(WP_LENGTH),
            col_names.index(WP_END_TYPE),
        }

    def _row_col_from_index(self, index):
        """Get the row and column from the index."""
        return index.row(), index.column()

    def _arc_type_from_index(self, index, row):
        """Get the arc type from the index."""
        mod = index.model()
        type_col_index = mod.index(row, self.type_col)
        return type_col_index.data()

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (:obj:`QModelIndex`): The item's model index

        Returns:
            (:obj:`Qt.ItemFlags`): Updated flags for the item
        """
        ret_flags = super().flags(index)
        row, col = self._row_col_from_index(index)
        if col > self.type_col:
            arc_type = self._arc_type_from_index(index, row)
            if arc_type not in ['Wall pier (WP)', 'Pier group (PG)']:
                ret_flags = ret_flags & ~Qt.ItemIsEnabled
            else:
                enable_col = False
                if col == self.elem_width_col:
                    enable_col = True
                elif arc_type == 'Pier group (PG)' and col in self.pier_group_cols:
                    enable_col = True
                elif arc_type == 'Wall pier (WP)' and col in self.wall_pier_cols:
                    enable_col = True
                if enable_col:
                    ret_flags |= Qt.ItemIsEnabled
                else:
                    ret_flags = ret_flags & ~Qt.ItemIsEnabled
        return ret_flags


def _table_definition():
    """Returns a table definition used in this dialog."""
    choices = [
        'Unassigned',
        ARC_TYPE_BRIDGE,
        ARC_TYPE_PG,
        ARC_TYPE_WP,
        ARC_TYPE_ABUTMENT,
    ]
    pier_end_type = ['Square', 'Round', 'Sharp']
    return TableDefinition(
        [
            IntColumnType(header='Arc ID', enabled=False, default=0),
            StringColumnType(header='Type', choices=choices, default='Unassigned'),
            FloatColumnType(header=ELEM_WRAP_WIDTH, default=0.0),
            FloatColumnType(header=PG_DIAMETER, default=0.0),
            IntColumnType(header=PG_NUM_PIERS, default=1, low=1),
            FloatColumnType(header=PG_SPACING, default=0.0),
            FloatColumnType(header=WP_WIDTH, default=0.0),
            FloatColumnType(header=WP_LENGTH, default=0.0),
            StringColumnType(header=WP_END_TYPE, choices=pier_end_type, default='Sharp'),
        ]
    )


def arc_property_column_names():
    """Returns a list of the column names used in this dialog."""
    td = _table_definition()
    return [col.header for col in td.column_types]
