"""The help pane in the Structure properties dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QGroupBox, QHBoxLayout, QLabel, QLineEdit, QPushButton, QVBoxLayout
)

# 3. Aquaveo modules
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules


class StructureBridgeTab:
    """The profiles tab in the structure dialog."""
    def __init__(self, parent, widgets):
        """Initializes the class.

        Args:
            parent (:obj:`QWidget`): Parent dialog
            widgets (:obj:`dict`): Dictionary of widgets
        """
        self._parent = parent
        self._widgets = widgets
        self._add_widgets()

    def _add_widgets(self):
        """Set up the UI."""
        vlayout = self._widgets['tab1_scroll_area_vert_layout']
        data = self._parent.data

        # option to assign properties to individual arcs
        self._widgets['tog_assign_arcs'] = QCheckBox('Assign properties to individual arcs')
        vlayout.addWidget(self._widgets['tog_assign_arcs'])
        self._widgets['tog_assign_arcs'].stateChanged.connect(self.on_assign_arcs)
        self._widgets['grp_arc_props'] = QGroupBox('Arc properties')
        vlayout.addWidget(self._widgets['grp_arc_props'])
        self._widgets['vlayout_arc_props'] = QVBoxLayout()
        vlayout_grp = self._widgets['vlayout_arc_props']
        self._widgets['grp_arc_props'].setLayout(vlayout_grp)
        self._parent.arc_props.add_widgets(vlayout_grp)

        # bridge width
        self._parent.add_floating_point_widget(
            'width', 'Bridge width (upstream to downstream):', data.data_dict['bridge_width'], vlayout
        )
        # manning's n for underside of bridge
        self._parent.add_floating_point_widget(
            'manning', 'Bridge ceiling manning roughness coefficient:', data.data_dict["bridge_ceiling_roughness"],
            vlayout
        )

        # Piers
        self._widgets['grp_piers'] = QGroupBox('Arc properties')
        vlayout.addWidget(self._widgets['grp_piers'])
        self._widgets['vlayout_piers'] = QVBoxLayout()
        vlayout_grp = self._widgets['vlayout_piers']
        self._widgets['grp_piers'].setLayout(vlayout_grp)
        # combo box for pier type
        self._widgets['txt_pier_type'] = QLabel("Pier type:")
        vlayout_grp.addWidget(self._widgets['txt_pier_type'])
        pier_types = ['-- None Selected --', 'Group', 'Wall']
        self._widgets['cbx_pier_type'] = QComboBox()
        self._widgets['cbx_pier_type'].addItems(pier_types)
        self._widgets['cbx_pier_type'].setCurrentIndex(0)
        if data.data_dict['pier_type'] in pier_types:
            self._widgets['cbx_pier_type'].setCurrentText(data.data_dict['pier_type'])
        vlayout_grp.addWidget(self._widgets['cbx_pier_type'])
        self._widgets['cbx_pier_type'].currentIndexChanged.connect(self._on_change_pier_type)
        # pier diameter
        self._widgets['txt_pier_diameter'] = QLabel("Pier diameter:")
        vlayout_grp.addWidget(self._widgets['txt_pier_diameter'])
        self._widgets['edt_pier_diameter'] = QLineEdit(f'{data.data_dict["pier_group_diameter"]}')
        pos_validator = QxDoubleValidator(parent=self._parent)
        pos_validator.setBottom(0.0)
        self._widgets['edt_pier_diameter'].setValidator(pos_validator)
        vlayout_grp.addWidget(self._widgets['edt_pier_diameter'])
        # wall width
        self._widgets['txt_wall_width'] = QLabel("Wall width:")
        vlayout_grp.addWidget(self._widgets['txt_wall_width'])
        self._widgets['edt_wall_width'] = QLineEdit(f'{data.data_dict["pier_wall_width"]}')
        pos_validator = QxDoubleValidator(parent=self._parent)
        pos_validator.setBottom(0.0)
        self._widgets['edt_wall_width'].setValidator(pos_validator)
        vlayout_grp.addWidget(self._widgets['edt_wall_width'])
        # number of piers in group
        self._widgets['txt_num_piers'] = QLabel("Number of piers in group:")
        vlayout_grp.addWidget(self._widgets['txt_num_piers'])
        self._widgets['edt_num_piers'] = QLineEdit(f'{int(data.data_dict["pier_group_num_in_group"])}')
        validator = QxIntValidator(parent=self._parent)
        validator.setBottom(1)
        self._widgets['edt_num_piers'].setValidator(validator)
        vlayout_grp.addWidget(self._widgets['edt_num_piers'])
        # pier group spacing
        self._widgets['txt_pier_spacing'] = QLabel("Pier group spacing (used if more than one pier in group):")
        vlayout_grp.addWidget(self._widgets['txt_pier_spacing'])
        self._widgets['edt_pier_spacing'] = QLineEdit(f'{data.data_dict["pier_group_spacing"]}')
        pos_validator = QxDoubleValidator(parent=self._parent)
        pos_validator.setBottom(0.0)
        self._widgets['edt_pier_spacing'].setValidator(pos_validator)
        vlayout_grp.addWidget(self._widgets['edt_pier_spacing'])

        # wall length
        self._widgets['txt_wall_length'] = QLabel("Wall pier length:")
        vlayout_grp.addWidget(self._widgets['txt_wall_length'])
        self._widgets['edt_wall_length'] = QLineEdit(f'{data.data_dict["pier_wall_length"]}')
        pos_validator = QxDoubleValidator(parent=self._parent)
        pos_validator.setBottom(0.0)
        self._widgets['edt_wall_length'].setValidator(pos_validator)
        vlayout_grp.addWidget(self._widgets['edt_wall_length'])
        # wall pier end type
        self._widgets['txt_pier_end_type'] = QLabel("Pier end type:")
        vlayout_grp.addWidget(self._widgets['txt_pier_end_type'])
        end_types = ['Square', 'Round', 'Sharp']
        self._widgets['cbx_pier_end_type'] = QComboBox()
        self._widgets['cbx_pier_end_type'].addItems(end_types)
        self._widgets['cbx_pier_end_type'].setCurrentIndex(0)
        if data.data_dict['pier_wall_end_type'] in end_types:
            self._widgets['cbx_pier_end_type'].setCurrentText(data.data_dict['pier_wall_end_type'])
        vlayout_grp.addWidget(self._widgets['cbx_pier_end_type'])
        # specify number of segments on the bridge center line
        self._widgets['tog_bridge_cl_segments'] = QCheckBox('Specify number of segments on bridge center line')
        checked = True if data.data_dict['specify_bridge_cl_num_segments'] else False
        self._widgets['tog_bridge_cl_segments'].setChecked(checked)
        vlayout_grp.addWidget(self._widgets['tog_bridge_cl_segments'])
        self._widgets['tog_bridge_cl_segments'].stateChanged.connect(self._on_specify_bridge_cl_num_segments)
        self._widgets['txt_bridge_cl_num_segments'] = QLabel("Number of bridge center line segments:")
        vlayout_grp.addWidget(self._widgets['txt_bridge_cl_num_segments'])
        cl_seg = int(data.data_dict['number_bridge_cl_segments'])
        self._widgets['edt_bridge_cl_num_segments'] = QLineEdit(f'{cl_seg}')
        validator = QxIntValidator(parent=self._parent)
        validator.setBottom(1)
        self._widgets['edt_bridge_cl_num_segments'].setValidator(validator)
        vlayout_grp.addWidget(self._widgets['edt_bridge_cl_num_segments'])
        if not checked:
            self._widgets['txt_bridge_cl_num_segments'].setVisible(False)
            self._widgets['edt_bridge_cl_num_segments'].setVisible(False)
        # toggle for number of segments
        self._widgets['tog_specify_segments'] = QCheckBox('Specify number of segments on bridge faces')
        checked = True if data.data_dict['specify_number_segments'] else False
        self._widgets['tog_specify_segments'].setChecked(checked)
        vlayout_grp.addWidget(self._widgets['tog_specify_segments'])
        self._widgets['tog_specify_segments'].stateChanged.connect(self._on_specify_num_segments)
        # number of segments
        self._widgets['txt_num_segments'] = QLabel("Number of segments:")
        vlayout_grp.addWidget(self._widgets['txt_num_segments'])
        self._widgets['edt_num_segments'] = QLineEdit(f'{int(data.data_dict["number_segments"])}')
        validator = QxIntValidator(parent=self._parent)
        validator.setBottom(1)
        self._widgets['edt_num_segments'].setValidator(validator)
        vlayout_grp.addWidget(self._widgets['edt_num_segments'])
        if not checked:
            self._widgets['txt_num_segments'].setVisible(False)
            self._widgets['edt_num_segments'].setVisible(False)
        # toggle for "has abutments"
        self._widgets['tog_abutments'] = QCheckBox('Has abutments')
        checked = True if data.data_dict['has_abutments'] else False
        self._widgets['tog_abutments'].setChecked(checked)
        vlayout_grp.addWidget(self._widgets['tog_abutments'])
        # element wrapping width
        self._widgets['txt_elem_wrap_width'] = QLabel("Element width around piers:")
        vlayout_grp.addWidget(self._widgets['txt_elem_wrap_width'])
        self._widgets['edt_elem_wrap_width'] = QLineEdit(f'{data.data_dict["pier_wrapping_width"]}')
        pos_validator = QxDoubleValidator(parent=self._parent)
        pos_validator.setBottom(0.0)
        self._widgets['edt_elem_wrap_width'].setValidator(pos_validator)
        vlayout_grp.addWidget(self._widgets['edt_elem_wrap_width'])
        # wall number of side elements
        self._widgets['txt_wall_num_side'] = QLabel("Wall pier number of side elements:")
        vlayout_grp.addWidget(self._widgets['txt_wall_num_side'])
        self._widgets['edt_wall_num_side'] = QLineEdit(f'{int(data.data_dict["pier_wall_num_side_elem"])}')
        validator = QxIntValidator(parent=self._parent)
        validator.setBottom(1)
        self._widgets['edt_wall_num_side'].setValidator(validator)
        vlayout_grp.addWidget(self._widgets['edt_wall_num_side'])
        self._on_change_pier_type()

        # 2d mesh generation
        self._widgets['grp_bridge_mesh'] = QGroupBox('Bridge 2D mesh generation')
        vlayout.addWidget(self._widgets['grp_bridge_mesh'])
        self._widgets['vlayout_bridge_mesh'] = QVBoxLayout()
        vlayout_grp = self._widgets['vlayout_bridge_mesh']
        self._widgets['grp_bridge_mesh'].setLayout(vlayout_grp)
        # bridge wrapping width
        self._widgets['txt_wrap_width'] = QLabel("Bridge wrapping width:")
        vlayout_grp.addWidget(self._widgets['txt_wrap_width'])
        self._widgets['edt_wrap_width'] = QLineEdit(f'{data.data_dict["bridge_wrapping_width"]}')
        pos_validator = QxDoubleValidator(parent=self._parent)
        pos_validator.setBottom(0.0)
        self._widgets['edt_wrap_width'].setValidator(pos_validator)
        vlayout_grp.addWidget(self._widgets['edt_wrap_width'])
        # toggle for upstream wrapping
        self._widgets['tog_wrap_upstream'] = QCheckBox('Wrap upstream')
        vlayout_grp.addWidget(self._widgets['tog_wrap_upstream'])
        if data.data_dict['bridge_wrap_upstream']:
            self._widgets['tog_wrap_upstream'].setChecked(True)
        # toggle for downstream wrapping
        self._widgets['tog_wrap_downstream'] = QCheckBox('Wrap downstream')
        vlayout_grp.addWidget(self._widgets['tog_wrap_downstream'])
        if data.data_dict['bridge_wrap_downstream']:
            self._widgets['tog_wrap_downstream'].setChecked(True)
        # number of elements through bridge width
        label = "Number of elements through bridge width (used when properties assigned to individual arcs):"
        self._widgets['txt_bridge_num_side_elem'] = QLabel(label)
        vlayout_grp.addWidget(self._widgets['txt_bridge_num_side_elem'])
        self._widgets['edt_bridge_num_side_elem'] = QLineEdit(f'{int(data.data_dict["bridge_num_side_elem"])}')
        validator = QxIntValidator(parent=self._parent)
        validator.setBottom(1)
        self._widgets['edt_bridge_num_side_elem'].setValidator(validator)
        vlayout_grp.addWidget(self._widgets['edt_bridge_num_side_elem'])
        # button elevation raster
        self._widgets['elev_raster_layout'] = QHBoxLayout()
        vlayout_grp.addLayout(self._widgets['elev_raster_layout'])
        self._widgets['btn_elev_raster'] = QPushButton('Select elevation raster...')
        self._widgets['btn_elev_raster'].clicked.connect(self._parent.on_btn_elev_raster)
        self._widgets['elev_raster_layout'].addWidget(self._widgets['btn_elev_raster'])
        self._widgets['txt_elev_raster'] = QLabel('Not selected')
        self._widgets['elev_raster_layout'].addWidget(self._widgets['txt_elev_raster'])
        self._widgets['elev_raster_layout'].addStretch()
        # preview mesh button
        self._widgets['preview_layout'] = QHBoxLayout()
        vlayout_grp.addLayout(self._widgets['preview_layout'])
        self._widgets['btn_preview_mesh'] = QPushButton('Preview bridge mesh...')
        self._widgets['btn_preview_mesh'].clicked.connect(self._parent.on_btn_preview_mesh)
        self._widgets['preview_layout'].addWidget(self._widgets['btn_preview_mesh'])
        # preview mesh in browser button
        self._widgets['btn_preview_mesh_browser'] = QPushButton('Preview bridge mesh (compatibility mode)...')
        self._widgets['btn_preview_mesh_browser'].clicked.connect(self._parent.on_btn_preview_mesh_browser)
        self._widgets['preview_layout'].addWidget(self._widgets['btn_preview_mesh_browser'])
        self._widgets['preview_layout'].addStretch()

        # data to add to SMS on OK
        self._widgets['grp_add_data'] = QGroupBox('Add data to SMS on OK')
        vlayout.addWidget(self._widgets['grp_add_data'])
        self._widgets['vlayout_add_data'] = QVBoxLayout()
        vlayout_grp = self._widgets['vlayout_add_data']
        self._widgets['grp_add_data'].setLayout(vlayout_grp)
        # toggle to add 3d ugrid bridge to SMS on OK
        self._widgets['tog_add_ugrid'] = QCheckBox('Add 3D Bridge UGrid')
        vlayout_grp.addWidget(self._widgets['tog_add_ugrid'])
        self._widgets['tog_add_ugrid'].stateChanged.connect(self._on_add_ugrid)
        # bridge pier base elevations
        self._widgets['txt_pier_base_elev'] = QLabel('Pier base elevations (ignored if no piers defined):')
        vlayout_grp.addWidget(self._widgets['txt_pier_base_elev'])
        self._widgets['edt_pier_base_elev'] = QLineEdit(f'{data.data_dict["bridge_pier_base_elev"]}')
        validator = QxDoubleValidator(parent=self._parent)
        self._widgets['edt_pier_base_elev'].setValidator(validator)
        vlayout_grp.addWidget(self._widgets['edt_pier_base_elev'])
        self._on_add_ugrid()
        # toggle to add coverage to SMS on OK
        self._widgets['tog_add_coverage'] = QCheckBox('Add "structure mesh" coverage')
        vlayout_grp.addWidget(self._widgets['tog_add_coverage'])
        # toggle to add footprint mesh to SMS on OK
        self._widgets['tog_add_footprint_ugrid'] = QCheckBox('Add "structure mesh" 2D mesh')
        vlayout_grp.addWidget(self._widgets['tog_add_footprint_ugrid'])

        # set this last because it triggers a signal and may change values in other widgets
        if data.data_dict['specify_arc_properties']:
            self._widgets['tog_assign_arcs'].setChecked(True)

        vlayout.addStretch()

    def on_assign_arcs(self):
        """Called when the assign properties to individual arcs toggle checked."""
        vis = self._widgets['tog_assign_arcs'].isChecked()
        self._widgets['grp_arc_props'].setVisible(vis)
        # self._widgets['tabs'].setTabEnabled(2, vis)
        if vis:
            self._widgets['cbx_pier_type'].setCurrentIndex(0)
            self._widgets['tog_bridge_cl_segments'].setChecked(False)
            self._widgets['tog_specify_segments'].setChecked(False)
            self._widgets['tog_abutments'].setChecked(False)
            self._widgets['grp_piers'].setVisible(False)
            self._widgets['txt_bridge_num_side_elem'].setVisible(True)
            self._widgets['edt_bridge_num_side_elem'].setVisible(True)
        else:
            self._widgets['grp_piers'].setVisible(True)
            self._widgets['txt_bridge_num_side_elem'].setVisible(False)
            self._widgets['edt_bridge_num_side_elem'].setVisible(False)

    def _on_add_ugrid(self):
        """Called when the add ugrid toggle checked."""
        self._widgets['txt_pier_base_elev'].setVisible(self._widgets['tog_add_ugrid'].isChecked())
        self._widgets['edt_pier_base_elev'].setVisible(self._widgets['tog_add_ugrid'].isChecked())

    def _on_change_pier_type(self):
        """Change to the pier type combobox."""
        wall_widgets = [
            'txt_wall_width', 'edt_wall_width', 'txt_elem_wrap_width', 'edt_elem_wrap_width', 'txt_wall_length',
            'edt_wall_length', 'txt_wall_num_side', 'edt_wall_num_side', 'txt_pier_end_type', 'cbx_pier_end_type'
        ]
        pier_widgets = [
            'txt_pier_diameter', 'edt_pier_diameter', 'txt_elem_wrap_width', 'edt_elem_wrap_width', 'txt_num_piers',
            'edt_num_piers', 'txt_pier_spacing', 'edt_pier_spacing'
        ]
        # hide all
        for w in pier_widgets:
            self._widgets[w].setVisible(False)
        for w in wall_widgets:
            self._widgets[w].setVisible(False)

        if self._widgets['cbx_pier_type'].currentIndex() == 1:
            for w in pier_widgets:
                self._widgets[w].setVisible(True)
        elif self._widgets['cbx_pier_type'].currentIndex() == 2:
            for w in wall_widgets:
                self._widgets[w].setVisible(True)

    def _on_specify_bridge_cl_num_segments(self):
        """Handle the checkbox for specify number of segments."""
        self._widgets['txt_bridge_cl_num_segments'].setVisible(self._widgets['tog_bridge_cl_segments'].isChecked())
        self._widgets['edt_bridge_cl_num_segments'].setVisible(self._widgets['tog_bridge_cl_segments'].isChecked())

    def _on_specify_num_segments(self):
        """Handle the checkbox for specify number of segments."""
        self._widgets['txt_num_segments'].setVisible(self._widgets['tog_specify_segments'].isChecked())
        self._widgets['edt_num_segments'].setVisible(self._widgets['tog_specify_segments'].isChecked())
