"""DMI Python definition for the 3D Bridge coverage component."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem
from xms.components.bases.coverage_component_base import CoverageComponentBase
from xms.components.display.display_options_io import read_display_options_from_json, write_display_options_to_json
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog

# 4. Local modules
from xms.bridge.dmi.xms_data import XmsData
from xms.bridge.gui.structure_dialog import run_structure_dlg
from xms.bridge.gui.summary_dialog import run_summary_dlg
from xms.bridge.structure_data import StructureData


def _duplicate_display_opts(new_path, disp_opts_fname):
    """Duplicates display options.

    Args:
        new_path (:obj:`str`): Path to the new save location.
        disp_opts_fname (:obj:`str`): The filename (no path) of the display options JSON file

    Returns:
        (:obj:`json_dict`): dict containing the display options

    """
    fname = os.path.join(new_path, disp_opts_fname)
    json_dict = read_display_options_from_json(fname)
    if 'uuid' in json_dict:
        json_dict['uuid'] = str(uuid.uuid4())
        json_dict['comp_uuid'] = os.path.basename(new_path)
        categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
        categories.from_dict(json_dict)
        write_display_options_to_json(fname, categories)
    return json_dict


class StructureComponent(CoverageComponentBase):
    """3D Structure component."""
    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file (:obj:`str`): The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = StructureData(self.main_file)
        self.uuid = os.path.basename(os.path.dirname(self.main_file))
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.disp_opts_file = os.path.join(os.path.dirname(self.main_file), 'structure_display_options.json')
        if not os.path.isfile(self.disp_opts_file):
            categories = CategoryDisplayOptionList()
            default_file = os.path.join(
                os.path.dirname(__file__), 'gui', 'rc', 'default_structure_display_options.json'
            )
            json_dict = read_display_options_from_json(default_file)
            json_dict['comp_uuid'] = os.path.basename(os.path.dirname(self.main_file))
            categories.from_dict(json_dict)
            write_display_options_to_json(self.disp_opts_file, categories)
            # Save our display list UUID to the main file
            self.data.info.attrs['display_uuid'] = categories.uuid
            self.data.commit()

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.

            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path if
                necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        new_main_file = os.path.join(new_path, os.path.basename(self.main_file))
        if save_type == 'DUPLICATE':
            json_dict = _duplicate_display_opts(new_path, os.path.basename(self.disp_opts_file))
            data = StructureData(new_main_file)
            data.info.attrs['cov_uuid'] = ''
            data.info.attrs['display_uuid'] = json_dict['uuid']
            data.commit()
        return new_main_file, [], []

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        if not os.path.isfile(self.main_file):
            self.data.commit()
        action = ActionRequest(
            main_file=self.main_file,
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='get_initial_display_options'
        )
        return [], [action]

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        self.ensure_cov_uuid(query)
        self.add_display_message()
        return [], []

    def add_display_message(self):
        """Add the display message to the list."""
        self.display_option_list.append(XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid))

    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (:obj:`list[str]`): A list of the main files of the selected components of this type.

        Returns:
            menu_items (:obj:`list[xms.api.dmi.MenuItem]`):
                A list of menus and menu items to be shown. Note that this list can have objects of type
                xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be added to the list to indicate a
                separator.
        """
        if len(main_file_list) > 1 or not main_file_list:
            return []  # Multi-select or nothing selected

        menu_list = [None]  # None == spacer
        # Add the Open menu command that will display the well dialog when clicked.
        prop_action = ActionRequest(
            modality='MODAL',
            module_name='xms.bridge.structure_component',
            class_name=self.class_name,
            method_name='edit_structure_dialog',
            main_file=self.main_file
        )
        prop_item = MenuItem(text='Edit Structure...', action=prop_action)
        menu_list.append(prop_item)
        prop_action = ActionRequest(
            modality='MODAL',
            module_name='xms.bridge.structure_component',
            class_name=self.class_name,
            method_name='summary_dialog',
            main_file=self.main_file
        )
        prop_item = MenuItem(text='3D Structure Summary...', action=prop_action)
        menu_list.append(prop_item)
        prop_action = ActionRequest(
            modality='MODAL',
            module_name='xms.bridge.structure_component',
            class_name=self.class_name,
            method_name='display_options',
            main_file=self.main_file
        )
        prop_item = MenuItem(text='Display Options...', action=prop_action)
        menu_list.append(prop_item)
        menu_list.append(None)  # Adds a separator
        return menu_list

    def ensure_cov_uuid(self, query):
        """Ensure the cov uuid is set.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
        """
        if self.cov_uuid != '':
            return [], []
        self.cov_uuid = query.parent_item_uuid()
        if not self.cov_uuid:
            return [('ERROR', 'Could not get 3D Structure coverage UUID.')], []
        self.data.info.attrs['cov_uuid'] = self.cov_uuid
        self.data.commit()
        return [], []

    def edit_structure_dialog(self, query, params, win_cont):
        """Opens the bridge dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`list[str]`): Unused
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`list[tuple(str)]`), (:obj:`list[xms.api.dmi.ActionRequest]`):
            messages (:obj:`list[tuple(str)]`): A list of tuples with the first element of the tuple
            being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.

            action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): A list of actions for XMS to perform.

        """
        self.ensure_cov_uuid(query)
        xms_data = XmsData(query, self)
        run_structure_dlg(win_cont, xms_data)
        return [], []

    def summary_dialog(self, query, params, win_cont):
        """Opens the bridge dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`list[str]`): Unused
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`list[tuple(str)]`), (:obj:`list[xms.api.dmi.ActionRequest]`):
            messages (:obj:`list[tuple(str)]`): A list of tuples with the first element of the tuple being the message
            level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.

            action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): A list of actions for XMS to perform.

        """
        xms_data = XmsData(query, self)
        run_summary_dlg(win_cont, xms_data)
        return [], []

    def display_options(self, query, params, win_cont):
        """Opens the bridge dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`list[str]`): Unused
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`list[tuple(str)]`), (:obj:`list[xms.api.dmi.ActionRequest]`):
            messages (:obj:`list[tuple(str)]`): A list of tuples with the first element of the tuple being the message
            level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.

            action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): A list of actions for XMS to perform.

        """
        xms_data = XmsData(query, self)
        categories_list = [xms_data.display_categories()]

        dlg = self._get_display_opts_dlg(categories_list, win_cont)
        if dlg.exec():
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                write_display_options_to_json(self.disp_opts_file, category_list)
            self.add_display_message()
        return [], []

    def _get_display_opts_dlg(self, categories_list, win_cont):
        """Get the display options dialog."""
        return CategoryDisplayOptionsDialog(categories_list, win_cont)
