"""MaterialData class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from importlib.metadata import version

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem

# 4. Local modules


class StructureData(XarrayBase):
    """Class for storing the SRH Material properties (Manning's N and sediment)."""
    def __init__(self, filename):
        """Constructor.

        Args:
            filename (:obj:`str`): file name
        """
        super().__init__(filename)
        self.info.attrs['FILE_TYPE'] = '3D_STRUCTURE_FILE'
        if 'cov_uuid' not in self.info.attrs:
            self.info.attrs['cov_uuid'] = ''  # gets set later
        self.data_dict = self._default_data()
        for k, _ in self.data_dict.items():
            if k in self.info.attrs:
                self.data_dict[k] = self.info.attrs[k]
        self.curve_names = ['top_profile', 'upstream_profile', 'downstream_profile']
        self.curves = {}
        for crv in self.curve_names:
            self.curves[crv] = pd.DataFrame({'Distance': [0.0], 'Elevation': [0.0]})
            self.curves[crv].index += 1
        for crv in self.curve_names:
            ds = self.get_dataset(crv, False)
            if ds is not None:
                self.curves[crv] = ds.to_dataframe()
        self.close()

    def _default_data(self):
        """Creates a default material data set.

        Returns:
            (:obj:`xarray.Dataset`): The material list dataset
        """
        default_data = {
            'structure_type': 'Bridge',  # Bridge, Culvert
            'bridge_ceiling_roughness': 0.05,
            'bridge_specify_downstream_profile': 0,
            'bridge_width': 0.0,
            'bridge_pier_base_elev': 0.0,
            'bridge_wrapping_width': 0.0,
            'bridge_wrap_upstream': 1,
            'bridge_wrap_downstream': 1,
            'bridge_num_side_elem': 1,
            'specify_bridge_cl_num_segments': 0,
            'number_bridge_cl_segments': 1,
            'specify_number_segments': 0,
            'number_segments': 1,
            'has_abutments': 0,
            'pier_type': '-- None Selected --',  # None, Group, Wall
            'pier_wrapping_width': 0.0,
            'pier_group_diameter': 0.0,
            'pier_group_num_in_group': 1,
            'pier_group_spacing': 0.0,
            'pier_wall_width': 0.0,
            'pier_wall_length': 0.0,
            'pier_wall_num_side_elem': 1,
            'pier_wall_end_type': 'Square',  # Square, Round, Sharp
            'srh_overtopping': 0,
            'srh_weir_length': 0.0,
            'srh_weir_type': 'Paved',  # Paved, Gravel, Single, Double, Sharp, Broad, User
            'srh_weir_cw': 0.0,
            'srh_weir_a': 0.0,
            'srh_weir_b': 0.0,
            'culvert_type': 'Circular',  # Circular, Box
            'culvert_diameter': 0.0,
            'culvert_rise': 0.0,
            'culvert_span': 0.0,
            'culvert_up_invert': 0.0,
            'culvert_dn_invert': 0.0,
            'culvert_embed_depth': 0.0,
            'culvert_num_barrels': 1,
            'culvert_wall_width': 0.0,
            'culvert_num_seg_barrel': 5,
            'srh_mapping_info': '',
            'elev_raster': '',
            'specify_arc_properties': 0,
            'arc_properties': '',
            'specify_culvert_arc_properties': 0,
            'culvert_arc_properties': '',
            'summary_profile_coverage_uuid': '',
        }
        return default_data

    def commit(self):
        """Save in memory datasets to the NetCDF file."""
        self.info.attrs['VERSION'] = version('xmsbridge')
        filesystem.removefile(self._filename)
        for k, v in self.data_dict.items():
            self.info.attrs[k] = v
        super().commit()
        for crv in self.curve_names:
            self.curves[crv].to_xarray().to_netcdf(self._filename, group=crv, mode='a')

    def close(self):
        """Closes the H5 file and does not write any data that is in memory."""
        super().close()
