"""This module imports cmcards files into SMS."""

# 1. Standard Python modules
import datetime
import math
import os
import shlex
import sys
import uuid

# 2. Third party modules
import h5py
from harmonica.tidal_constituents import Constituents
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.coverage_component_builder import CoverageComponentBuilder
from xms.components.display.display_options_io import write_display_option_ids
from xms.components.dmi.xms_data import XmsData
from xms.constraint import read_grid_from_file
from xms.data_objects.parameters import Arc, Coverage, Point, Projection, UGrid
from xms.datasets.dataset_reader import DatasetReader
from xms.datasets.dataset_writer import DatasetWriter
from xms.gdal.utilities.gdal_utils import (
    GdalResult, wkt_from_geographic_horizontal_datum, wkt_from_stateplane_horizontal_datum,
    wkt_from_utm_horizontal_datum
)
from xms.gmi.data.coverage_data import CoverageData
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData
from xms.guipy.data.target_type import TargetType
from xms.tides.components.tidal_component import TidalComponent
from xms.tides.data.tidal_data import ADCIRC_INDEX, FES2014_INDEX, LEPROVOST_INDEX, TPX08_INDEX, TPX09_INDEX

# 4. Local modules
from xms.cmsflow._cmsflow_gridh5_import import CMSFlowH5GridImporter
from xms.cmsflow._cmsflow_tel_import import CMSFlowTelImporter
from xms.cmsflow.components.bc_component import BCComponent
from xms.cmsflow.components.id_files import (
    BC_INITIAL_ATT_ID_FILE, BC_INITIAL_COMP_ID_FILE, SAVE_INITIAL_ATT_ID_FILE, SAVE_INITIAL_COMP_ID_FILE
)
from xms.cmsflow.components.save_points_component import SavePointsComponent
from xms.cmsflow.components.sim_component import SimComponent
from xms.cmsflow.components.structures_component import StructuresComponent
from xms.cmsflow.data.bc_data import SUPPORTED_CONSTITUENTS
from xms.cmsflow.data.model import get_model, parameter_from_file
from xms.cmsflow.data.rm_structures_data import PARAMETER_TYPE_CONSTANT_IDX
from xms.cmsflow.data.save_points_data import SavePointsData
from xms.cmsflow.feedback.xmlog import XmLog
from xms.cmsflow.file_io import io_util
from xms.cmsflow.file_io.parent_adcirc_reader import ParentAdcircReader

BC_UNASSIGNED = 0
BC_FLUX = 1
BC_WSE = 2

# These are the "CSType" values from old .mp files
BC_LEGACY_UNASSIGNED = 0
BC_LEGACY_TIDAL = 1
BC_LEGACY_CLOSED = 4
BC_LEGACY_WSE_DEF = 5
BC_LEGACY_FLOW = 6
BC_LEGACY_WSE_EXT = 7
BC_LEGACY_WSE_VEL_EXT = 8
BC_LEGACY_LAND = 9
BC_LEGACY_CROSS_SHORE = 10
BC_LEGACY_HARMONIC = 11

# This is a list of cards that are not handled in the traditional sense, but do not need to be added to the
#   Advanced Card list. This could be for a few reasons. Those I have discovered include:
#   - The data represented by the cards is read from another file
#       GRID_ORIGIN_* - is read from the tel file
#       *_CELLSTRING  - the boundary conditions are read from the mp.h5 file
#   - The card is not used by CMS
#       2D_MODE       - this is left over from the original version that included a 3D version
#       SMS_VERSION   - this is only included for SMS and commentary
#       OFFSET        - This is only valid in a block so it is meaningless
UNHANDLED_CARDS = {
    '2D_MODE', 'SMS_VERSION', 'FLOW_RATE_INCREMENT', 'FLUX_UNITS', 'END_PARAMETERS', 'ETA_TIME_SERIES_OUTPUT',
    'GLBL_CONC_CAPAC_OUTPUT', 'GLBL_CONCENTRATION_OUTPUT', 'GLOBAL_MORPHOLOGY_OUTPUT', 'GLOBAL_TRANS_RATE_OUTPUT',
    'GLOBAL_VELOCITY_OUTPUT', 'GLOBAL_WATER_LEVEL_OUTPUT', 'GRID_MODIFICATION_NUMBER', 'GRID_ORIGIN_X', 'GRID_ORIGIN_Y',
    'HDRIVER_CELLSTRING', 'HORIZONTAL_PROJECTION_END', 'PARAMS_FILE', 'Q_TRANS_RATE_INCREMENT', 'TIME_SERIES_INCREMENT',
    'U_TIME_SERIES_OUTPUT', 'USE_ADVECTION_TERMS', 'USE_MIXING_TERMS', 'V_TIME_SERIES_OUTPUT',
    'VERTICAL_PROJECTION_END', 'X_FLOW_RATE_OUTPUT', 'Y_FLOW_RATE_OUTPUT', 'QDRIVER_CELLSTRING',
    "MULTI_VDRIVER_CELLSTRING", "MULTI_HDRIVER_CELLSTRING", "OFFSET"
}


def card_to_time_unit(card):
    """Convert a CMS-Flow time unit card to the appropriate DMI keyword.

    Args:
         card (str): Time unit card from the .cmcards file.

    Returns:
        (str): The DMI keyword corresponding to the CMS-Flow card or empty string if unrecognized

    """
    card = clean_token(card).lower()
    if card == "sec":
        return 'seconds'
    elif card == "min":
        return 'minutes'
    elif card == "hrs":
        return 'hours'
    elif card == "days":
        return 'days'
    elif card == "weeks":
        return 'weeks'
    else:
        return ''


def card_to_projection_unit(card):
    """Convert a CMS-Flow projection unit card to the appropriate DMI keyword.

    If the unit is not explicitly recognized by the API, the original card text is returned.

    Args:
         card (str): Velocity unit card from the .cmcards file.

    Returns:
        (str): The DMI keyword corresponding to the CMS-Flow card or empty string if unrecognized

    """
    card = clean_token(card).upper()
    if card == "DEGREES":
        return 'ARC_DEGREES'
    elif card == "INTL_FEET":
        return 'FEET (INTERNATIONAL)'
    elif card == "FEET":
        return 'FEET (U.S. SURVEY)'
    # Meters card matches API keyword
    else:
        return card


def safe_shlex_split(line, err_msg, min_tokens=2):
    """Carefully shlex split a line, logging an error if split fails.

    Args:
        line (str): The line to split.
        err_msg (str): Message to log if exception thrown
        min_tokens (Optional[int]): The minimum number of tokens that should be parsed from the line. If the number
            of tokens parsed is less than this value, an error is logged.

    Returns:
        list[str]: The line split using shlex, empty list on error
    """
    try:
        tokens = shlex.split(line, posix='win' not in sys.platform)
        if len(tokens) < min_tokens:
            raise RuntimeError()  # shlex split did not fail, but we don't have enough values. Log an error.
    except Exception:
        if err_msg:
            XmLog().instance.error(err_msg)
        return []
    return tokens


def clean_token(token):
    """Stips single qoutes, double quotes, and whitespace from a token in that order.

    Args:
        token (str): The parsed token to clean

    Returns:
        str: The cleaned token
    """
    return token.strip("'").strip('"').strip()


def is_comment(token):
    """Check if a string is a comment (begins with '!' or '*').

    Args:
        token (str): The string token to clean

    Returns:
        bool: True if the string is a comment
    """
    if token.startswith('!') or token.startswith('*') or token.startswith('#'):
        return True
    return False


class CmCardsImporter:
    """.cmcards file importer for the CMS-Flow DMI interface."""
    def __init__(self):
        """Constructor for the import class."""
        self.xms_data = None
        self.filename = ''  # The file being read. Should be a .cmcards file.
        self.lineno = 0  # The current line of the file being read.
        self.save_points_cov_values = {}
        self.save_points_cov_name = ''  # Name to assign the save points coverage.
        self.dependent_widgets = {}  # Seems to be info that goes into the simulation component.
        self.reading_met_station = False  # Whether we're currently in a meteorological station block
        self.reading_sediment_size = False  # Whether we're in a sediment size block
        self.reading_bed_layer = False  # Whether we're in a bed lay block
        self.reading_horizontal_proj = True  # Flag to tell us if we are reading horizontal or vertical projection data
        self.reading_global_constituents = False  # Whether we're reading global tidal constituents.
        self.reading_bc = False  # Whether we're reading a boundary condition block.
        self.reading_rubble_mound = False  # Whether we're reading a rubble mound block.
        self.reading_tidal = False  # Whether we're reading a local tidal constituent block.
        self.reading_harmonic = False  # Whether we're reading a tidal harmonic block
        self.reading_weir = False  # Whether we're reading a weir struct block
        self.reading_culvert = False
        self.reading_tide_gate = False
        self.reading_tidal_db = False
        self.tidal_db_added = False
        self.reading_dredge_operation = False
        self.reading_dredge = False
        self.reading_placement = False
        self.placement_number = 0
        self.tidal_db_idx = -1
        self.tidal_db_type = ''
        self.tidal_db_cons = []
        self.curves = {}
        self.bc_curves = {}
        self.datasets = {}
        self.telescoping_filename = ''
        self.grid_filename = ''
        self.bc_filename = ''
        self.current_orig_line = ''
        self.grid_projection = Projection()  # Pass this to the .tel file reader
        self._horizontal_datum = 'LOCAL'
        self.grid_angle = None
        self.bathymetry = None  # [filename, group_path]
        self.cell_activity_group = ''  # Old-style H5 grids
        self.bc_mainfile = ''
        self.tidal_mainfile = ''
        self.cms_version = []  # Major then minor version as ints
        self.refinement_idx = []
        self.bc_cov = None

        self.current_structure = get_model()

        self.weirs: list[list[int]] = []
        self.weir_values: list[str] = []

        self.culverts: list[tuple[int, int]] = []
        self.culvert_values: list[str] = []

        self.tide_gates: list[list[int]] = []
        self.tide_gate_values: list[str] = []

        self.sim_data = None
        self.comp_dir = os.path.join(XmEnv.xms_environ_temp_directory(), 'Components')
        self.met_station_values = {'name': [], 'x': [], 'y': [], 'height': [], 'direction': []}
        self.sediment_size_values = {
            'diameter_value': [],
            'diameter_units': [],
            'fall_velocity_method': [],
            'fall_velocity_value': [],
            'fall_velocity_units': [],
            'corey_shape_factor': [],
            'critical_shear_method': [],
            'critical_shear_stress': []
        }
        self.bed_layer_values = {
            'layer_id': [],
            'layer_thickness_type': [],
            'layer_thickness': [],
            'layer_thickness_const': [],
            'd05': [],
            'd10': [],
            'd16': [],
            'd20': [],
            'd30': [],
            'd35': [],
            'd50': [],
            'd65': [],
            'd84': [],
            'd90': [],
            'd95': []
        }
        self.mixing_layer_thickness_type = ''
        self.mixing_layer_thickness_const = 0.0

        # Data for building parent ADCIRC mesh and solutions
        self.parent_adcirc_wse = {
            'arc_indices': [],
            'locs': [],
            'times': None,
            'wse': [],
        }
        self.parent_adcirc_vel = {  # Always a WSE solution, but velocity is optional.
            'arc_indices': [],
            'locs': [],
            'times': None,
            'wse': [],
            'velx': [],
            'vely': [],
        }

        self.bc_arc_values = {
            'name': np.array([], dtype=object),
            'bc_type': np.array([], dtype=object),
            'flow_source': np.array([], dtype=object),
            'constant_flow': np.array([], dtype=float),
            'flow_curve': np.array([], dtype=int),
            'specify_inflow_direction': np.array([], dtype=int),
            'flow_direction': np.array([], dtype=float),
            'flow_conveyance': np.array([], dtype=float),
            'wse_source': np.array([], dtype=object),
            'wse_const': np.array([], dtype=float),
            'wse_forcing_curve': np.array([], dtype=int),
            'use_velocity': np.array([], dtype=int),
            'parent_cmsflow': np.array([], dtype=object),
            'parent_adcirc_14_uuid': np.array([], dtype=object),
            'parent_adcirc_14': np.array([], dtype=object),
            'parent_adcirc_solution_type': np.array([], dtype=object),
            'parent_adcirc_63': np.array([], dtype=object),
            'parent_adcirc_64': np.array([], dtype=object),
            'parent_adcirc_solution': np.array([], dtype=object),
            'parent_adcirc_solution_wse': np.array([], dtype=object),
            'parent_adcirc_wse_path': np.array([], dtype=object),
            'parent_adcirc_velocity_path': np.array([], dtype=object),
            'parent_adcirc_start_time': np.array([], dtype=object),
            'wse_offset_type': np.array([], dtype=object),
            'wse_offset_const': np.array([], dtype=float),
            'wse_offset_curve': np.array([], dtype=int),
            'use_salinity_curve': np.array([], dtype=int),
            'salinity_curve': np.array([], dtype=int),
            'use_temperature_curve': np.array([], dtype=int),
            'temperature_curve': np.array([], dtype=int),
            'harmonic_table': np.array([], dtype=int),
            'tidal_table': np.array([], dtype=int)
        }
        self.table_id = 0
        self.bc_arc_comp_ids = []
        self.bc_arc_types = []
        self.in_bc_flux = False
        self.in_flux_block = False
        self.in_wse_block = False
        self.quad_tree = None
        self.quad_ugrid = None
        self.rect_cells = None
        self.activity_cov = None
        self.save_points_cov = None  # testing output
        self.save_points_values = {
            'point': [],
            'name': np.array([], dtype=object),
            'hydro': np.array([], dtype=np.int32),
            'sediment': np.array([], dtype=np.int32),
            'salinity': np.array([], dtype=np.int32),
            'waves': np.array([], dtype=np.int32),
        }
        self.rubble_mound_cov = None  # testing output
        self.rubble_mound_cov_name = ''
        self.rubble_mound_rock_diameter_dsets = {}  # {(filename, h5_path): uuid}
        self.rubble_mound_porosity_dsets = {}  # {(filename, h5_path): uuid}
        self.rubble_mound_base_depth_dsets = {}  # {(filename, h5_path): uuid}
        self.rubble_mound_values = {
            'name': np.array([], dtype=object),
            'rock_diameter': np.array([], dtype=float),
            'rock_diameter_type': np.array([], dtype=int),
            'ROCK_DIAMETER_DATASET': np.array([], dtype=object),
            'porosity': np.array([], dtype=float),
            'porosity_type': np.array([], dtype=int),
            'STRUCTURE_POROSITY_DATASET': np.array([], dtype=object),
            'base_depth': np.array([], dtype=float),
            'base_depth_type': np.array([], dtype=int),
            'STRUCTURE_BASE_DEPTH_DATASET': np.array([], dtype=object),
            'calculation_method': np.array([], dtype=object),
        }
        self.rubble_mound_cells = np.empty(0, dtype=int)
        self.current_rubble_mound_id = 1
        self.harmonic_values = {}
        self.tidal_values = {}
        self.global_constituents = {'constituent': [], 'amplitude': [], 'phase': []}
        self.advanced_card_blocks = []
        self.advanced_card_cards = []
        self.advanced_card_values = []
        self.sim_comp_dir = ''
        self.build_data = {
            'sim': None,
            'sim_comp': None,
            'links': [],
            'tidal_sims': [],
            'bc_cov': None,
            'bc_cov_comp': None,
            'ugrids': [],
            'datasets': [],
            'activity_cov': None,
            'save_pts_cov': None,
            'save_pts_cov_comp': None,
            'structure_coverage': None,
            'structure_component': None,
            'structure_keywords': None,
        }
        self._test_file_count = 0

    def _build_grid_wkt(self):
        """Build a WKT for the grid projection if a vertical datum was specified.

        The data_objects::parameters::Projection does not have a member variable for vertical datum and XMS does not
        directly store it. Has to be parsed from the WKT on export, and we need to use GDAL on import to generate a
        WKT if the vertical datum was specified.
        """
        if not self._horizontal_datum:
            return

        coordsys = self.grid_projection.coordinate_system
        if coordsys not in ['GEOGRAPHIC', 'UTM', 'STATEPLANE']:
            # Don't need to mess with local projections, I don't think.
            return

        if self._horizontal_datum not in ['NAD83', 'NAD27']:
            msg = (f'Unsupported horizontal datum specified: {self._horizontal_datum}. Defaulting to NAD83.')
            XmLog().instance.warning(msg)
            self._horizontal_datum = 'NAD83'

        if coordsys == 'GEOGRAPHIC' and self.grid_projection.horizontal_units not in ['', 'ARC_DEGREES']:
            XmLog().instance.warning(
                'Invalid units specified for geographic projection system: '
                f'{self.grid_projection.horizontal_units}. Defaulting to degrees.'
            )
            self.grid_projection.horizontal_units = 'ARC_DEGREES'
        elif coordsys == 'UTM' and self.grid_projection.horizontal_units not in ['', 'METERS']:
            XmLog().instance.warning(
                f'Invalid units specified for UTM projection system: {self.grid_projection.horizontal_units}.'
                'Defaulting to meters.'
            )
            self.grid_projection.horizontal_units = 'METERS'

        if coordsys == 'GEOGRAPHIC':
            result, wkt = wkt_from_geographic_horizontal_datum(self._horizontal_datum)
        elif coordsys == 'UTM':
            result, wkt = wkt_from_utm_horizontal_datum(self._horizontal_datum, self.grid_projection.coordinate_zone)
        elif coordsys == 'STATEPLANE':
            result, wkt = wkt_from_stateplane_horizontal_datum(
                self._horizontal_datum, self.grid_projection.horizontal_units, self.grid_projection.coordinate_zone
            )
        else:
            raise AssertionError('Unsupported coordsys')

        if result == GdalResult.success:
            return
        elif result == GdalResult.bad_horizontal_units:
            XmLog().instance.error("Invalid horizontal units. The projection may be wrong.")
        elif result == GdalResult.bad_horizontal_datum:
            XmLog().instance.error("Invalid horizontal datum. The projection may be wrong.")
        elif result == GdalResult.bad_zone:
            XmLog().instance.error("Invalid coordinate zone. The projection may be wrong.")
        else:
            XmLog().instance.error('An error occurred parsing the projection. It may be wrong.')

    def _add_h5_dependent_tables(self):
        """Adds the tables that have curves and datasets in them."""
        # add meteorological station data
        if self.met_station_values['direction']:
            meteorological_stations = {'NEXT_CURVE_ID': np.int32(max(self.met_station_values['direction']) + 1)}
            for key, item in self.met_station_values.items():
                self.met_station_values[key] = xr.DataArray(item)
            self.sim_data.meteorological_stations_table = xr.Dataset(
                attrs=meteorological_stations, data_vars=self.met_station_values
            )

        if self.sim_data.sediment.attrs['ENABLE_SIMPLIFIED_MULTI_GRAIN_SIZE'] == 0:
            # add bed layers
            if self.bed_layer_values['layer_id']:
                # Find layer id one and set the mixing values
                have_mixing = False
                mixing_idx = 0
                if 0 in self.bed_layer_values['layer_id']:
                    try:
                        # Try to get the first unassigned layer id.
                        mixing_idx = self.bed_layer_values['layer_id'].index(0)
                        self.bed_layer_values['layer_id'] = [layer + 1 for layer in self.bed_layer_values['layer_id']]
                        have_mixing = True
                    except ValueError:
                        pass
                else:
                    try:
                        mixing_idx = self.bed_layer_values['layer_id'].index(1)
                        have_mixing = True
                    except ValueError:
                        pass
                # There should always be a layer id 1, which is the mixing layer.
                # However, some old files don't have this.
                if have_mixing:
                    self.bed_layer_values['layer_thickness_type'][mixing_idx] = self.mixing_layer_thickness_type
                    self.bed_layer_values['layer_thickness_const'][mixing_idx] = self.mixing_layer_thickness_const
                    if len(self.bed_layer_values['layer_thickness_type']) == 1:
                        self.bed_layer_values['layer_thickness_type'][mixing_idx] = 'Automatic'

                layer_id = xr.DataArray(data=np.array(self.bed_layer_values['layer_id'], dtype=int))
                data = np.array(self.bed_layer_values['layer_thickness_type'], dtype=object)
                layer_thickness_type = xr.DataArray(data=data)
                layer_thickness = xr.DataArray(data=np.array(self.bed_layer_values['layer_thickness'], dtype=object))
                data = np.array(self.bed_layer_values['layer_thickness_const'], dtype=float)
                layer_thickness_const = xr.DataArray(data=data)
                d05 = xr.DataArray(data=np.array(self.bed_layer_values['d05'], dtype=object))
                d10 = xr.DataArray(data=np.array(self.bed_layer_values['d10'], dtype=object))
                d16 = xr.DataArray(data=np.array(self.bed_layer_values['d16'], dtype=object))
                d20 = xr.DataArray(data=np.array(self.bed_layer_values['d20'], dtype=object))
                d30 = xr.DataArray(data=np.array(self.bed_layer_values['d30'], dtype=object))
                d35 = xr.DataArray(data=np.array(self.bed_layer_values['d35'], dtype=object))
                d50 = xr.DataArray(data=np.array(self.bed_layer_values['d50'], dtype=object))
                d65 = xr.DataArray(data=np.array(self.bed_layer_values['d65'], dtype=object))
                d84 = xr.DataArray(data=np.array(self.bed_layer_values['d84'], dtype=object))
                d90 = xr.DataArray(data=np.array(self.bed_layer_values['d90'], dtype=object))
                d95 = xr.DataArray(data=np.array(self.bed_layer_values['d95'], dtype=object))

                bed_layer_table = {
                    'layer_id': layer_id,
                    'layer_thickness_type': layer_thickness_type,
                    'layer_thickness': layer_thickness,
                    'layer_thickness_const': layer_thickness_const,
                    'd05': d05,
                    'd10': d10,
                    'd16': d16,
                    'd20': d20,
                    'd30': d30,
                    'd35': d35,
                    'd50': d50,
                    'd65': d65,
                    'd84': d84,
                    'd90': d90,
                    'd95': d95,
                }
                self.sim_data.bed_layer_table = xr.Dataset(data_vars=bed_layer_table)
        if len(self.bed_layer_values['d50']) == 1:
            if self.sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] == 'D50 Sigma':
                self.sim_data.sediment.attrs['MULTI_D50'] = self.bed_layer_values['d50'][0]
            elif self.sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] == 'D16 D50 D84':
                self.sim_data.sediment.attrs['MULTI_D16'] = self.bed_layer_values['d16'][0]
                self.sim_data.sediment.attrs['MULTI_D50'] = self.bed_layer_values['d50'][0]
                self.sim_data.sediment.attrs['MULTI_D84'] = self.bed_layer_values['d84'][0]
            elif self.sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] == 'D35 D50 D90':
                self.sim_data.sediment.attrs['MULTI_D35'] = self.bed_layer_values['d35'][0]
                self.sim_data.sediment.attrs['MULTI_D50'] = self.bed_layer_values['d50'][0]
                self.sim_data.sediment.attrs['MULTI_D90'] = self.bed_layer_values['d90'][0]

    @staticmethod
    def _write_renumber_file(renumber_file, tel_reader):
        """Writes the renumber file for the simulation component to use later.

        Args:
            renumber_file (str): The path and name of the renumber file.
            tel_reader (:obj:`CMSFlowTelImporter`): The tel file reader.
        """
        with open(renumber_file, 'w') as renumber:
            # Write out the new index, in old tel file order.
            for renum in tel_reader.refinement_idx:
                renumber.write(f'{renum[tel_reader.new_cell_idx_column]}\n')

    def _update_input_datasets(self, geom_uuid, renumber_index):
        """Update the input datasets so they will read correctly in SMS.

        Args:
            geom_uuid (str): UUID of the quadtree
            renumber_index (list[int]): The refinement renumber indices
        """
        # Link up datasets now that we have the Quadtree's UUID
        for (filename, group_path), storages in self.datasets.items():
            reader = DatasetReader(h5_filename=filename, group_path=group_path)
            # Change the order of the values in the old XMDF file to match the newer numbering.
            values = reader.values[:]
            if self.refinement_idx:
                values = np.array([timestep[renumber_index] for timestep in values])
            # Write the dataset to a new file in temp.
            dset_uuid = str(uuid.uuid4())
            # We use comp_dir so tests can control where we put temp files.
            new_file_name = os.path.normpath(os.path.join(self.comp_dir, '..', dset_uuid + '.h5'))
            writer = DatasetWriter(
                h5_filename=new_file_name,
                name=reader.name,
                dset_uuid=dset_uuid,
                geom_uuid=geom_uuid,
                num_components=reader.num_components,
                ref_time=reader.ref_time,
                null_value=-999.0,
                time_units=reader.time_units,
                location='cells'
            )
            writer.write_xmdf_dataset(times=reader.times[:], data=values)
            self.build_data['datasets'].append(writer)

            # Update the references in the model control and rubble mound polygon coverage.
            for container, key, row in storages:
                if row == -1:
                    container[key] = dset_uuid
                else:
                    container[key][row] = dset_uuid

    def _add_tidal_component(self):
        """Add the tidal component that came from a tidal DB card block."""
        # Check to see if a tidal database has already been added.
        if self.tidal_db_added:
            return

        # Create a folder and UUID for the new tidal component.
        comp_uuid = str(uuid.uuid4())
        tidal_comp_dir = os.path.join(self.comp_dir, comp_uuid)
        os.makedirs(tidal_comp_dir, exist_ok=True)
        tidal_main_file = os.path.join(tidal_comp_dir, 'tidal_comp.nc')
        self.tidal_mainfile = tidal_main_file

        # Fill component data from widget values in the old database.
        tidal_py_comp = TidalComponent(tidal_main_file)  # Initialize some default data
        data = tidal_py_comp.data
        data._info.attrs['source'] = self.tidal_db_idx
        tidal_model = Constituents(self.tidal_db_type)
        all_cons = data.get_default_con_dataset(tidal_model)
        all_cons = all_cons.sortby('name')
        con_list = all_cons.name.data.tolist()
        for idx, con in enumerate(con_list):
            if con in self.tidal_db_cons:
                all_cons.enabled[idx] = 1
        data.cons = all_cons
        data.commit()

        self.build_data['tidal_sims'].append((tidal_py_comp, 'Tidal Constituents'))

        self.tidal_db_idx = -1
        self.tidal_db_type = ''
        self.tidal_db_cons = []

        # Set to true so only one Tidal Database simulation is added
        self.tidal_db_added = True

    def _add_advanced_card(self, block, line_data):
        """Add an advanced card from the line.

        Args:
            block (str): The block name this line came from.
            line_data (list): The list of strings in the line after it is split by whitespace.
        """
        lspaces = len(self.current_orig_line) - len(self.current_orig_line.lstrip())  # To count leading spaces.
        if len(line_data) > 0 and line_data[0] not in UNHANDLED_CARDS:
            self.advanced_card_blocks.append(block)
            self.advanced_card_cards.append((' ' * lspaces) + line_data[0])  # Add leading spaces for adv. card name.
            value = ' '.join(line_data[1:]) if len(line_data) > 1 else ''
            self.advanced_card_values.append(value)

    def _add_advanced_cards(self):
        """Adds advanced cards to the simulation."""
        self.sim_data.set_advanced_card_table(
            self.advanced_card_blocks, self.advanced_card_cards, self.advanced_card_values
        )

    def _write_fake_fort14(self, filename, node_locs, arc_indices):
        """Write a fake fort.14 file for deprecated WSE/velocity extraction boundaries.

        Args:
            filename (str): Path to the file to write
            node_locs (list[tuple[float, float, float]]): x,y,z coordinate tuples of the node locations to write
            arc_indices (list[int]): Indices of the BC arcs in the BC data cube
        """
        XmLog().instance.info(
            f'Writing fort.14 for deprecated WSE/Velocity extraction boundaries: {io_util.logging_filename(filename)}'
        )
        with open(filename, 'w') as f:
            f.write(f'\n0  {len(node_locs)}\n')
            for i, loc in enumerate(node_locs):
                f.write(f'{i + 1:10d} {loc[0]:15.10f} {loc[1]:15.10f} {loc[2] * -1.0:15.10f}\n')
            f.write('0 = Number of open boundaries\n')
            f.write('0 = Total number of open boundary nodes\n')
            f.write('0 = Number of land boundaries\n')
            f.write('0 = Total number of land boundary nodes\n')
        # Now update links in BC data cube, we'll clear these out later so the temp file doesn't show up in the GUI.
        for idx in arc_indices:
            self.bc_arc_values['parent_adcirc_14'][idx] = filename

    def _create_fake_fort63(self, filename, data_dict):
        """Creates a fake fort.63 dataset for deprecated WSE/velocity extraction boundaries.

        Args:
            filename (str): Path to the file to write
            data_dict (dict): The dict of extracted parent data. Should be self.parent_adcirc_wse or
                self.parent_adcirc_vel

        Returns:
            str: UUID generated for the dataset's geometry
        """
        geom_uuid = str(uuid.uuid4())  # Return this in case we also need to write the velocity dataset
        writer = DatasetWriter(
            h5_filename=filename,
            name='Water Surface Elevation (63)',
            geom_uuid=geom_uuid,
            null_value=-99999.0,
            time_units='Hours'
        )
        writer.write_xmdf_dataset(data_dict['times'], np.array(data_dict['wse']).T)
        # Update the dataset references in the BC data cube
        for idx in data_dict['arc_indices']:
            self.bc_arc_values['parent_adcirc_solution_wse'][idx] = filename
            self.bc_arc_values['parent_adcirc_wse_path'][idx] = writer.group_path
        return geom_uuid

    def _create_fake_fort64(self, filename, geom_uuid):
        """Creates a fake fort.64 dataset for deprecated WSE/velocity extraction boundaries.

        Args:
            filename (str): Path to the file to write
            geom_uuid (str): UUID of the geometry (that doesn't exist yet). Should be the same one used for the fort.63
                dataset.
        """
        writer = DatasetWriter(
            h5_filename=filename,
            name='Depth-averaged Velocity (64)',
            geom_uuid=geom_uuid,
            null_value=-99999.0,
            time_units='Hours',
            num_components=2,
            overwrite=False
        )
        # Transpose and stack the vx and vy arrays so we have one array of shape (num_times, num_nodes, 2)
        values = np.stack(
            [np.array(self.parent_adcirc_vel['velx']).T,
             np.array(self.parent_adcirc_vel['vely']).T], axis=-1
        )
        writer.write_xmdf_dataset(self.parent_adcirc_vel['times'], values)
        # Update the dataset references in the BC data cube
        for idx in self.parent_adcirc_vel['arc_indices']:
            self.bc_arc_values['parent_adcirc_solution'][idx] = filename
            self.bc_arc_values['parent_adcirc_velocity_path'][idx] = writer.group_path

    def _get_temp_adcirc_filename(self, process_temp, extension):
        """Get a filename in the temp area for writing fake parent ADCIRC files.

        Args:
            process_temp (bool): True if the file should be in the process temp folder, False if in the XMS temp folder
            extension (str): The file extension, without the '.'

        Returns:
            str: See description
        """
        temp_dir = XmEnv.xms_environ_process_temp_directory() if process_temp else XmEnv.xms_environ_temp_directory()
        if XmEnv.xms_environ_running_tests() == 'TRUE':  # If testing, generate a filename that is not so random.
            count = str(self._test_file_count)
            basename = f'{count * 8}-{count * 4}-{count * 4}-{count * 4}-{count * 12}'
            self._test_file_count += 1
            return os.path.join(temp_dir, f'{basename}.{extension}')
        else:
            return os.path.join(temp_dir, f'{str(uuid.uuid4())}.{extension}')

    def _write_parent_adcirc_files(self):
        """Writes fake parent ADCIRC files for legacy WSE/velocity extraction boundaries."""
        if self.parent_adcirc_wse['arc_indices']:
            filename = self._get_temp_adcirc_filename(True, '14')
            self._write_fake_fort14(filename, self.parent_adcirc_wse['locs'], self.parent_adcirc_wse['arc_indices'])
            filename = self._get_temp_adcirc_filename(False, 'h5')
            self._create_fake_fort63(filename, self.parent_adcirc_wse)
        if self.parent_adcirc_vel['arc_indices']:
            filename = self._get_temp_adcirc_filename(True, '14')
            self._write_fake_fort14(filename, self.parent_adcirc_vel['locs'], self.parent_adcirc_vel['arc_indices'])
            filename = self._get_temp_adcirc_filename(False, 'h5')
            geom_uuid = self._create_fake_fort63(filename, self.parent_adcirc_vel)
            self._create_fake_fort64(filename, geom_uuid)

    def _read_parent_adcirc_xmdf_files(self):
        """Read parent Adcric XMDF files."""
        # If no xms_data, we don't have a project tree
        if not self.xms_data:
            return

        # Write ADCIRC files for any legacy WSE extraction boundaries first.
        self._write_parent_adcirc_files()

        XmLog().instance.info('Reading parent ADCIRC files')
        adcirc_files = {}
        for bc_arc in range(len(self.bc_arc_values['parent_adcirc_solution_type'])):
            if self.bc_arc_values['parent_adcirc_solution_type'][bc_arc] != 'XMDF':
                # Default is ASCII
                continue
            bc_files = {'wse_solution': None, 'velocity_solution': None, 'fort14': ''}
            if self.bc_arc_values['parent_adcirc_solution'][bc_arc] != '(none selected)':
                solution = self.bc_arc_values['parent_adcirc_solution'][bc_arc]
                path = self.bc_arc_values['parent_adcirc_velocity_path'][bc_arc]
                bc_files['velocity_solution'] = (solution, path)
            if self.bc_arc_values['parent_adcirc_solution_wse'][bc_arc] != '(none selected)':
                solution = self.bc_arc_values['parent_adcirc_solution_wse'][bc_arc]
                path = self.bc_arc_values['parent_adcirc_wse_path'][bc_arc]
                bc_files['wse_solution'] = (solution, path)
            if self.bc_arc_values['parent_adcirc_14'][bc_arc] != '(none selected)':
                bc_files['fort14'] = self.bc_arc_values['parent_adcirc_14'][bc_arc]
            adcirc_files[bc_arc] = bc_files

        parent_reader = ParentAdcircReader(self.xms_data, file_map=adcirc_files)
        parent_reader.read()
        self.build_data['ugrids'].extend(parent_reader.ugrids)
        self.build_data['datasets'].extend([dataset for _, dataset in parent_reader.datasets.items()])
        # Set UUID reference links in the BC data
        for bc_arc, (geom_uuid, elev_uuid, vel_uuid) in parent_reader.bc_dset_uuids.items():
            self.bc_arc_values['parent_adcirc_14_uuid'][bc_arc] = geom_uuid
            self.bc_arc_values['parent_adcirc_solution_wse'][bc_arc] = elev_uuid
            self.bc_arc_values['parent_adcirc_solution'][bc_arc] = vel_uuid
        # Clear the fort.14 file references in the BC data cube for legacy WSE/Velocity extraction boundaries. We don't
        # want them showing up in the GUI.
        for idx in self.parent_adcirc_wse['arc_indices']:
            self.bc_arc_values['parent_adcirc_14'][idx] = ''
        for idx in self.parent_adcirc_vel['arc_indices']:
            self.bc_arc_values['parent_adcirc_14'][idx] = ''

    def extract_filename(self, line, pos=1):
        """Extract a filename from a line of text.

        Args:
            line (str): The line of text containing the filename
            pos (Optional[int]) The list index of the filename after splitting the line on whitespace

        Returns:
            str: The normalized absolute path and filename without quotes. Empty string on error.
        """
        msg = f'Error extracting filename from line {self.lineno}: {line}'
        tokens = safe_shlex_split(line, msg, min_tokens=pos + 1)
        return self.normalize_path(tokens[pos]) if tokens else ''

    def normalize_path(self, file_path):
        """Normalize a file path.

        Relative paths are converted to absolute. All characters in the path will be lower case. Not valid on Unix, but
        convenient for use as dict keys.

        Args:
            file_path (str): The filename to be normalized.

        Returns:
            str: The normalized absolute path and filename. Empty string on error
        """
        try:
            abs_path = clean_token(file_path)
            if not os.path.isabs(abs_path):
                abs_path = os.path.join(os.path.dirname(self.filename), abs_path)
            return os.path.normpath(os.path.normcase(abs_path)).replace('\\', '/')
        except Exception:
            XmLog().instance.error(f'Unable to resolve file reference on line {self.lineno}: {file_path}')
            return ''

    def _extract_parent_data(self, locs, cell_ids, arc_group, idx, use_velocity):
        """Extract data to create a fake parent ADCIRC solution from legacy WSE extraction boundaries.

        Args:
            locs (list[tuple[float, float, float]]): The x,y,z coordinate tuples of the arc's locations
            cell_ids (list[int]): Cell ids of the arc's locations
            arc_group (h5py.Group): The BC arc's H5 group in the _mp.h5 file
            idx (int): Index of the BC arc's attributes in the BC data cube
            use_velocity (bool): True if we should extract velocity data, False if only WSE
        """
        # Read the WSE values, they are always present
        if 'WSE' not in arc_group:
            XmLog().instance.error('Unable to find WSE curve for extracting legacy BC data.')
            return
        data_dict = self.parent_adcirc_vel if use_velocity else self.parent_adcirc_wse
        data_dict['locs'].extend(locs)
        data_dict['arc_indices'].append(idx)
        # Only need to get the times once. I'm assuming all boundaries in the simulation are from the same run.
        if data_dict['times'] is None:
            data_dict['times'] = arc_group['Times'][:]
        wse_group = arc_group['WSE']
        for cell_id in cell_ids:
            data_dict['wse'].append(wse_group[f'WaterLevel_{cell_id}'][:])

        # Now extract the velocities, if they exist.
        if not use_velocity:
            return
        top_group = arc_group['Top']
        bottom_group = arc_group['Bottom']
        left_group = arc_group['Left']
        right_group = arc_group['Right']
        angle = math.radians(self.quad_ugrid.angle if self.quad_ugrid else 0.0)
        cosangle = math.cos(angle)
        sinangle = math.sin(angle)
        for cell_id in cell_ids:
            dset_name = f'Velocity_{cell_id}'
            tops = top_group[dset_name][:]
            bottoms = bottom_group[dset_name][:]
            lefts = left_group[dset_name][:]
            rights = right_group[dset_name][:]
            vu = (lefts + rights) / 2.0
            vv = (tops + bottoms) / 2.0
            vx = (cosangle * vu) + (-1.0 * sinangle * vv)
            vy = (sinangle * vu) + (cosangle * vv)
            data_dict['velx'].append(vx)
            data_dict['vely'].append(vy)

    def _read_legacy_bc(self, arc_group, idx):
        """Read BC data from legacy .mp H5 files.

        Args:
            arc_group (h5py.Group): The BC arc's group handle
            idx (int): Index of the BC arc in self.bc_arc_values

        Returns:
            Union[bool, None]: True if we need to extract WSE and velocity parent data, False if we need to extract only
                WSE parent data, None if no parent data extraction needed.
        """
        name = os.path.basename(arc_group.name)
        self.bc_arc_values['name'][idx] = name
        bc_type = arc_group['CSType'][0]
        extract_parent = None
        legacy_types = [
            BC_LEGACY_TIDAL, BC_LEGACY_WSE_DEF, BC_LEGACY_WSE_EXT, BC_LEGACY_WSE_VEL_EXT, BC_LEGACY_HARMONIC
        ]
        if bc_type == BC_LEGACY_FLOW:
            self.bc_arc_types[idx] = BC_FLUX
            self.bc_arc_values['bc_type'][idx] = 'Flow rate-forcing'
            self.bc_arc_values['flow_source'][idx] = 'Curve'

        elif bc_type in legacy_types:
            self.bc_arc_types[idx] = BC_WSE
            self.bc_arc_values['bc_type'][idx] = 'WSE-forcing'
            if bc_type in [BC_LEGACY_WSE_EXT, BC_LEGACY_WSE_VEL_EXT]:  # Parent ADCIRC types
                self.bc_arc_values['wse_source'][idx] = 'Parent ADCIRC'
                self.bc_arc_values['parent_adcirc_solution_type'][idx] = 'XMDF'
                if bc_type == BC_LEGACY_WSE_VEL_EXT:
                    self.bc_arc_values['use_velocity'][idx] = 1
                    extract_parent = True
                else:
                    extract_parent = False
            elif bc_type == BC_LEGACY_TIDAL:
                self.bc_arc_values['wse_source'][idx] = 'Tidal constituent'
            elif bc_type == BC_LEGACY_HARMONIC:
                self.bc_arc_values['wse_source'][idx] = 'Harmonic'
            else:  # BC_LEGACY_WSE_DEF
                self.bc_arc_values['wse_source'][idx] = 'Curve'
        elif bc_type != BC_LEGACY_UNASSIGNED:  # Warn if boundary type is unsupported
            XmLog().instance.warning(
                f'Boundary cellstring with name "{name}" was assigned an unsupported boundary '
                'type. It will be switched to "Unassigned".'
            )
        return extract_parent

    def read_boundary_conditions(self):
        """Read the _mp.h5 file."""
        proj_name = os.path.splitext(os.path.basename(self.filename))[0]
        comp_uuid = str(uuid.uuid4())
        bc_comp_dir = os.path.join(self.comp_dir, comp_uuid)
        os.makedirs(bc_comp_dir, exist_ok=True)
        bc_mainfile = os.path.join(bc_comp_dir, 'bc_comp.nc')
        self.bc_mainfile = bc_mainfile
        bc_comp = BCComponent(bc_mainfile)
        bc_data = bc_comp.data

        # In case the user changed the name of the .cmcards file but kept the rest intact, use real _mp.h5 filename.
        mp_filename = self.bc_filename
        if len(mp_filename) == 0:
            mp_filename = os.path.normpath(os.path.join(os.path.dirname(self.filename), f'{proj_name}_mp.h5'))

        mp_filename_compare = mp_filename.replace('\\', '/')
        curve_tuples = []
        if mp_filename_compare in self.curves:
            curve_tuples = self.curves[mp_filename_compare]
        with h5py.File(mp_filename, 'r') as file:
            # Import meteorological data
            param_group = file.get('PROPERTIES/Model Params')
            if not param_group:
                XmLog().instance.error("No boundary conditions are present.")
                return
            met_name = 'Met Stations'
            for item in list(param_group.keys()):  # Older files may have name 'Meterological Stations' instead
                if 'Meteorological' in item:
                    met_name = item
            if met_name in param_group:
                met_group = param_group.get(met_name)
                for curve_group_name in met_group.keys():
                    curve_group = met_group.get(curve_group_name)
                    curve_id = int(float(curve_group_name.strip('Sta')) + 0.5)
                    data_vars = {
                        'time': xr.DataArray(data=curve_group['Times'][:]),
                        'direction': xr.DataArray(data=curve_group['Direction'][:]),
                        'velocity': xr.DataArray(data=curve_group['Magnitude'][:])
                    }
                    curve_data = xr.Dataset(data_vars=data_vars)
                    self.sim_data.set_direction_curve_from_meteorological_station(curve_id, curve_data)
                    curve_group_name_compare = f'PROPERTIES/Model Params/{met_name}/{curve_group_name}'
                    for curve in curve_tuples:
                        if curve[1] == 'met' and curve[0] == curve_group_name_compare:
                            self.met_station_values['direction'][curve[2]] = np.int64(curve_id)
            if 'WindCurve' in param_group:
                wind_group = param_group.get('WindCurve')
                wind_from_table = {
                    'time': xr.DataArray(wind_group['Times'][:]),
                    'direction': xr.DataArray(wind_group['Direction'][:]),
                    'velocity': xr.DataArray(wind_group['Magnitude'][:])
                }
                self.sim_data.wind_from_table = xr.Dataset(data_vars=wind_from_table)

            if 'TemperatureParameters' in param_group:
                temp_group = param_group.get('TemperatureParameters')
                atmospheric_table = {
                    'time': xr.DataArray(temp_group['Times'][:]),
                    'air_temp': xr.DataArray(temp_group['AirTemp'][:]),
                    'dewpoint': xr.DataArray(temp_group['DewPoint'][:]),
                    'cloud_cover': xr.DataArray(temp_group['CloudCover'][:]),
                    'solar_radiation': xr.DataArray(temp_group['SolarRadiation'][:])
                }
                self.sim_data.atmospheric_table = xr.Dataset(data_vars=atmospheric_table)

            group_names = param_group.keys()
            bc_curve_id = 1
            quad_cells = {}
            arcs = []
            unassigned_arcs = []
            flux_arcs = []
            wse_arcs = []
            next_pt_id = 1
            if self.quad_tree:
                if self.rect_cells:
                    quad_cells = {cell.id: cell.get_location() for cell in self.rect_cells}
                elif self.quad_ugrid:
                    xm_ugrid = self.quad_ugrid.ugrid
                    num_cells = xm_ugrid.cell_count
                    quad_cells = {}
                    for i in range(num_cells):
                        if self.refinement_idx:
                            idx = self.refinement_idx[i][CMSFlowTelImporter.new_cell_idx_column]
                        else:
                            idx = i
                        centroid = xm_ugrid.get_cell_centroid(idx)
                        if centroid[0]:
                            quad_cells[i + 1] = Point(centroid[1][0], centroid[1][1], centroid[1][2])

            offset = 0
            for name in group_names:
                # Groups used to be named 'Boundary', 'Boundary (1)', ..., 'Boundary (n+1)'. Now they are 'Boundary_#n'.
                if name == 'Boundary' or name.find('Boundary ') >= 0:
                    offset = 1
                    break

            for name in group_names:
                # Groups used to be named 'Boundary', 'Boundary (1)', ..., 'Boundary (n+1)'. Now they are 'Boundary_#n'.
                if name.find('Boundary_#') >= 0:
                    arc_id = int(name.split('#')[1]) + offset  # New style group name
                elif name.find('Boundary') >= 0:
                    arc_id = 1 if name == 'Boundary' else int(name.replace('Boundary (', '').replace(')', '')) + offset
                else:
                    continue

                arc = Arc(feature_id=arc_id)
                # older versions have this information in the mp file
                if arc_id not in self.bc_arc_comp_ids:
                    self.add_bc_arc()
                    self.bc_arc_comp_ids[-1] = arc_id
                idx = self.bc_arc_comp_ids.index(arc_id)
                # Check for BC type in legacy "CSType" dataset
                arc_group = param_group.get(name)
                extract_parent = None
                if 'CSType' in arc_group:
                    extract_parent = self._read_legacy_bc(arc_group, idx)

                if self.bc_arc_types[idx] == BC_UNASSIGNED:
                    unassigned_arcs.append(arc_id)
                elif self.bc_arc_types[idx] == BC_FLUX:
                    flux_arcs.append(arc_id)
                elif self.bc_arc_types[idx] == BC_WSE:
                    wse_arcs.append(arc_id)
                if 'Flow' in arc_group:
                    time_values = arc_group.get('Times')[:]
                    other_values = arc_group.get('Flow')[:]
                    curve_data = {
                        'Time (hrs)': xr.DataArray(data=time_values),
                        'Flow (m^3/s Total, not per cell)': xr.DataArray(data=other_values),
                    }
                    curve = xr.Dataset(data_vars=curve_data)
                    bc_data.set_flow_forcing_curve(bc_curve_id, curve)
                    self.bc_arc_values['flow_curve'][idx] = bc_curve_id
                    bc_curve_id += 1
                elif 'WaterLevel' in arc_group:
                    time_values = arc_group.get('Times')[:]
                    other_values = arc_group.get('WaterLevel')[:]
                    curve_data = {
                        'Time (hrs)': xr.DataArray(data=time_values),
                        'WSE (m)': xr.DataArray(data=other_values),
                    }
                    curve = xr.Dataset(data_vars=curve_data)
                    bc_data.set_wse_forcing_curve(bc_curve_id, curve)
                    self.bc_arc_values['wse_forcing_curve'][idx] = bc_curve_id
                    bc_curve_id += 1
                if 'Offset' in arc_group:
                    time_values = arc_group.get('Offset_Times')[:]
                    other_values = arc_group.get('Offset')[:]
                    curve_data = {
                        'Time (hrs)': xr.DataArray(data=time_values),
                        'WSE (m)': xr.DataArray(data=other_values),
                    }
                    curve = xr.Dataset(data_vars=curve_data)
                    bc_data.set_wse_offset_curve(bc_curve_id, curve)
                    self.bc_arc_values['wse_offset_curve'][idx] = bc_curve_id
                    bc_curve_id += 1
                if 'Salinity' in arc_group:
                    time_values = arc_group.get('Sal_Times')[:]
                    other_values = arc_group.get('Salinity')[:]
                    curve_data = {
                        'Time (hrs)': xr.DataArray(data=time_values),
                        'WSE (m)': xr.DataArray(data=other_values),
                    }
                    curve = xr.Dataset(data_vars=curve_data)
                    bc_data.set_salinity_curve(bc_curve_id, curve)
                    self.bc_arc_values['salinity_curve'][idx] = bc_curve_id
                    bc_curve_id += 1
                if 'Temperature' in arc_group:
                    time_values = arc_group.get('Temp_Times')[:]
                    other_values = arc_group.get('Temperature')[:]
                    curve_data = {
                        'Time (hrs)': xr.DataArray(data=time_values),
                        'WSE (m)': xr.DataArray(data=other_values),
                    }
                    curve = xr.Dataset(data_vars=curve_data)
                    bc_data.set_temperature_curve(bc_curve_id, curve)
                    self.bc_arc_values['temperature_curve'][idx] = bc_curve_id
                    bc_curve_id += 1
                if 'Cells' in arc_group and quad_cells:
                    arc_cells = arc_group['Cells'][:]
                    num_arc_cells = len(arc_cells)
                    locs = []  # Store the locations in case we need to do a parent data extraction
                    cell_ids = []
                    if num_arc_cells > 0:
                        cell_id = arc_cells[0]
                        start_pt = quad_cells[cell_id]
                        cell_ids.append(cell_id)
                        cell_id = arc_cells[-1]
                        end_pt = quad_cells[arc_cells[-1]]
                        cell_ids.append(cell_id)
                        if start_pt.id <= 0:
                            start_pt.id = next_pt_id
                            next_pt_id += 1
                        if end_pt.id <= 0:
                            end_pt.id = next_pt_id
                            next_pt_id += 1
                        locs.append((start_pt.x, start_pt.y, start_pt.z))
                        locs.append((end_pt.x, end_pt.y, end_pt.z))
                        arc.start_node = start_pt
                        arc.end_node = end_pt
                        if num_arc_cells > 2:
                            verts = []
                            for cell_id in arc_cells[1:-1]:
                                vert = quad_cells[cell_id]
                                cell_ids.append(cell_id)
                                locs.append((vert.x, vert.y, vert.z))
                                verts.append(vert)
                            arc.vertices = verts
                        arcs.append(arc)
                        # If we need to extract parent data, we can do it now that we have the locations.
                        if extract_parent is not None:
                            self._extract_parent_data(locs, cell_ids, arc_group, idx, extract_parent)

        if len(self.bc_arc_comp_ids) > 0:
            self._read_parent_adcirc_xmdf_files()
            cov_uuid = str(uuid.uuid4())
            self.bc_cov = Coverage(uuid=cov_uuid, name='Boundary Conditions', projection=self.grid_projection)
            self.bc_cov.arcs = arcs
            self.bc_cov.complete()

            bc_data.info.attrs['cov_uuid'] = cov_uuid
            bc_data.info.attrs['next_comp_id'] = max(self.bc_arc_comp_ids) + 1
            # bc_data.info.attrs['arc_display_uuid'] = ''
            # bc_data.arcs = self.bc_arc_values
            arc_table = {}
            for key in self.bc_arc_values.keys():
                arc_table[key] = ('comp_id', self.bc_arc_values[key])
            coords = {'comp_id': self.bc_arc_comp_ids}
            bc_data.arcs = xr.Dataset(data_vars=arc_table, coords=coords)
            for table, harmonic in self.harmonic_values.items():
                harmonic_table_data = {
                    'speed': xr.DataArray(data=np.array(harmonic['speed'], dtype=float)),
                    'amplitude': xr.DataArray(data=np.array(harmonic['amplitude'], dtype=float)),
                    'phase': xr.DataArray(data=np.array(harmonic['phase'], dtype=float))
                }
                bc_data.set_harmonic_table(table, xr.Dataset(data_vars=harmonic_table_data))

            # Old files had global tidal constituents. Use them if we have them
            global_tidal_data = {
                'constituent': xr.DataArray(data=np.array(self.global_constituents['constituent'], dtype=object)),
                'amplitude': xr.DataArray(data=np.array(self.global_constituents['amplitude'], dtype=float)),
                'phase': xr.DataArray(data=np.array(self.global_constituents['phase'], dtype=float))
            } if self.global_constituents['constituent'] else None
            for table, tidal in self.tidal_values.items():
                tidal_table_data = {
                    'constituent': xr.DataArray(data=np.array(tidal['constituent'], dtype=object)),
                    'amplitude': xr.DataArray(data=np.array(tidal['amplitude'], dtype=float)),
                    'phase': xr.DataArray(data=np.array(tidal['phase'], dtype=float))
                } if global_tidal_data is None else global_tidal_data
                bc_data.set_tidal_table(table, xr.Dataset(data_vars=tidal_table_data))
            bc_data.commit()

            # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
            id_file = os.path.join(bc_comp_dir, BC_INITIAL_ATT_ID_FILE)
            write_display_option_ids(id_file, self.bc_arc_comp_ids)
            id_file = os.path.join(bc_comp_dir, BC_INITIAL_COMP_ID_FILE)
            write_display_option_ids(id_file, self.bc_arc_comp_ids)
            id_file = os.path.join(bc_comp_dir, 'unassigned_arcs.display_ids')
            write_display_option_ids(id_file, unassigned_arcs)
            id_file = os.path.join(bc_comp_dir, 'flow_arcs.display_ids')
            write_display_option_ids(id_file, flux_arcs)
            id_file = os.path.join(bc_comp_dir, 'wse_arcs.display_ids')
            write_display_option_ids(id_file, wse_arcs)
            self.build_data['bc_cov_comp'] = bc_comp
            self.build_data['bc_cov'] = self.bc_cov
        else:
            XmLog().instance.warning('No boundary condition information found')

    def read_geometry(self):
        """Read the tel file.

        We need the Quadtree UUID for linking up dataset selectors.
        """
        # Look for a .worldtel if the TELESCOPING card was read, but the referenced file doesn't exist. This is due to
        # some smelly stuff we did ~12.3 that would preprocess a .wordtel file with telfix before running CMS-Flow.
        if self.telescoping_filename and not os.path.isfile(self.telescoping_filename):
            self.telescoping_filename = os.path.splitext(self.telescoping_filename)[0] + '.worldtel'
        # If reading a <= 5.2 version, invert the grid angle. CMS-Flow and SMS convention is CCW from the east, but
        # we did some more smelly stuff in older versions where we would invert on both import and export.
        need_invert = len(self.cms_version) > 1 and self.cms_version[0] <= 5 and self.cms_version[1] <= 2
        if need_invert and self.grid_angle is not None:
            self.grid_angle = 360.0 - self.grid_angle

        if os.path.isfile(self.telescoping_filename):
            reader = CMSFlowTelImporter(
                self.telescoping_filename, self.grid_projection, self.cms_version, self.bathymetry
            )
            reader.angle = self.grid_angle  # If conflicting grid angles in .tel and .cmcards, angle in .cmcards wins.
            reader.read_tel_file()
            self.quad_ugrid = reader.quad_ugrid
            # Write out a mapping for reading datasets later
            if reader.refinement_idx:
                self.refinement_idx = reader.refinement_idx
                renumber_file = os.path.join(self.sim_comp_dir, 'renumber.txt')
                self._write_renumber_file(renumber_file, reader)
        elif os.path.isfile(self.grid_filename):  # Read the geometry from legacy grid.h5 file
            reader = CMSFlowH5GridImporter(
                self.grid_filename, self.grid_projection, self.bathymetry, self.cell_activity_group
            )
            reader.read_h5_grid_file()
            self.quad_ugrid = reader.quad_ugrid
        else:
            XmLog().warning('Unable to find referenced grid file. Partial import will be attempted.')
            return

        if not self.quad_ugrid:
            XmLog().warning('Unable to read grid file. Partial import will be attempted.')
            return

        # Add the activity coverage if we created one.
        if reader.activity_coverage:
            self.build_data['activity_cov'] = reader.activity_coverage

        temp_dir = self.comp_dir
        xmc_file = os.path.join(temp_dir, f'{str(self.quad_ugrid.uuid)}.xmc')
        self.quad_ugrid.write_to_file(xmc_file)
        geom_uuid = self.quad_ugrid.uuid
        # get a name for the new quadtree we are creating
        ugrid_name = self.telescoping_filename
        ugrid_name = os.path.basename(self.telescoping_filename)
        ugrid = UGrid(xmc_file, uuid=geom_uuid, projection=self.grid_projection, name=ugrid_name)
        self.quad_tree = ugrid

        if self.quad_tree:
            self.sim_data.info.attrs['domain_uuid'] = geom_uuid
            self.build_data['ugrids'].append(self.quad_tree)
            renumber_index = [0 for _ in self.refinement_idx]
            if self.refinement_idx:
                for old_index, new_index_list in enumerate(self.refinement_idx):
                    renumber_index[new_index_list[CMSFlowTelImporter.new_cell_idx_column]] = old_index
            self._update_input_datasets(geom_uuid, renumber_index)

    def parse_dataset_reference(self, full_line, dset_name, uuid_container, row=-1):
        """Parse a dataset reference's filename and group path from a line in the file.

        Args:
            full_line (str): The entire line containing the dataset reference, including the card
            dset_name (str): The dataset name for storage
            uuid_container (obj): The object holding the dataset uuid
            row (Optional[int]): Table row index of the dataset uuid
        """
        msg = f'Invalid {dset_name} dataset reference found on line {self.lineno}: {full_line}'
        if len(self.cms_version) > 1 and self.cms_version[0] == 5 and self.cms_version[1] == 0:
            # We had a dumb exporter in SMS 12.3 (CMS-Flow v5.0) that wrote incomplete dataset references. Suppress the
            # error message if one of those.
            msg = ''
        tokens = safe_shlex_split(full_line, msg, min_tokens=3)
        if tokens and tokens[1] != '""':  # Some D##_DATASET cards still exists with empty strings for location of dset.
            self.add_dataset(tokens[1], tokens[2], dset_name, uuid_container, row)

    def add_met_station(self):
        """Adds a meteorological station."""
        self.met_station_values['name'].append('')
        self.met_station_values['x'].append(0.0)
        self.met_station_values['y'].append(0.0)
        self.met_station_values['height'].append(0.0)
        self.met_station_values['direction'].append(-1)

    def add_sediment_size(self):
        """Adds a sediment diameter size."""
        self.sediment_size_values['diameter_value'].append(0.2)
        self.sediment_size_values['diameter_units'].append('mm')
        self.sediment_size_values['fall_velocity_method'].append('Soulsby (1997)')
        self.sediment_size_values['fall_velocity_value'].append(0.0)
        self.sediment_size_values['fall_velocity_units'].append('m/s')
        self.sediment_size_values['corey_shape_factor'].append(0.7)
        self.sediment_size_values['critical_shear_method'].append('Soulsby (1997)')
        self.sediment_size_values['critical_shear_stress'].append(0.2)

    def add_bed_layer(self):
        """Adds a bed layer."""
        self.bed_layer_values['layer_id'].append(0)
        self.bed_layer_values['layer_thickness_type'].append('Dataset')
        self.bed_layer_values['layer_thickness'].append('')
        self.bed_layer_values['layer_thickness_const'].append(0.5)
        self.bed_layer_values['d05'].append('')
        self.bed_layer_values['d10'].append('')
        self.bed_layer_values['d16'].append('')
        self.bed_layer_values['d20'].append('')
        self.bed_layer_values['d30'].append('')
        self.bed_layer_values['d35'].append('')
        self.bed_layer_values['d50'].append('')
        self.bed_layer_values['d65'].append('')
        self.bed_layer_values['d84'].append('')
        self.bed_layer_values['d90'].append('')
        self.bed_layer_values['d95'].append('')

    def add_bc_arc(self):
        """Adds the attributes for an arc to the lists.
        """
        self.bc_arc_values['name'] = np.append(self.bc_arc_values['name'], '(none selected)')
        self.bc_arc_values['bc_type'] = np.append(self.bc_arc_values['bc_type'], 'Unassigned')
        self.bc_arc_values['flow_source'] = np.append(self.bc_arc_values['flow_source'], 'Constant')
        self.bc_arc_values['constant_flow'] = np.append(self.bc_arc_values['constant_flow'], 0.0)
        self.bc_arc_values['flow_curve'] = np.append(self.bc_arc_values['flow_curve'], 0)
        self.bc_arc_values['specify_inflow_direction'] = np.append(self.bc_arc_values['specify_inflow_direction'], 0)
        self.bc_arc_values['flow_direction'] = np.append(self.bc_arc_values['flow_direction'], 0.0)
        self.bc_arc_values['flow_conveyance'] = np.append(self.bc_arc_values['flow_conveyance'], 0.667)
        self.bc_arc_values['wse_source'] = np.append(self.bc_arc_values['wse_source'], 'Constant')
        self.bc_arc_values['wse_const'] = np.append(self.bc_arc_values['wse_const'], 0.0)
        self.bc_arc_values['wse_forcing_curve'] = np.append(self.bc_arc_values['wse_forcing_curve'], 0)
        self.bc_arc_values['use_velocity'] = np.append(self.bc_arc_values['use_velocity'], 0)
        self.bc_arc_values['parent_cmsflow'] = np.append(self.bc_arc_values['parent_cmsflow'], '(none selected)')
        self.bc_arc_values['parent_adcirc_14_uuid'] = np.append(self.bc_arc_values['parent_adcirc_14_uuid'], '')
        self.bc_arc_values['parent_adcirc_14'] = np.append(self.bc_arc_values['parent_adcirc_14'], '(none selected)')
        self.bc_arc_values['parent_adcirc_solution_type'] = \
            np.append(self.bc_arc_values['parent_adcirc_solution_type'], 'ASCII')
        self.bc_arc_values['parent_adcirc_63'] = np.append(self.bc_arc_values['parent_adcirc_63'], '(none selected)')
        self.bc_arc_values['parent_adcirc_64'] = np.append(self.bc_arc_values['parent_adcirc_64'], '(none selected)')
        self.bc_arc_values['parent_adcirc_solution'] = \
            np.append(self.bc_arc_values['parent_adcirc_solution'], '(none selected)')
        self.bc_arc_values['parent_adcirc_solution_wse'] = \
            np.append(self.bc_arc_values['parent_adcirc_solution_wse'], '(none selected)')
        self.bc_arc_values['parent_adcirc_wse_path'] = \
            np.append(self.bc_arc_values['parent_adcirc_wse_path'], '(none selected)')
        self.bc_arc_values['parent_adcirc_velocity_path'] = \
            np.append(self.bc_arc_values['parent_adcirc_velocity_path'], '(none selected)')
        self.bc_arc_values['parent_adcirc_start_time'] = \
            np.append(self.bc_arc_values['parent_adcirc_start_time'], '2000-01-01T00:00:00')
        self.bc_arc_values['wse_offset_type'] = np.append(self.bc_arc_values['wse_offset_type'], 'Constant')
        self.bc_arc_values['wse_offset_const'] = np.append(self.bc_arc_values['wse_offset_const'], 0.0)
        self.bc_arc_values['wse_offset_curve'] = np.append(self.bc_arc_values['wse_offset_curve'], 0)
        self.bc_arc_values['use_salinity_curve'] = np.append(self.bc_arc_values['use_salinity_curve'], 0)
        self.bc_arc_values['salinity_curve'] = np.append(self.bc_arc_values['salinity_curve'], 0)
        self.bc_arc_values['use_temperature_curve'] = np.append(self.bc_arc_values['use_temperature_curve'], 0)
        self.bc_arc_values['temperature_curve'] = np.append(self.bc_arc_values['temperature_curve'], 0)
        self.table_id += 1
        self.bc_arc_values['harmonic_table'] = np.append(self.bc_arc_values['harmonic_table'], self.table_id)
        self.bc_arc_values['tidal_table'] = np.append(self.bc_arc_values['tidal_table'], self.table_id)
        self.bc_arc_comp_ids.append(-1)
        self.bc_arc_types.append(BC_UNASSIGNED)
        self.harmonic_values[self.table_id] = {
            'speed': [],
            'amplitude': [],
            'phase': [],
        }
        self.tidal_values[self.table_id] = {
            'constituent': [],
            'amplitude': [],
            'phase': [],
        }

    def add_rubble_mound(self):
        """Adds the attributes for a polygon to the lists."""
        self.rubble_mound_values['name'] = np.append(self.rubble_mound_values['name'], '(none selected)')
        self.rubble_mound_values['rock_diameter'] = np.append(self.rubble_mound_values['rock_diameter'], 0.0)
        self.rubble_mound_values['rock_diameter_type'] = np.append(
            self.rubble_mound_values['rock_diameter_type'], PARAMETER_TYPE_CONSTANT_IDX
        )
        self.rubble_mound_values['ROCK_DIAMETER_DATASET'] = np.append(
            self.rubble_mound_values['ROCK_DIAMETER_DATASET'], ''
        )
        self.rubble_mound_values['porosity'] = np.append(self.rubble_mound_values['porosity'], 0.4)
        self.rubble_mound_values['porosity_type'] = np.append(
            self.rubble_mound_values['porosity_type'], PARAMETER_TYPE_CONSTANT_IDX
        )
        self.rubble_mound_values['STRUCTURE_POROSITY_DATASET'] = np.append(
            self.rubble_mound_values['STRUCTURE_POROSITY_DATASET'], ''
        )
        self.rubble_mound_values['base_depth'] = np.append(self.rubble_mound_values['base_depth'], 0.0)
        self.rubble_mound_values['base_depth_type'] = np.append(
            self.rubble_mound_values['base_depth_type'], PARAMETER_TYPE_CONSTANT_IDX
        )
        self.rubble_mound_values['STRUCTURE_BASE_DEPTH_DATASET'] = np.append(
            self.rubble_mound_values['STRUCTURE_BASE_DEPTH_DATASET'], ''
        )
        self.rubble_mound_values['calculation_method'] = np.append(
            self.rubble_mound_values['calculation_method'], 'Sidiropoulou et al. (2007)'
        )

    def parse_id_from_h5_path(self, path):
        """Parses a boundary condition arc's id from its group path in an H5 file.

        This method relies on the CMS-Flow naming convention of ending the group name with "#x", where 'x'
        is the arc id.

        Args:
            path (str): Path inside an H5 file to a boundary condition group.
        """
        try:
            pos = path.rfind('#')
            return int(clean_token(path[pos + 1:]))
        except Exception:
            XmLog().instance.error(f'Unable to parse BC arc id from H5 path on line {self.lineno}: {path}')
            return -1

    def add_bc_curve(self, file_name, path_in_file, widget_name, arc_id):
        """Add an arc bc curve to be read from an H5 file.

        Args:
            file_name (str): Name of the file containing the curve
            path_in_file (str): Path in the H5 file to the curve dataset
            widget_name (str): XML unique name of the widget
            arc_id (int): ID of the arc

        """
        path_in_file = clean_token(path_in_file)
        norm_path = self.normalize_path(file_name)
        if not norm_path:
            return  # Problem resolving the file reference
        if norm_path not in self.bc_curves:
            self.bc_curves[norm_path] = []
        self.bc_curves[norm_path].append((path_in_file, widget_name, arc_id))

    def add_curve(self, full_line, card_name, curve_name, row=-1):
        """Add a curve to be read from an H5 file.

        Args:
            full_line (str): Full line from the file, including the card
            card_name (str): Curve card, used for error reporting
            curve_name (str): The name of where to put the curve id
            row (Optional[int]): Table row index of the curve id
        """
        msg = f'Invalid {card_name} reference on line {self.lineno}: {full_line}'
        tokens = safe_shlex_split(full_line, msg, min_tokens=3)
        if not tokens:
            return  # Error parsing the line

        path_in_file = clean_token(tokens[2])
        filename = self.normalize_path(tokens[1])
        if not filename:
            return  # Error resolving the filepath
        self.curves.setdefault(filename, []).append((path_in_file, curve_name, row))

    def add_dataset(self, file_name, path_in_file, dataset_name, uuid_container, row):
        """Add a dataset to be read from an H5 file.

        Need to link up to the dataset selector using the UUID of the Quadtree geometry after it has been read
        and created.

        Args:
            file_name (str): Name of the file containing the dataset
            path_in_file (str): Path in the H5 file to the dataset
            dataset_name (str): The name of where to put the uuid.
            uuid_container (obj): The object holding the dataset uuid.
            row (int): Table row index of the dataset uuid.
        """
        path_in_file = clean_token(path_in_file)
        norm_path = self.normalize_path(file_name)
        path_tuple = (norm_path, path_in_file)
        if path_tuple not in self.datasets:
            self.datasets[path_tuple] = []
        self.datasets[path_tuple].append((uuid_container, dataset_name, row))

    def read_time_list(self, line_data, time_list_index):
        """Read a time list card line.

        Args:
             line_data (:obj:`list` of :obj:`str`): The time list card line split on whitespace
             time_list_index (int): The index of the time list (1-4).
        """
        num_rows = int(line_data[1])
        if num_rows > 0:
            idx = 2
            starts = []
            ends = []
            incs = []
            for _ in range(num_rows):
                starts.append(float(line_data[idx]))
                idx += 1
                ends.append(float(line_data[3]))
                idx += 1
                incs.append(float(line_data[4]))
                idx += 1
            list_table = {
                'start_time': xr.DataArray(data=np.array(starts, dtype=float)),
                'increment': xr.DataArray(data=np.array(incs, dtype=float)),
                'end_time': xr.DataArray(data=np.array(ends, dtype=float))
            }
            if time_list_index == 1:
                self.sim_data._list_1_table = xr.Dataset(data_vars=list_table)
            elif time_list_index == 2:
                self.sim_data._list_2_table = xr.Dataset(data_vars=list_table)
            elif time_list_index == 3:
                self.sim_data._list_3_table = xr.Dataset(data_vars=list_table)
            elif time_list_index == 4:
                self.sim_data._list_4_table = xr.Dataset(data_vars=list_table)

    def read_card(self, card_line):
        """Read a single line from the .cmcards file.

        Args:
             card_line (str): Line of text from the .cmcards file.

        """
        card_line_orig = card_line
        line_data = card_line.split()
        if not line_data:
            return
        card = line_data[0].upper()  # only want the card name uppercase, nothing else!

        if self.reading_met_station:
            # Meteorological wind station cards for table.
            if card == 'STATION_NAME':
                msg = f'Unable to parse meteorological station name on line {self.lineno}: {card_line_orig}'
                tokens = safe_shlex_split(card_line_orig, msg)
                if tokens:
                    self.met_station_values['name'][-1] = clean_token(line_data[1])
            elif card == 'COORDINATES':
                self.met_station_values['x'][-1] = float(line_data[1])
                self.met_station_values['y'][-1] = float(line_data[2])
            elif card == 'WIND_INPUT_CURVE':
                self.add_curve(card_line_orig, 'WIND_INPUT_CURVE', 'met', len(self.met_station_values['direction']) - 1)
            elif card == 'ANEMOMETER_HEIGHT':
                self.met_station_values['height'][-1] = float(line_data[1])
            elif card == 'MET_STATION_END':
                self.reading_met_station = False
            else:
                self._add_advanced_card('MET_STATION', line_data)

        elif self.reading_sediment_size:
            # Sediment size class cards for table.
            if card == 'DIAMETER':
                self.sediment_size_values['diameter_value'][-1] = float(line_data[1])
                self.sediment_size_values['diameter_units'][-1] = clean_token(line_data[2])
            elif card == 'FALL_VELOCITY_FORMULA':
                if len(line_data) > 1:
                    fall_velocity = ''
                    if line_data[1] == 'SOULSBY':
                        fall_velocity = 'Soulsby (1997)'
                    elif line_data[1] == 'WU_WANG':
                        fall_velocity = 'Wu and Wang (2006)'
                    elif line_data[1] == 'MANUAL':
                        fall_velocity = 'User specified'
                    self.sediment_size_values['fall_velocity_method'][-1] = fall_velocity
                    self.sim_data.sediment.attrs['USE_ADVANCED_SIZE_CLASSES'] = 1
            elif card == 'FALL_VELOCITY':
                msg = f'Unable to parse fall velocity on line {self.lineno}: {card_line_orig}'
                tokens = safe_shlex_split(card_line_orig, msg)
                if tokens:
                    self.sediment_size_values['fall_velocity_value'][-1] = float(clean_token(tokens[1]))
                    if len(tokens) > 2:  # Not sure if this is optional or not
                        self.sediment_size_values['fall_velocity_units'][-1] = clean_token(tokens[2])
            elif card == 'COREY_SHAPE_FACTOR':
                if len(line_data) > 1:
                    self.sediment_size_values['corey_shape_factor'][-1] = float(line_data[1])
            elif card == 'CRITICAL_SHEAR_FORMULA':
                if len(line_data) > 1:
                    shear_formula = ''
                    if line_data[1] == 'SOULSBY':
                        shear_formula = 'Soulsby (1997)'
                    elif line_data[1] == 'VAN_RIJN':
                        shear_formula = 'van Rijn (2007)'
                    elif line_data[1] == 'WU_WANG':
                        shear_formula = 'Wu and Wang (1999)'
                    elif line_data[1] == 'MANUAL':
                        shear_formula = 'User specified'
                    self.sediment_size_values['critical_shear_method'][-1] = shear_formula
            elif card == 'CRITICAL_SHEAR':
                if len(line_data) > 1:
                    self.sediment_size_values['critical_shear_stress'][-1] = float(line_data[1])
            elif card == 'SEDIMENT_SIZE_CLASS_END':
                self.reading_sediment_size = False
            else:
                self._add_advanced_card('SEDIMENT_SIZE_CLASS', line_data)

        # handle the simplified sediment case of a single layer with a single d50 AKZ TESTING THIS
        elif card == 'D50_DATASET' and not self.reading_bed_layer and len(self.grid_filename) > 1:
            # and len(self.bed_layer_values['d50'] == 0):
            grid_file = os.path.basename(self.grid_filename)
            msg = ''
            tokens = safe_shlex_split(card_line_orig, msg)
            modified_card_line = tokens[0] + ' "' + grid_file + '" ' + tokens[1]
            # add a bed layer
            self.add_bed_layer()
            # add the d50
            self.add_bed_layer_dataset(modified_card_line, 'd50')

        elif self.reading_bed_layer:
            # Bed layer composition cards for table.
            if card == 'LAYER':
                self.bed_layer_values['layer_id'][-1] = int(line_data[1])
            elif card == 'THICKNESS_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'layer_thickness')
                self.bed_layer_values['layer_thickness_type'][-1] = 'Dataset'
            elif card == 'THICKNESS_CONSTANT':
                self.bed_layer_values['layer_thickness_const'][-1] = float(line_data[1])
                self.bed_layer_values['layer_thickness_type'][-1] = 'Constant'
            elif card == 'D05_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd05')
            elif card == 'D10_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd10')
            elif card == 'D16_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd16')
            elif card == 'D20_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd20')
            elif card == 'D30_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd30')
            elif card == 'D35_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd35')
            elif card == 'D50_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd50')
            elif card == 'D65_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd65')
            elif card == 'D84_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd84')
            elif card == 'D90_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd90')
            elif card == 'D95_DATASET':
                self.add_bed_layer_dataset(card_line_orig, 'd95')
            elif card == 'BED_LAYER_END':
                self.reading_bed_layer = False
            else:
                self._add_advanced_card('BED_LAYER', line_data)

        # These are the old global tidal constituent blocks. If they exist, they will be assigned to all tidal BC
        # arcs. Later versions of CMS-Flow moved the tidal constituent specification to a per-arc basis.
        elif card == 'TIDAL_CONSTITUENTS_BEGIN':
            self.reading_global_constituents = True
        elif card == 'TIDAL_CONSTITUENTS_END':
            self.reading_global_constituents = False
        elif self.reading_global_constituents:
            constitiuent = line_data[0].upper().replace('TIDAL_CONSTITUENT_', '')
            if constitiuent not in SUPPORTED_CONSTITUENTS:
                constitiuent = SUPPORTED_CONSTITUENTS[0]  # default the value to something we recognize
            self.global_constituents['constituent'].append(constitiuent)
            self.global_constituents['amplitude'].append(float(line_data[1]))
            self.global_constituents['phase'].append(float(line_data[2]))

        elif self.reading_bc:
            if self.reading_harmonic:
                if card == 'CONSTITUENT':
                    self.harmonic_values[self.table_id]['speed'].append(float(line_data[1]))
                    self.harmonic_values[self.table_id]['amplitude'].append(float(line_data[2]))
                    self.harmonic_values[self.table_id]['phase'].append(float(line_data[3]))
                elif card == 'HARMONIC_END':
                    self.reading_harmonic = False
                else:
                    self._add_advanced_card('HARMONIC', line_data)
            elif self.reading_tidal:
                if card in ['CONSTITUENT', 'TIDAL_CONSTITUENT']:
                    constitiuent = line_data[1].upper()
                    if constitiuent not in SUPPORTED_CONSTITUENTS:
                        constitiuent = SUPPORTED_CONSTITUENTS[0]  # default the value to something we recognize
                    self.tidal_values[self.table_id]['constituent'].append(constitiuent)
                    self.tidal_values[self.table_id]['amplitude'].append(float(line_data[2]))
                    self.tidal_values[self.table_id]['phase'].append(float(line_data[3]))
                elif card == 'TIDAL_END':
                    self.reading_tidal = False
                else:
                    self._add_advanced_card('TIDAL', line_data)
            elif self.reading_tidal_db:
                if card == 'CONSTITUENTS':
                    self.tidal_db_cons = line_data[1:]
                elif card in ['NAME', 'DATABASE_NAME']:
                    msg = f'Invalid tidal database name on line {self.lineno}, defaulting to the ADCIRC2015 ' \
                          f'database: {card_line_orig}'
                    tokens = safe_shlex_split(card_line_orig, msg)
                    name = clean_token(tokens[1]).lower() if tokens else ''
                    if name.find('adcirc') != -1 or name.find('enpac') != -1 or name.find('ec2001') != -1:
                        self.tidal_db_idx = ADCIRC_INDEX
                        self.tidal_db_type = 'adcirc2015'
                    elif name.find('leprovost') != -1:
                        self.tidal_db_idx = LEPROVOST_INDEX
                        self.tidal_db_type = 'leprovost'
                    elif name.find('tpxo8') != -1:
                        self.tidal_db_idx = TPX08_INDEX
                        self.tidal_db_type = 'tpxo8'
                    elif name.find('tpxo9') != -1:
                        self.tidal_db_idx = TPX09_INDEX
                        self.tidal_db_type = 'tpxo9'
                    elif name.find('fes') != -1:
                        self.tidal_db_idx = FES2014_INDEX
                        self.tidal_db_type = 'fes2014'
                    else:
                        self.tidal_db_idx = ADCIRC_INDEX  # default to ADCIRC, long list of constituents it supports
                        self.tidal_db_type = 'adcirc2015'
                elif card == 'TIDAL_DATABASE_END':
                    self.reading_tidal_db = False
                    self._add_tidal_component()
            elif card == 'NAME':
                msg = f'Invalid tidal BC name on line {self.lineno}: {card_line_orig}'
                tokens = safe_shlex_split(card_line_orig, msg)
                self.bc_arc_values['name'][-1] = clean_token(tokens[1]) if tokens else ''
            elif card == 'CELLSTRING':
                msg = f'Unable to parse BC arc id on line {self.lineno}: {card_line_orig}'
                tokens = safe_shlex_split(card_line_orig, msg, min_tokens=3)
                self.bc_arc_comp_ids[-1] = self.parse_id_from_h5_path(tokens[2]) if tokens else -1
                if len(self.bc_filename) == 0:
                    self.bc_filename = self.extract_filename(card_line_orig)
            elif card == 'FLUX_BEGIN':
                self.in_bc_flux = True
                self.in_flux_block = True
                self.bc_arc_types[-1] = BC_FLUX
            elif card == 'FLUX_END':
                self.in_flux_block = False
            elif card == 'FLUX_VALUE' or (card == 'VALUE' and self.in_bc_flux):
                self.bc_arc_values['bc_type'][-1] = 'Flow rate-forcing'
                self.bc_arc_values['flow_source'][-1] = 'Constant'
                self.bc_arc_values['constant_flow'][-1] = float(line_data[1])
            elif card == 'FLUX_CURVE' or (card == 'CURVE' and self.in_bc_flux):
                self.bc_arc_values['bc_type'][-1] = 'Flow rate-forcing'
                self.bc_arc_values['flow_source'][-1] = 'Curve'
            elif card == 'DIRECTION' or card == 'INFLOW_DIRECTION':
                self.bc_arc_values['specify_inflow_direction'][-1] = 1
                self.bc_arc_values['flow_direction'][-1] = float(line_data[1])
            elif card == 'CONVEYANCE':
                self.bc_arc_values['flow_conveyance'][-1] = float(line_data[1])
            elif card == 'WSE_BEGIN':
                self.in_bc_flux = False
                self.in_wse_block = True
                self.bc_arc_types[-1] = BC_WSE
            elif card == 'WSE_END':
                self.in_wse_block = False
            elif card == 'WSE_VALUE' or (card == 'VALUE' and not self.in_bc_flux):
                self.bc_arc_values['bc_type'][-1] = 'WSE-forcing'
                self.bc_arc_values['wse_source'][-1] = 'Constant'
                self.bc_arc_values['wse_const'][-1] = float(line_data[1])
            elif card == 'WSE_CURVE' or (card == 'CURVE' and not self.in_bc_flux):
                self.bc_arc_values['bc_type'][-1] = 'WSE-forcing'
                self.bc_arc_values['wse_source'][-1] = 'Curve'
            elif card == 'OFFSET_CONSTANT':
                self.bc_arc_values['wse_offset_type'][-1] = 'Constant'
                self.bc_arc_values['wse_offset_const'][-1] = float(line_data[1])
            elif card == 'OFFSET_CURVE':
                self.bc_arc_values['wse_offset_type'][-1] = 'Curve'
            elif card == 'CONTROL_FILE':
                self.bc_arc_values['bc_type'][-1] = 'WSE-forcing'
                self.bc_arc_values['wse_source'][-1] = 'Parent CMS-Flow'
                self.bc_arc_values['parent_cmsflow'][-1] = self.extract_filename(card_line_orig)
            elif card == 'GRID_FILE':
                self.bc_arc_values['bc_type'][-1] = 'WSE-forcing'
                self.bc_arc_values['wse_source'][-1] = 'Parent ADCIRC'
                self.bc_arc_values['parent_adcirc_14'][-1] = self.extract_filename(card_line_orig)
            elif card == 'OUTPUT_FILE':
                self.bc_arc_values['parent_adcirc_63'][-1] = self.extract_filename(card_line_orig)
            elif card == 'WSE_FILE':
                msg = f'Unable to resolve path to WSE_FILE on line {self.lineno}: {card_line_orig}'
                tokens = safe_shlex_split(card_line_orig, msg)
                if len(tokens) > 2:
                    self.bc_arc_values['parent_adcirc_solution_wse'][-1] = self.normalize_path(tokens[1])
                    self.bc_arc_values['parent_adcirc_wse_path'][-1] = clean_token(tokens[2])
                    self.bc_arc_values['parent_adcirc_solution_type'][-1] = 'XMDF'
                elif tokens:  # Will be empty list if parsing error
                    self.bc_arc_values['parent_adcirc_63'][-1] = self.normalize_path(tokens[1])
                    self.bc_arc_values['parent_adcirc_solution_type'][-1] = 'ASCII'
            elif card == 'VEL_FILE':
                self.bc_arc_values['use_velocity'][-1] = 1
                msg = f'Unable to resolve path to VEL_FILE on line {self.lineno}: {card_line_orig}'
                tokens = safe_shlex_split(card_line_orig, msg)
                if len(tokens) > 2:
                    self.bc_arc_values['parent_adcirc_solution'][-1] = self.normalize_path(tokens[1])
                    self.bc_arc_values['parent_adcirc_velocity_path'][-1] = clean_token(tokens[2])
                elif tokens:  # Will be empty list if parsing error
                    self.bc_arc_values['parent_adcirc_64'][-1] = self.normalize_path(tokens[1])
            elif card == 'STARTING_DATE_TIME':
                year = int(line_data[1][:4])
                month = int(line_data[1][5:7])
                day = int(line_data[1][8:10])
                hour = int(line_data[2][:2])
                minute = int(line_data[2][3:5])
                second = int(line_data[2][6:8])
                py_dt = datetime.datetime(year, month, day, hour, minute, second)
                self.bc_arc_values['parent_adcirc_start_time'][-1] = py_dt.strftime('%Y-%m-%dT%H:%M:%S')
            elif card == 'TIDAL_BEGIN':
                self.bc_arc_values['bc_type'][-1] = 'WSE-forcing'
                self.bc_arc_values['wse_source'][-1] = 'Tidal constituent'
                self.reading_tidal = True
            elif card == 'HARMONIC_BEGIN':
                self.bc_arc_values['bc_type'][-1] = 'WSE-forcing'
                self.bc_arc_values['wse_source'][-1] = 'Harmonic'
                self.reading_harmonic = True
            elif card == 'INCIDENT_DIRECTION':
                pass
                # ['edtDirection'] = float(line_data[1])
            elif card == 'TIDAL_DATABASE_BEGIN':
                self.bc_arc_values['bc_type'][-1] = 'WSE-forcing'
                self.bc_arc_values['wse_source'][-1] = 'External tidal'
                self.reading_tidal_db = True
            elif card == 'BOUNDARY_END':
                self.reading_bc = False
            elif self.in_flux_block:
                self._add_advanced_card('FLUX', line_data)
            elif self.in_wse_block:
                self._add_advanced_card('WSE', line_data)
            else:
                self._add_advanced_card('BOUNDARY', line_data)
        elif card == 'WEIR_STRUCT_BEGIN':
            self.reading_weir = True
            self.current_structure.arc_parameters.clear_values()
            self.current_structure.arc_parameters.group('weir').is_active = True
        elif self.reading_weir:
            weir_group = self.current_structure.arc_parameters.group('weir')
            if card == 'CELL_IDS':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read weir cell IDs', min_tokens=2)
                ids = [int(token) - 1 for token in tokens[2:]]  # Discard the card name and ID count, make 0-based
                self.weirs.append(ids)
            elif card == 'DISTRIBUTION_COEFFICIENT':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read weir distribution coefficient', min_tokens=2)
                coefficient = float(tokens[1])
                weir_group.parameter('distribution_coeff').value = coefficient
            elif card == 'ORIENTATION_SEA':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read orientation sea', min_tokens=2)
                orientation = clean_token(tokens[1])
                parameter = weir_group.parameter('orientation')
                parameter_from_file(self.lineno, orientation, parameter)
            elif card == 'CREST_TYPE':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read crest type', min_tokens=2)
                crest_type = tokens[1].strip("'")
                parameter = weir_group.parameter('weir_type')
                parameter_from_file(self.lineno, crest_type, parameter)
            elif card == 'FLOW_COEFFICIENT_FROM_BAY':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read crest type', min_tokens=2)
                coefficient = float(tokens[1])
                parameter = weir_group.parameter('flow_coeff_bay_to_sea')
                parameter.value = coefficient
            elif card == 'FLOW_COEFFICIENT_FROM_SEA':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read crest type', min_tokens=2)
                coefficient = float(tokens[1])
                parameter = weir_group.parameter('flow_coeff_sea_to_bay')
                parameter.value = coefficient
            elif card == 'CREST_ELEVATION':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read crest type', min_tokens=2)
                elevation = float(tokens[1])
                weir_group.parameter('crest_elevation').value = elevation
            elif card == 'METHOD':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read crest type', min_tokens=2)
                parameter = weir_group.parameter('calc_method')
                parameter_from_file(self.lineno, tokens[1], parameter)
            elif card == 'WEIR_STRUCT_END':
                self.weir_values.append(self.current_structure.arc_parameters.extract_values())
                self.reading_weir = False
        # This has to be the first card before the rubble mound blocks
        elif card == 'RUBBLE_MOUND_ID_DATASET':  # New style IDs
            tokens = safe_shlex_split(card_line_orig, 'Unable to read rubble mound cell IDs.', min_tokens=3)
            filename = os.path.join(os.path.dirname(self.filename), clean_token(tokens[1]))
            with h5py.File(filename) as f:
                path = os.path.join(clean_token(tokens[2]), 'Values').replace('\\', '/')
                # f[path] comes in as a table with one row, containing one column for each cell, and all floats.
                # Pull out the one row to be a column, and convert to int.
                self.rubble_mound_cells = f[path][0].astype(int)
        elif self.reading_rubble_mound:
            if card == 'NAME':
                msg = f'Unable to read rubble mound polygon name on line {self.lineno}: {card_line_orig}'
                tokens = safe_shlex_split(card_line_orig, msg)
                self.rubble_mound_values['name'][-1] = clean_token(tokens[1]) if tokens else ''
            elif card == 'CELL_IDS':  # Old style IDs
                cell_indexes = [int(item) - 1 for item in line_data[2:]]  # Skip card and length, make 0-based
                needed_length = max(cell_indexes) + 1
                if self.rubble_mound_cells.size < needed_length:
                    append_length = needed_length - self.rubble_mound_cells.size
                    self.rubble_mound_cells = np.pad(self.rubble_mound_cells, (0, append_length))
                self.rubble_mound_cells[cell_indexes] = self.current_rubble_mound_id
                self.current_rubble_mound_id += 1
            elif card == 'ROCK_DIAMETER_CONSTANT':
                self.rubble_mound_values['rock_diameter'][-1] = float(line_data[1])
            elif card == 'STRUCTURE_POROSITY_CONSTANT':
                self.rubble_mound_values['porosity'][-1] = float(line_data[1])
            elif card == 'STRUCTURE_BASE_DEPTH_CONSTANT':
                self.rubble_mound_values['base_depth'][-1] = float(line_data[1])
            # Spatially varying dataset options
            elif card == 'ROCK_DIAMETER_DATASET':
                self.parse_dataset_reference(
                    full_line=card_line_orig,
                    dset_name='ROCK_DIAMETER_DATASET',
                    uuid_container=self.rubble_mound_values,
                    row=len(self.rubble_mound_values['name']) - 1
                )
                self.rubble_mound_values['rock_diameter_type'][-1] = 1
            elif card == 'STRUCTURE_POROSITY_DATASET':
                self.parse_dataset_reference(
                    full_line=card_line_orig,
                    dset_name='STRUCTURE_POROSITY_DATASET',
                    uuid_container=self.rubble_mound_values,
                    row=len(self.rubble_mound_values['name']) - 1
                )
                self.rubble_mound_values['porosity_type'][-1] = 1
            elif card == 'STRUCTURE_BASE_DEPTH_DATASET':
                self.parse_dataset_reference(
                    full_line=card_line_orig,
                    dset_name='STRUCTURE_BASE_DEPTH_DATASET',
                    uuid_container=self.rubble_mound_values,
                    row=len(self.rubble_mound_values['name']) - 1
                )
                self.rubble_mound_values['base_depth_type'][-1] = 1
            elif card == 'FORCHHEIMER_COEFF_METHOD':
                method_id = int(line_data[1])
                method = ['Sidiropoulou et al. (2007)', 'Kadlec and Knight (1996)', 'Ward (1964)'][method_id - 1]
                self.rubble_mound_values['calculation_method'][-1] = method
            elif card == 'RUBBLE_MOUND_END':
                self.reading_rubble_mound = False
            else:
                self._add_advanced_card('RUBBLE_MOUND', line_data)
        elif card == 'CULVERT_STRUCT_BEGIN':
            self.reading_culvert = True
            self.current_structure.arc_parameters.clear_values()
            self.current_structure.arc_parameters.group('culvert').is_active = True
        elif self.reading_culvert:
            culvert_group = self.current_structure.arc_parameters.group('culvert')
            if card == 'CELLS':
                cell_indexes = tuple(int(item) - 1 for item in line_data[1:])  # Skip card, make 0-based
                if len(cell_indexes) != 2:
                    raise RuntimeError(f'Line {self.lineno}: Wrong number of cell indexes')
                self.culverts.append(cell_indexes)
            elif card == 'TYPE':
                msg = 'Wrong number of values for TYPE card'
                tokens = safe_shlex_split(card_line_orig, msg)
                shape = clean_token(tokens[1])
                parameter = culvert_group.parameter('culvert_type')
                parameter_from_file(self.lineno, shape, parameter)
            elif card == 'FLAP_GATE':
                msg = 'Wrong number of values for FLAP_GATE card'
                tokens = safe_shlex_split(card_line_orig, msg)
                state = clean_token(tokens[1])
                parameter = culvert_group.parameter('flap_gate')
                parameter_from_file(self.lineno, state, parameter)
            elif card == 'RADIUS':
                value = float(line_data[1])
                culvert_group.parameter('radius').value = value
            elif card == 'WIDTH':
                value = float(line_data[1])
                culvert_group.parameter('width').value = value
            elif card == 'HEIGHT':
                value = float(line_data[1])
                culvert_group.parameter('height').value = value
            elif card == 'LENGTH':
                value = float(line_data[1])
                culvert_group.parameter('length').value = value
            elif card == 'DARCY_FRICTION_FACTOR':
                value = float(line_data[1])
                culvert_group.parameter('darcy_friction').value = value
            elif card == 'MANNINGS_COEFFICIENT':
                value = float(line_data[1])
                culvert_group.parameter('manning_n').value = value
            elif card == 'INVERT_ELEVATIONS':
                bay, sea = int(line_data[1]), int(line_data[2])
                XmLog().instance.warning(f'Ignored card: INVERT_ELEVATIONS, with values {bay} {sea}')
            elif card == 'ENTRY_HEAD_LOSS_BAY':
                value = float(line_data[1])
                culvert_group.parameter('entry_head_loss_bay').value = value
            elif card == 'ENTRY_HEAD_LOSS_SEA':
                value = float(line_data[1])
                culvert_group.parameter('entry_head_loss_sea').value = value
            elif card == 'EXIT_HEAD_LOSS_BAY':
                value = float(line_data[1])
                culvert_group.parameter('exit_head_loss_bay').value = value
            elif card == 'EXIT_HEAD_LOSS_SEA':
                value = float(line_data[1])
                culvert_group.parameter('exit_head_loss_sea').value = value
            elif card == 'OUTFLOW_ANGLES':
                # The value for this card depends on the grid and arc geometries. We'll recompute it on export to save
                # the user the trouble of specifying it manually, so there's no need to preserve it on read.
                pass
            elif card == 'CULVERT_STRUCT_END':
                self.reading_culvert = False
                self.culvert_values.append(self.current_structure.arc_parameters.extract_values())
        elif card == 'TIDEGATE_STRUCT_BEGIN':
            self.reading_tide_gate = True
            self.current_structure.arc_parameters.clear_values()
            self.current_structure.arc_parameters.group('tide_gate').is_active = True
        elif self.reading_tide_gate:
            gate_group = self.current_structure.arc_parameters.group('tide_gate')
            if card == 'CELL_IDS':
                cell_indexes = [int(item) - 1 for item in line_data[2:]]  # Skip card and length, make 0-based
                self.tide_gates.append(cell_indexes)
            elif card == 'DISTRIBUTION_COEFFICIENT':
                value = float(line_data[1])
                gate_group.parameter('distribution_coefficient').value = value
            elif card == 'ORIENTATION_SEA':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read orientation sea', min_tokens=2)
                orientation = clean_token(tokens[1])
                mapping = {'NORTH': 'North', 'SOUTH': 'South', 'EAST': 'East', 'WEST': 'West'}
                orientation = mapping[orientation]
                gate_group.parameter('orientation_sea').value = orientation
            elif card == 'FLOW_COEFF_FROM_BAY':
                coefficient = float(line_data[1])
                gate_group.parameter('flow_coeff_from_bay').value = coefficient
            elif card == 'FLOW_COEFF_FROM_SEA':
                coefficient = float(line_data[1])
                gate_group.parameter('flow_coeff_from_sea').value = coefficient
            elif card == 'OPENING_HEIGHT':
                height = float(line_data[1])
                gate_group.parameter('opening_height').value = height
            elif card == 'BOTTOM_ELEVATION':
                height = float(line_data[1])
                gate_group.parameter('bottom_elevation').value = height
            elif card == 'METHOD':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read approach method', min_tokens=2)
                index = int(tokens[1]) - 1
                parameter = gate_group.parameter('method')
                method = parameter.options[index]
                parameter.value = method
            elif card == 'SCHED_OPERATION_TYPE':
                tokens = safe_shlex_split(card_line_orig, 'Unable to read operation type', min_tokens=2)
                mapping = {
                    'REG': 'Regular time interval',
                    'DES': 'Designated times and durations',
                    'EBB': 'Open for ebb tide, close for flood',
                    'UCG': 'Uncontrolled',
                }
                label = mapping[tokens[1]]
                parameter = gate_group.parameter('schedule_type')
                parameter.value = label
            elif card == 'REG_START_TIME':
                start = int(line_data[1])
                gate_group.parameter('start_time').value = start
            elif card == 'REG_OPEN_FREQUENCY':
                frequency = int(line_data[1])
                gate_group.parameter('open_frequency').value = frequency
            elif card == 'REG_OPEN_DURATION':
                duration = int(line_data[1])
                gate_group.parameter('open_duration').value = duration
            elif card == 'NUM_DESIGNATED_TIMES':
                num_times = int(line_data[1])
                parameter = gate_group.parameter('designated_times')
                table = parameter.value
                while len(table) < num_times:
                    table.append([0, 0])
                parameter.value = table
            elif card == 'DES_START_TIME' or card == 'DES_OPEN_DURATION':
                column = 0 if card == 'DES_START_TIME' else 1
                parameter = gate_group.parameter('designated_times')
                table = parameter.value
                times = [int(time) for time in line_data[1:]]  # Skip card
                for index, time in enumerate(times):
                    table[index][column] = time
                parameter.value = table
            elif card == 'TIDEGATE_STRUCT_END':
                self.reading_tide_gate = False
                self.tide_gate_values.append(self.current_structure.arc_parameters.extract_values())
        elif self.reading_dredge_operation:
            if self.reading_dredge:
                if card == 'DEPTH_DATASET':
                    self.sim_data.dredge.attrs['ENABLE_DREDGE'] = 1
                    self.parse_dataset_reference(card_line_orig, 'DREDGE_DATASET', self.sim_data.dredge.attrs)
                elif card == 'START_METHOD':
                    msg = f'Insufficient arguments for start method found on line {self.lineno}: {card_line_orig}'
                    tokens = safe_shlex_split(card_line_orig, msg, 2)
                    method = clean_token(tokens[1]).upper()
                    if tokens:
                        if 'SHALLOW' in method:
                            self.sim_data.dredge.attrs['DREDGE_METHOD'] = 'Shallowest cell'
                        elif 'CELL' in method:  # Older versions may have simply 'CELL' instead of 'SPECIFIED CELL'
                            self.sim_data.dredge.attrs['DREDGE_METHOD'] = 'Specified cell'
                        else:
                            XmLog().instance.error(f"Unrecognized dredge method: {method}")
                elif card == 'START_CELL':
                    self.sim_data.dredge.attrs['SPECIFIED_CELL'] = int(line_data[1])
                elif card == 'DREDGE_RATE':
                    self.sim_data.dredge.attrs['DREDGE_RATE_VALUE'] = float(line_data[1])
                    if len(line_data) > 2:
                        self.sim_data.dredge.attrs['DREDGE_RATE_UNITS'] = clean_token(line_data[2])
                elif card == 'TRIGGER_METHOD':
                    msg = f'Insufficient arguments for trigger method found on line {self.lineno}: {card_line_orig}'
                    tokens = safe_shlex_split(card_line_orig, msg, 2)
                    method = clean_token(tokens[1]).upper()
                    if tokens:
                        if 'DEPTH' in method:
                            self.sim_data.dredge.attrs['TRIGGER_METHOD'] = 'Depth'
                        elif 'VOLUME' in method:
                            self.sim_data.dredge.attrs['TRIGGER_METHOD'] = 'Volume'
                        elif 'PERCENT' in method:
                            self.sim_data.dredge.attrs['TRIGGER_METHOD'] = 'Percent'
                        elif 'TIME' in method:
                            self.sim_data.dredge.attrs['TRIGGER_METHOD'] = 'Time periods'
                        else:
                            XmLog().instance.error(f"Unrecognized trigger method: {method}")
                elif card == 'TRIGGER_DEPTH':
                    self.sim_data.dredge.attrs['TRIGGER_DEPTH_VALUE'] = float(line_data[1])
                    if len(line_data) > 2:
                        self.sim_data.dredge.attrs['TRIGGER_DEPTH_UNITS'] = clean_token(line_data[2])
                    self.sim_data.dredge.attrs['TRIGGER_PERCENT_DEPTH_VALUE'] = float(line_data[1])
                    if len(line_data) > 2:
                        self.sim_data.dredge.attrs['TRIGGER_PERCENT_DEPTH_UNITS'] = clean_token(line_data[2])
                elif card == 'TRIGGER_VOLUME':
                    self.sim_data.dredge.attrs['TRIGGER_VOLUME_VALUE'] = float(line_data[1])
                    if len(line_data) > 2:
                        self.sim_data.dredge.attrs['TRIGGER_VOLUME_UNITS'] = clean_token(line_data[2])
                elif card == 'TRIGGER_PERCENT':
                    self.sim_data.dredge.attrs['TRIGGER_PERCENT'] = float(line_data[1])
                elif card == 'TRIGGER_TIME_PERIODS':
                    starts = [float(size) for size in line_data[2:-1:2]]
                    finishes = [float(size) for size in line_data[3:-1:2]]
                    num_periods = int(line_data[1])
                    if len(starts) == num_periods and len(finishes) == num_periods:
                        dredge_time_periods_table = {
                            'start': xr.DataArray(data=np.array(starts, dtype=float)),
                            'finish': xr.DataArray(data=np.array(finishes, dtype=float))
                        }
                        self.sim_data.dredge_time_periods = xr.Dataset(data_vars=dredge_time_periods_table)
                elif card == 'DISTRIBUTION':
                    msg = f'Insufficient arguments for distribution found on line {self.lineno}: {card_line_orig}'
                    tokens = safe_shlex_split(card_line_orig, msg, 2)
                    method = clean_token(tokens[1]).upper()
                    if tokens:
                        if 'PERCENT' in method:
                            self.sim_data.dredge.attrs['DISTRIBUTION'] = 'Percent'
                        elif 'SEQUENTIAL' in method:
                            self.sim_data.dredge.attrs['DISTRIBUTION'] = 'Sequential'
                        else:
                            XmLog().instance.error(f"Unrecognized distribution: {method}")
                elif card == 'DREDGE_END':
                    self.reading_dredge = False
                else:
                    self._add_advanced_card('DREDGE', line_data)
            elif self.reading_placement:
                if card == 'DISTRIBUTION_PERCENTAGE':
                    percentage = f'PLACEMENT_{self.placement_number}_PERCENTAGE'
                    self.sim_data.dredge_placement.attrs[percentage] = float(line_data[1])
                elif card == 'AREA_DATASET' or 'AREA' in card:
                    dataset = f'PLACEMENT_{self.placement_number}_DATASET'
                    self.parse_dataset_reference(card_line_orig, dataset, self.sim_data.dredge_placement.attrs)
                elif card == 'PLACEMENT_METHOD':
                    key = f'PLACEMENT_{self.placement_number}_METHOD'
                    msg = (
                        f'Insufficient arguments for placement method value found on line '
                        f'{self.lineno}: {card_line_orig}'
                    )
                    tokens = safe_shlex_split(card_line_orig, msg, 2)
                    method = clean_token(tokens[1]).upper()
                    if tokens:
                        if 'UNIFORM' in method:
                            self.sim_data.dredge_placement.attrs[key] = 'Uniform'
                        elif 'CELL' in method:  # Older versions may have simply 'CELL' instead of 'SPECIFIED CELL'
                            self.sim_data.dredge_placement.attrs[key] = 'Specified cell'
                        else:
                            XmLog().instance.error(f"Unrecognized placement method: {method}")
                elif card == 'START_CELL':
                    method_cell = f"PLACEMENT_{self.placement_number}_METHOD_CELL"
                    self.sim_data.dredge_placement.attrs[method_cell] = int(line_data[1])
                elif card == 'DEPTH_LIMIT':
                    msg = f'Unable to parse depth limit on line {self.lineno}: {card_line_orig}'
                    tokens = safe_shlex_split(card_line_orig, msg, min_tokens=2)
                    if tokens:
                        limit_method = f"PLACEMENT_{self.placement_number}_LIMIT_METHOD"
                        limit_depth_value = f"PLACEMENT_{self.placement_number}_LIMIT_DEPTH_VALUE"
                        limit_depth_units = f"PLACEMENT_{self.placement_number}_LIMIT_DEPTH_UNITS"
                        self.sim_data.dredge_placement.attrs[limit_method] = 'Depth'
                        self.sim_data.dredge_placement.attrs[limit_depth_value] = float(line_data[1])
                        self.sim_data.dredge_placement.attrs[limit_depth_units] = clean_token(line_data[2]) \
                            if len(tokens) > 2 else 'm'
                elif card == 'THICKNESS_LIMIT':
                    msg = f'Unable to parse thickness limit on line {self.lineno}: {card_line_orig}'
                    tokens = safe_shlex_split(card_line_orig, msg, min_tokens=2)
                    if tokens:
                        limit_method = f"PLACEMENT_{self.placement_number}_LIMIT_METHOD"
                        limit_thickness_value = f"PLACEMENT_{self.placement_number}_LIMIT_THICKNESS_VALUE"
                        limit_thickness_units = f"PLACEMENT_{self.placement_number}_LIMIT_THICKNESS_UNITS"
                        self.sim_data.dredge_placement.attrs[limit_method] = 'Thickness'
                        self.sim_data.dredge_placement.attrs[limit_thickness_value] = float(line_data[1])
                        self.sim_data.dredge_placement.attrs[limit_thickness_units] = clean_token(line_data[2]) \
                            if len(tokens) > 2 else 'm'
                elif card == 'PLACEMENT_END':
                    self.reading_placement = False
                else:
                    self._add_advanced_card('PLACEMENT', line_data)
            elif card == 'NAME':
                msg = f'Unable to parse dredge placement name on line {self.lineno}: {card_line_orig}'
                tokens = safe_shlex_split(card_line_orig, msg)
                self.sim_data.dredge.attrs['DREDGE_NAME'] = clean_token(tokens[1]) if tokens else ''
            elif card == 'DREDGE_BEGIN':
                self.reading_dredge = True
            elif card == 'PLACEMENT_BEGIN':
                self.reading_placement = True
                self.placement_number += 1
                enable_placement = f"DEFINE_PLACEMENT_{self.placement_number}"
                self.sim_data.dredge_placement.attrs[enable_placement] = 1
            elif card == 'DREDGE_OPERATION_END':
                self.reading_dredge_operation = False
            else:
                self._add_advanced_card('DREDGE_OPERATION', line_data)

        # General parameters
        elif card == 'CMS_VERSION':
            self.cms_version = [int(val) for val in line_data[1].split('.')]
        elif card == 'AVERAGE_LATITUDE':
            self.sim_data.flow.attrs['LATITUDE_CORIOLIS'] = 'User specified'
            self.sim_data.flow.attrs['DEGREES'] = float(line_data[1])
        elif card == 'TELESCOPING':
            self.telescoping_filename = self.extract_filename(card_line_orig)
        elif card == 'CELL_LATITUDES' or card == 'CELL_LONGITUDES':
            # Users can't edit these, and we generate them from cell locations and the projection on export, but in case
            # the user changed the name of the .cmcards file and kept the rest intact, extract real _mp.h5 filename.
            if len(self.bc_filename) == 0:
                self.bc_filename = self.extract_filename(card_line_orig)
        elif card == 'GRID_FILE':
            self.grid_filename = self.extract_filename(card_line_orig)
        elif card == 'BATHYMETRY_DATASET':
            msg = f'Unable to parse bathymetry dataset reference on line {self.lineno}: {card_line_orig}'
            tokens = safe_shlex_split(card_line_orig, msg)
            if len(tokens) > 2:
                self.bathymetry = (self.extract_filename(tokens[1], pos=0), f'{clean_token(tokens[2])}/Values')
        elif card == 'GRID_CELL_TYPES':
            msg = f'Unable to parse grid cell type dataset reference on line {self.lineno}: {card_line_orig}'
            tokens = safe_shlex_split(card_line_orig, msg)
            if len(tokens) > 1:
                self.cell_activity_group = clean_token(tokens[1])
        elif card == 'GRID_ANGLE':
            self.grid_angle = float(line_data[1])
        elif card == 'USE_WALL_FRICTION_TERMS':
            self.sim_data.flow.attrs['WALL_FRICTION'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'WAVE_MASS_FLUX':
            self.sim_data.flow.attrs['WAVE_FLUXES'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'ROLLER_MASS_FLUX':
            self.sim_data.flow.attrs['ROLLER_FLUXES'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'DRYING_DEPTH':
            self.sim_data.flow.attrs['WETTING_DEPTH'] = float(line_data[1])
        elif card == 'BED_SLOPE_FRICTION_FACTOR':
            self.sim_data.flow.attrs['BED_SLOPE_FRIC_COEFFICIENT'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'WAVE_CURRENT_MEAN_STRESS':
            display_text = ''
            if line_data[1] == 'QUAD':
                display_text = 'Quadratic'
            elif line_data[1] == 'DATA2':
                display_text = 'Soulsby (1995) Data2'
            elif line_data[1] == 'DATA13':
                display_text = 'Soulsby (1995) Data13'
            elif line_data[1] == 'F84':
                display_text = 'Fredsoe (1984)'
            elif line_data[1] == 'HT91':
                display_text = 'Huynh-Thanh and Termperville (1991)'
            self.sim_data.flow.attrs['WAVE_CURRENT_BOTTOM_FRIC_COEFFICIENT'] = display_text
        elif card == 'WAVE_BOTTOM_FRICTION_COEFFICIENT':
            self.sim_data.flow.attrs['QUAD_WAVE_BOTTOM_COEFFICIENT'] = float(line_data[1])
        elif card == 'TURBULENCE_MODEL':
            display_text = ''
            if line_data[1] == 'SUBGRID':
                display_text = 'Subgrid'
            elif line_data[1] == 'FALCONER':
                display_text = 'Falconer'
            elif line_data[1] == 'PARABOLIC':
                display_text = 'Parabolic'
            elif line_data[1] == 'MIXING_LENGTH':
                display_text = 'Mixing length'
            self.sim_data.flow.attrs['TURBULENCE_MODEL'] = display_text
        elif card == 'EDDY_VISCOSITY_BOTTOM':
            self.sim_data.flow.attrs['CURRENT_BOTTOM_COEFFICIENT'] = float(line_data[1])
            self.sim_data.flow.attrs['TURBULENCE_PARAMETERS'] = 1
        elif card == 'EDDY_VISCOSITY_HORIZONTAL':
            self.sim_data.flow.attrs['CURRENT_HORIZONTAL_COEFFICIENT'] = float(line_data[1])
        elif card == 'EDDY_VISCOSITY_CONSTANT':
            self.sim_data.flow.attrs['BASE_VALUE'] = float(line_data[1])
        elif card == 'EDDY_VISCOSITY_WAVE':
            self.sim_data.flow.attrs['WAVE_BOTTOM_COEFFICIENT'] = float(line_data[1])
        elif card == 'EDDY_VISCOSITY_BREAKING':
            self.sim_data.flow.attrs['WAVE_BREAKING_COEFFICIENT'] = float(line_data[1])
        elif card == 'SIMULATION_LABEL':
            msg = f'Unable to parse simulation label on line {self.lineno}: {card_line_orig}'
            tokens = safe_shlex_split(card_line_orig, msg)
            self.sim_data.output.attrs['SIMULATION_LABEL'] = clean_token(tokens[1]) if tokens else ''
        elif card == 'NUM_THREADS':
            self.sim_data.general.attrs['NUM_THREADS'] = int(line_data[1])
        # Timing parameters
        elif card == 'HYDRO_TIMESTEP':  # Need to read the solution scheme card to know the widget name
            self.sim_data.flow.attrs['HYDRO_TIME_STEP_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.flow.attrs['HYDRO_TIME_STEP_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'STARTING_JDATE':
            # First two digits are the year
            year = int(line_data[1][:2])
            if year > 20:
                year += 1900
            else:
                year += 2000

            days_in_year = int(line_data[1][2:])
            py_dt = datetime.datetime(year, 1, 1) + datetime.timedelta(days_in_year - 1)
            self.dependent_widgets['STARTING_JDATE'] = py_dt
        elif card == 'STARTING_JDATE_HOUR':
            hours = int(line_data[1])
            self.dependent_widgets['STARTING_JDATE_HOUR'] = hours
        elif card == 'STARTING_DATE_TIME':
            datetime_value = line_data[1] + ' ' + line_data[2]
            py_dt = datetime.datetime.strptime(datetime_value, '%Y-%m-%d %H:%M:%S')
            self.sim_data.general.attrs['DATE_START'] = py_dt.strftime('%Y-%m-%dT%H:%M:%S')
        elif card == 'DURATION_RUN':  # Need to read the solution scheme card to know the widget name
            self.sim_data.general.attrs['SIM_DURATION_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.general.attrs['SIM_DURATION_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'DURATION_RAMP':  # Need to read the solution scheme card to know the widget name
            self.sim_data.general.attrs['RAMP_DURATION_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.general.attrs['RAMP_DURATION_UNITS'] = card_to_time_unit(line_data[2])
        # Hot start parameters
        elif card == 'HOT_START_TIME':
            self.sim_data.general.attrs['HOT_WRITE_OUT_DURATION_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.general.attrs['HOT_WRITE_OUT_DURATION_UNITS'] = card_to_time_unit(line_data[2])
            self.dependent_widgets['HOT_START_TIME'] = True
        elif card in ['INITIAL_STARTUP_FILE', 'INITIAL_CONDITION_FILE']:
            file_path = self.extract_filename(card_line_orig)
            self.sim_data.general.attrs['INIT_CONDITIONS_FILE'] = file_path
            self.sim_data.general.attrs['USE_INIT_CONDITIONS_FILE'] = 1
        elif card == 'AUTO_HOT_START_INTERVAL':
            self.sim_data.general.attrs['AUTO_HOT_DURATION_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.general.attrs['AUTO_HOT_DURATION_UNITS'] = card_to_time_unit(line_data[2])
            self.dependent_widgets['AUTO_HOT_START_INTERVAL'] = True
        # Transport parameters
        elif card == 'CALC_SALINITY':
            self.sim_data.salinity.attrs['CALCULATE_SALINITY'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'SALINITY_IC':
            self.sim_data.salinity.attrs['GLOBAL_CONCENTRATION'] = float(line_data[1])
            self.sim_data.salinity.attrs['SALINITY_CONCENTRATION'] = 'Global concentration'
        elif card == 'SALINITY_CALC_INTERVAL':
            self.sim_data.salinity.attrs['SALINITY_TRANSPORT_RATE_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.salinity.attrs['SALINITY_TRANSPORT_RATE_UNITS'] = card_to_time_unit(line_data[2])
            self.sim_data.salinity.attrs['SALINITY_CONCENTRATION'] = 'Global concentration'
        elif card == 'SALINITY_IC_DATASET':
            self.sim_data.salinity.attrs['SALINITY_CONCENTRATION'] = 'Spatially varied'
            self.parse_dataset_reference(card_line_orig, 'SALINITY_INITIAL_CONCENTRATION', self.sim_data.salinity.attrs)
        elif card == 'WATER_DENSITY':
            self.sim_data.salinity.attrs['WATER_DENSITY'] = float(line_data[1])
        elif card == 'WATER_TEMPERATURE':
            self.sim_data.salinity.attrs['WATER_TEMP'] = float(line_data[1])
        elif card == 'CALC_TEMPERATURE':
            self.sim_data.salinity.attrs['CALCULATE_TEMPERATURE'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'TEMPERATURE_CALC_INTERVAL':
            self.sim_data.salinity.attrs['TEMPERATURE_TRANSPORT_RATE_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.salinity.attrs['TEMPERATURE_TRANSPORT_RATE_UNITS'] = card_to_time_unit(line_data[2])
            self.sim_data.salinity.attrs['INITIAL_TEMPERATURE_TYPE'] = 'Constant water temperature'
        elif card == 'TEMPERATURE_IC_DATASET':
            self.sim_data.salinity.attrs['INITIAL_TEMPERATURE_TYPE'] = 'Spatially varied'
            self.parse_dataset_reference(card_line_orig, 'INITIAL_TEMPERATURE_DATASET', self.sim_data.salinity.attrs)
        elif card == 'SEDIMENT_SIZE_CLASS_NUMBER':
            self.sim_data.sediment.attrs['MULTIPLE_GRAIN_SIZES'] = 'Specify number of size classes only'
            self.sim_data.sediment.attrs['SIMPLE_MULTI_SIZE'] = int(line_data[1])
        elif card == 'MULTIPLE_GRAIN_SIZES':
            self.sim_data.sediment.attrs['MULTIPLE_GRAIN_SIZES'] = 'Enter grain size values to use'
            sizes = [float(size) for size in line_data[2:]]
            simple_grain_sizes_table = {'grain_size': xr.DataArray(data=np.array(sizes, dtype=float))}
            self.sim_data.simple_grain_sizes_table = xr.Dataset(data_vars=simple_grain_sizes_table)
        elif card == 'SEDIMENT_STANDARD_DEVIATION':
            self.sim_data.sediment.attrs['SEDIMENT_STANDARD_DEVIATION'] = float(line_data[1])
            self.sim_data.sediment.attrs['ENABLE_SIMPLIFIED_MULTI_GRAIN_SIZE'] = 1
        elif card == 'BED_COMPOSITION_INPUT':
            value = line_data[1]
            if value == 'D50_SIGMA':
                self.sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] = 'D50 Sigma'
            elif value == 'D16_D50_D84':
                self.sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] = 'D16 D50 D84'
            elif value == 'D35_D50_D90':
                self.sim_data.sediment.attrs['BED_COMPOSITION_INPUT'] = 'D35 D50 D90'
        elif card == 'MIXING_LAYER_CONSTANT_THICKNESS':
            self.sim_data.sediment.attrs['THICKNESS_FOR_MIXING'] = float(line_data[1])
        # Some old files have an extra 'S' on BED_LAYER. CMS handles it, but SMS adds it to advanced cards.
        elif card == 'BED_LAYER_CONSTANT_THICKNESS' or card == 'BED_LAYERS_CONSTANT_THICKNESS':
            self.sim_data.sediment.attrs['THICKNESS_FOR_BED'] = float(line_data[1])
        elif card == 'NUMBER_BED_LAYERS':
            self.sim_data.sediment.attrs['NUMBER_BED_LAYERS'] = int(line_data[1])
        elif card == 'HARDBOTTOM_DATASET':
            self.sim_data.sediment.attrs['USE_HARD_BOTTOM'] = 1
            self.parse_dataset_reference(card_line_orig, 'HARD_BOTTOM', self.sim_data.sediment.attrs)
        # Save Points
        elif card == 'SAVE_POINT_LABEL':
            msg = f'Unable to parse save points coverage name on line {self.lineno}: {card_line_orig}'
            tokens = safe_shlex_split(card_line_orig, msg)
            self.save_points_cov_name = clean_token(tokens[1]) if tokens else ''
        elif card == 'HYDRO_OUTPUT_INTERVAL':
            self.save_points_cov_values['HYDRO_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.save_points_cov_values['HYDRO_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'SEDIMENT_OUTPUT_INTERVAL':
            self.save_points_cov_values['SEDIMENT_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.save_points_cov_values['SEDIMENT_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'SALINITY_OUTPUT_INTERVAL':
            self.save_points_cov_values['SALINITY_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.save_points_cov_values['SALINITY_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'WAVE_OUTPUT_INTERVAL':
            self.save_points_cov_values['WAVES_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.save_points_cov_values['WAVES_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'SAVE_POINT_CARDS':
            self.process_spcards_file(clean_token(line_data[1]))
        elif card == 'SAVE_POINT':
            msg = f'Error parsing save point data on line {self.lineno}: {card_line_orig}'
            tokens = safe_shlex_split(card_line_orig, msg, min_tokens=5)
            if not tokens:
                return
            self.process_save_point_card(tokens)
        elif card == 'TIME_LIST_1':
            self.read_time_list(line_data, 1)
        elif card == 'TIME_LIST_2':
            self.read_time_list(line_data, 2)
        elif card == 'TIME_LIST_3':
            self.read_time_list(line_data, 3)
        elif card == 'TIME_LIST_4':
            self.read_time_list(line_data, 4)
        elif card == 'WSE_OUT_TIMES_LIST':
            display_text = f'List {line_data[1] if line_data[1] != "0" else "1"}'
            self.sim_data.output.attrs['WSE_LIST'] = display_text
        elif card == 'VEL_OUT_TIMES_LIST':
            display_text = f'List {line_data[1] if line_data[1] != "0" else "1"}'
            self.sim_data.output.attrs['CURRENT_VELOCITY_LIST'] = display_text
        elif card == 'MORPH_OUT_TIMES_LIST':
            if line_data[1] != '0':
                display_text = 'List ' + line_data[1]
                self.sim_data.output.attrs['MORPHOLOGY_LIST'] = display_text
                self.sim_data.output.attrs['USE_MORPHOLOGY'] = 1
        elif card == 'TRANS_OUT_TIMES_LIST':
            if line_data[1] != '0':
                display_text = 'List ' + line_data[1]
                self.sim_data.output.attrs['TRANSPORT_LIST'] = display_text
                self.sim_data.output.attrs['USE_TRANSPORT'] = 1
        elif card == 'WAVE_OUT_TIMES_LIST' or card == 'WAVES_OUT_TIMES_LIST':  # card was renamed
            if line_data[1] != '0':
                display_text = 'List ' + line_data[1]
                self.sim_data.output.attrs['WAVE_LIST'] = display_text
                self.sim_data.output.attrs['USE_WAVE'] = 1
        elif card == 'WIND_OUT_TIMES_LIST':
            if line_data[1] != '0':
                display_text = 'List ' + line_data[1]
                self.sim_data.output.attrs['WIND_LIST'] = display_text
                self.sim_data.output.attrs['USE_WIND'] = 1
        elif card == 'VISC_OUT_TIMES_LIST' or card == 'EDDY_VISCOSITY_OUT_TIMES_LIST':  # card was renamed
            if line_data[1] != '0':
                display_text = 'List ' + line_data[1]
                self.sim_data.output.attrs['EDDY_VISCOSITY_LIST'] = display_text
                self.sim_data.output.attrs['USE_EDDY_VISCOSITY'] = 1
        # Output
        elif card == 'VEL_MAG_OUTPUT':
            self.sim_data.output.attrs['CURRENT_MAGNITUDE'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'MORPH_OUTPUT':
            self.sim_data.output.attrs['MORPHOLOGY_CHANGE'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'CONC_OUTPUT':
            self.sim_data.output.attrs['SEDIMENT_TOTAL_LOAD_CONCENTRATION'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'CAPAC_OUTPUT':
            self.sim_data.output.attrs['SEDIMENT_TOTAL_LOAD_CAPACITY'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'FRAC_SUSP_OUTPUT':
            self.sim_data.output.attrs['FRACTION_SUSPENDED'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'BED_FRAC_OUTPUT' or card == 'SIZE_FRACTION_OUTPUT':
            self.sim_data.output.attrs['FRACTION_BEDLOAD'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'WAVE_DISS_OUTPUT':
            self.sim_data.output.attrs['WAVE_DISSIPATION'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'WIND_MAG_OUTPUT':
            self.sim_data.output.attrs['WIND_SPEED'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'ATM_PRESS_OUTPUT':
            self.sim_data.output.attrs['ATM_PRESSURE'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'FLOW_STATISTICS':
            self.sim_data.output.attrs['ENABLE_STATISTICS'] = 1
            self.sim_data.output.attrs['ENABLE_HYDRO_STATISTICS'] = 1
            self.sim_data.output.attrs['HYDRO_START_TIME'] = float(line_data[1])
            self.sim_data.output.attrs['HYDRO_END_TIME'] = float(line_data[2])
            self.sim_data.output.attrs['HYDRO_INCREMENT'] = float(line_data[3])
        elif card == 'SEDIMENT_STATISTICS':
            self.sim_data.output.attrs['ENABLE_STATISTICS'] = 1
            self.sim_data.output.attrs['ENABLE_SEDIMENT_STATISTICS'] = 1
            self.sim_data.output.attrs['SEDIMENT_START_TIME'] = float(line_data[1])
            self.sim_data.output.attrs['SEDIMENT_END_TIME'] = float(line_data[2])
            self.sim_data.output.attrs['SEDIMENT_INCREMENT'] = float(line_data[3])
        elif card == 'SALINITY_STATISTICS':
            self.sim_data.output.attrs['ENABLE_STATISTICS'] = 1
            self.sim_data.output.attrs['ENABLE_SALINITY_STATISTICS'] = 1
            self.sim_data.output.attrs['SALINITY_START_TIME'] = float(line_data[1])
            self.sim_data.output.attrs['SALINITY_END_TIME'] = float(line_data[2])
            self.sim_data.output.attrs['SALINITY_INCREMENT'] = float(line_data[3])
        elif card == 'WAVE_STATISTICS':
            self.sim_data.output.attrs['ENABLE_STATISTICS'] = 1
            self.sim_data.output.attrs['ENABLE_WAVE_STATISTICS'] = 1
            self.sim_data.output.attrs['WAVE_START_TIME'] = float(line_data[1])
            self.sim_data.output.attrs['WAVE_END_TIME'] = float(line_data[2])
            self.sim_data.output.attrs['WAVE_INCREMENT'] = float(line_data[3])
        # File parameters
        elif card == 'USE_COMMON_SOLUTION_FILE':
            self.sim_data.output.attrs['SINGLE_SOLUTION'] = 1
        elif card == 'WRITE_ASCII_INPUT_FILES':
            self.sim_data.output.attrs['WRITE_ASCII'] = 1
        elif card == 'GLOBAL_TECPLOT_FILES':
            self.sim_data.output.attrs['TECPLOT'] = 1
        elif card == 'OUTPUT_FILE_TYPE':
            display_text = ''
            if line_data[1] == 'XMDF':
                display_text = 'XMDF binary output'
            elif line_data[1] == 'ASCII':
                display_text = 'ASCII output only'
            self.sim_data.output.attrs['SOLUTION_OUTPUT'] = display_text
        elif card == 'XMDF_COMPRESSION':
            self.sim_data.output.attrs['XMDF_COMPRESSION'] = int(line_data[1])
        # Implicit/explicit parameters
        elif card == 'SOLUTION_SCHEME':
            display_text = ''
            if line_data[1] == 'IMPLICIT':
                display_text = 'Implicit'
            elif line_data[1] == 'EXPLICIT':
                display_text = 'Explicit'
            self.sim_data.general.attrs['SOLUTION_SCHEME'] = display_text
        elif card == 'MATRIX_SOLVER':
            display_text = ''
            if line_data[1] == 'GAUSS-SEIDEL':
                display_text = 'Gauss-Seidel'
            elif line_data[1] == 'GAUSS-SEIDEL-SOR':
                display_text = 'Gauss-Seidel-SOR'
            elif line_data[1] == 'BICGSTAB':
                display_text = 'BiCGSTab'
            elif line_data[1] == 'GMRES':
                display_text = 'GMRES'
            self.sim_data.general.attrs['MATRIX_SOLVER'] = display_text
        elif card == 'SKEWNESS_CORRECTION':
            if line_data[1] == 'OFF':
                self.sim_data.general.attrs['SKEW_CORRECT'] = 0
        elif card == 'MET_STATION_BEGIN':
            self.reading_met_station = True
            self.add_met_station()
            self.sim_data.wind.attrs['WIND_TYPE'] = 'Meteorological stations'
        elif card == 'WIND_INPUT_CURVE':
            self.add_curve(card_line_orig, 'WIND_INPUT_CURVE', 'wind_from_table')
            if len(self.bc_filename) == 0:
                self.bc_filename = self.extract_filename(card_line_orig)
        elif card == 'ANEMOMETER_HEIGHT':
            self.sim_data.wind.attrs['WIND_TYPE'] = 'Spatially constant'
            self.sim_data.wind.attrs['ANEMOMETER'] = float(line_data[1])
        elif card == 'OCEANWEATHER_WIND_FILE':
            self.sim_data.wind.attrs['WIND_TYPE'] = 'Temporally and spatially varying from file'
            file_path = self.extract_filename(card_line_orig)
            self.sim_data.wind.attrs['OCEAN_WIND_FILE'] = file_path
            self.sim_data.wind.attrs['WIND_FILE_TYPE'] = 'OWI/PBL'
        elif card == 'OCEANWEATHER_PRES_FILE':
            file_path = self.extract_filename(card_line_orig)
            self.sim_data.wind.attrs['OCEAN_PRESSURE_FILE'] = file_path
            self.sim_data.wind.attrs['WIND_FILE_TYPE'] = 'OWI/PBL'
        elif card == 'OCEANWEATHER_XY_FILE':
            file_path = self.extract_filename(card_line_orig)
            self.sim_data.wind.attrs['OCEAN_XY_FILE'] = file_path
            self.sim_data.wind.attrs['WIND_FILE_TYPE'] = 'OWI/PBL'
        elif card == 'WIND_PRESSURE_SINGLE_FILE':
            self.sim_data.wind.attrs['WIND_TYPE'] = 'Temporally and spatially varying from file'
            file_path = self.extract_filename(card_line_orig)
            self.sim_data.wind.attrs['WIND_FILE'] = file_path
            self.sim_data.wind.attrs['WIND_FILE_TYPE'] = 'Single ASCII file'
        elif card == 'WIND_FLEET_FILE':
            self.sim_data.wind.attrs['WIND_TYPE'] = 'Temporally and spatially varying from file'
            file_path = self.extract_filename(card_line_orig)
            self.sim_data.wind.attrs['WIND_FILE'] = file_path
            self.sim_data.wind.attrs['WIND_FILE_TYPE'] = 'Navy fleet numeric with pressure'
        elif card == 'WIND_PRESSURE_XY_FILE':
            file_path = self.extract_filename(card_line_orig)
            self.sim_data.wind.attrs['WIND_GRID_FILE'] = file_path
            self.sim_data.wind.attrs['WIND_GRID_TYPE'] = 'XY file'
        elif card == 'WIND_PRESSURE_GRID_PARAM':
            self.sim_data.wind.attrs['WIND_GRID_NUM_Y_VALUES'] = int(line_data[1])
            self.sim_data.wind.attrs['WIND_GRID_NUM_X_VALUES'] = int(line_data[2])
            self.sim_data.wind.attrs['WIND_GRID_MAX_Y_LOCATION'] = float(line_data[3])
            self.sim_data.wind.attrs['WIND_GRID_MIN_X_LOCATION'] = float(line_data[4])
            self.sim_data.wind.attrs['WIND_GRID_Y_DISTANCE'] = float(line_data[5])
            self.sim_data.wind.attrs['WIND_GRID_X_DISTANCE'] = float(line_data[6])
            self.sim_data.wind.attrs['WIND_GRID_TYPE'] = 'Parameters'
        elif card == 'WIND_PRESSURE_TIME_INCREMENT':
            self.sim_data.wind.attrs['WIND_GRID_TIME_INCREMENT'] = float(line_data[1])
        elif card == 'WAVE_RSTRESS_DATASET':
            self.add_single_wave_dataset(card_line_orig, 'WAVE_RADIATION')
        elif card == 'WAVE_HEIGHT_DATASET':
            self.add_single_wave_dataset(card_line_orig, 'WAVE_HEIGHT')
        elif card == 'WAVE_PERIOD_DATASET':
            self.add_single_wave_dataset(card_line_orig, 'PEAK_PERIOD')
        elif card == 'WAVE_DIRECTION_DATASET':
            self.add_single_wave_dataset(card_line_orig, 'MEAN_WAVE_DIR')
        elif card == 'WAVE_DISS_DATASET':
            self.add_single_wave_dataset(card_line_orig, 'WAVE_BREAKING')
        # elif card == 'ROLLER_STRESS_DATASET':  # experimental
        #     self.sim_data.wave.attrs['WAVE_INFO'] = 'Single wave condition'
        #     self.parse_dataset_reference(card_line_orig, , )
        elif card == 'CMS-WAVE_SIM_FILE':
            filename = self.extract_filename(card_line_orig)
            self.sim_data.wave.attrs['WAVE_INFO'] = 'Inline steering'
            if filename and filename != 'NONE':
                self.sim_data.wave.attrs['FILE_WAVE_SIM'] = filename
        elif card == 'CMS-STEERING_INTERVAL':
            interval = float(line_data[1])
            self.sim_data.wave.attrs['STEERING_INTERVAL_TYPE'] = 'Constant'
            self.sim_data.wave.attrs['STEERING_INTERVAL_CONST'] = interval
        elif card == 'FLOW_EXTRAPOLATION_DISTANCE':
            self.sim_data.wave.attrs['EXTRAPOLATION_DISTANCE'] = 1
            distance = float(line_data[1])
            if distance == -999.0:
                self.sim_data.wave.attrs['FLOW_TO_WAVE'] = 'Automatic'
            else:
                self.sim_data.wave.attrs['FLOW_TO_WAVE'] = 'User specified'
                self.sim_data.wave.attrs['FLOW_TO_WAVE_USER_VALUE'] = distance
                if len(line_data) > 2:
                    self.sim_data.wave.attrs['FLOW_TO_WAVE_USER_UNITS'] = clean_token(line_data[2])
        elif card == 'WAVE_EXTRAPOLATION_DISTANCE':
            distance = float(line_data[1])
            if distance == -999.0:
                self.sim_data.wave.attrs['WAVE_TO_FLOW'] = 'Automatic'
            else:
                self.sim_data.wave.attrs['WAVE_TO_FLOW'] = 'User specified'
                self.sim_data.wave.attrs['WAVE_TO_FLOW_USER_VALUE'] = distance
                if len(line_data) > 2:
                    self.sim_data.wave.attrs['WAVE_TO_FLOW_USER_UNITS'] = clean_token(line_data[2])
        elif card == 'WAVE_WATER_LEVEL':
            predictor = ''
            if line_data[1] == 'LAST':
                predictor = 'Last time step'
            elif line_data[1] == 'TIDAL':
                predictor = 'Tidal'
            elif line_data[1] == 'TIDAL_PLUS_VARIATION':
                predictor = 'Tidal plus variation'
            self.sim_data.wave.attrs['WAVE_WATER_PREDICTOR'] = predictor

        # Bottom friction
        elif card == 'MANNINGS_N_DATASET':
            self.sim_data.flow.attrs['ROUGHNESS_SOURCE'] = 'Dataset'
            self.sim_data.flow.attrs['BOTTOM_ROUGHNESS'] = 'Mannings N'
            self.parse_dataset_reference(card_line_orig, 'BOTTOM_ROUGHNESS_DSET', self.sim_data.flow.attrs)
        elif card == 'BOTTOM_FRICTION_COEF_DATASET':
            self.sim_data.flow.attrs['ROUGHNESS_SOURCE'] = 'Dataset'
            self.sim_data.flow.attrs['BOTTOM_ROUGHNESS'] = 'Bottom friction coefficient'
            self.parse_dataset_reference(card_line_orig, 'BOTTOM_ROUGHNESS_DSET', self.sim_data.flow.attrs)
        elif card == 'ROUGHNESS_HEIGHT_DATASET':
            self.sim_data.flow.attrs['ROUGHNESS_SOURCE'] = 'Dataset'
            self.sim_data.flow.attrs['BOTTOM_ROUGHNESS'] = 'Roughness height (m)'
            self.parse_dataset_reference(card_line_orig, 'BOTTOM_ROUGHNESS_DSET', self.sim_data.flow.attrs)
        elif card == 'MANNINGS_N_CONSTANT':
            self.sim_data.flow.attrs['ROUGHNESS_SOURCE'] = 'Constant'
            self.sim_data.flow.attrs['BOTTOM_ROUGHNESS'] = 'Mannings N'
            self.sim_data.flow.attrs['ROUGHNESS_CONSTANT'] = float(line_data[1])
        elif card == 'BOTTOM_FRICTION_COEF_CONSTANT':
            self.sim_data.flow.attrs['ROUGHNESS_SOURCE'] = 'Constant'
            self.sim_data.flow.attrs['BOTTOM_ROUGHNESS'] = 'Bottom friction coefficient'
            self.sim_data.flow.attrs['ROUGHNESS_CONSTANT'] = float(line_data[1])
        elif card == 'ROUGHNESS_HEIGHT_CONSTANT':
            self.sim_data.flow.attrs['ROUGHNESS_SOURCE'] = 'Constant'
            self.sim_data.flow.attrs['BOTTOM_ROUGHNESS'] = 'Roughness height (m)'
            self.sim_data.flow.attrs['ROUGHNESS_CONSTANT'] = float(line_data[1])

        # Sediment transport
        elif card == 'CALC_SEDIMENT_TRANSPORT':
            self.sim_data.sediment.attrs['CALCULATE_SEDIMENT'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'SED_TRAN_CALC_INTERVAL':
            self.sim_data.sediment.attrs['TRANSPORT_TIME_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['TRANSPORT_TIME_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'MORPH_UPDATE_INTERVAL':
            self.sim_data.sediment.attrs['MORPHOLOGIC_TIME_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['MORPHOLOGIC_TIME_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'MORPH_START_TIME':
            self.sim_data.sediment.attrs['MORPHOLOGY_TIME_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['MORPHOLOGY_TIME_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'SED_TRANS_FORMULATION' or card == 'SED_TRAN_FORMULATION':  # card was renamed
            units = ''
            if line_data[1] == 'EXNER':
                units = 'Equilibrium total load'
            elif line_data[1] == 'A-D':
                units = 'Equilibrium bed load plus nonequilibrium susp. load'
            elif line_data[1] == 'NET':
                units = 'Nonequilibrium total load'
            self.sim_data.sediment.attrs['FORMULATION_UNITS'] = units
        elif card == 'TRANSPORT_FORMULA' or card == 'NET_TRANSPORT_CAPACITY':  # card was renamed
            formula = ''
            if line_data[1] in ['LUND-CIR', 'LUND-CIRP']:
                formula = 'Lund-CIRP'
            elif line_data[1] == 'VAN_RIJN':
                formula = 'van Rijn'
            elif line_data[1] == 'SOULSBY-VAN_RIJN':
                formula = 'Soulsby-van Rijn'
            elif line_data[1] == 'WATANABE':
                formula = 'Watanabe'
            elif line_data[1] == 'C2SHORE':
                formula = 'C2SHORE'
            self.sim_data.sediment.attrs['TRANSPORT_FORMULA'] = formula
        elif card == 'CONCENTRATION_PROFILE':
            concentration = ''
            if line_data[1] == 'EXPONENTIAL':
                concentration = 'Exponential'
            elif line_data[1] == 'ROUSE':
                concentration = 'Rouse'
            elif line_data[1] == 'LUND-CIRP':
                concentration = 'Lund-CIRP'
            elif line_data[1] == 'VAN_RIJN':
                concentration = 'van Rijn'
            self.sim_data.sediment.attrs['CONCENTRATION_PROFILE'] = concentration
        elif card == 'A_COEFFICIENT_WATANABE':
            self.sim_data.sediment.attrs['WATANABE_RATE'] = float(line_data[1])
        elif card == 'C2SHORE_EFFICIENCY':
            self.sim_data.sediment.attrs['C2SHORE_EFFICIENCY'] = float(line_data[1])
        elif card == 'C2SHORE_BED_LOAD':
            self.sim_data.sediment.attrs['C2SHORE_BED_LOAD'] = float(line_data[1])
        elif card == 'C2SHORE_SUSP_LOAD':
            self.sim_data.sediment.attrs['C2SHORE_SUSP_LOAD'] = float(line_data[1])
        elif card == 'SEDIMENT_DENSITY':
            self.sim_data.sediment.attrs['SEDIMENT_DENSITY_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['SEDIMENT_DENSITY_UNITS'] = clean_token(line_data[2])
        elif card == 'SEDIMENT_POROSITY':
            self.sim_data.sediment.attrs['SEDIMENT_POROSITY'] = float(line_data[1])
        elif card == 'BED_LOAD_SCALE_FACTOR':
            self.sim_data.sediment.attrs['BED_LOAD_SCALING'] = float(line_data[1])
        elif card == 'SUSP_LOAD_SCALE_FACTOR':
            self.sim_data.sediment.attrs['SUSPENDED_LOAD_SCALING'] = float(line_data[1])
        elif card == 'MORPH_ACCEL_FACTOR':
            self.sim_data.sediment.attrs['MORPHOLOGIC_ACCELERATION'] = float(line_data[1])
        elif card == 'BEDSLOPE_COEFFICIENT' or card == 'SLOPE_COEFFICIENT':  # card was renamed
            self.sim_data.sediment.attrs['BED_SLOPE_DIFFUSION'] = float(line_data[1])
        elif card == 'HIDING_EXPOSURE_COEFFICIENT':
            self.sim_data.sediment.attrs['HIDING_AND_EXPOSURE'] = float(line_data[1])
        elif card == 'CONSTANT_GRAIN_SIZE':
            self.sim_data.sediment.attrs['GRAIN_SIZE_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['GRAIN_SIZE_UNITS'] = clean_token(line_data[2])
        elif card == 'ADAPTATION_METHOD_TOTAL':
            adapt_method = ''
            if line_data[1] == 'CONSTANT_LENGTH':
                adapt_method = 'Constant length'
            elif line_data[1] == 'CONSTANT_TIME':
                adapt_method = 'Constant time'
            elif line_data[1] == 'MAX_BED_SUSP_LENGTH':
                adapt_method = 'Maximum of bed and suspended adaptation lengths'
            elif line_data[1] == 'WGHT_AVG_BED_SUSP_LENGTH':
                adapt_method = 'Weighted average of bed and suspended adaptation lengths'
            self.sim_data.sediment.attrs['TOTAL_ADAPTATION_METHOD'] = adapt_method
        elif card == 'ADAPTATION_LENGTH_TOTAL':
            self.sim_data.sediment.attrs['TOTAL_ADAPTATION_LENGTH_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['TOTAL_ADAPTATION_LENGTH_UNITS'] = clean_token(line_data[2])
        elif card == 'ADAPTATION_TIME_TOTAL':
            self.sim_data.sediment.attrs['TOTAL_ADAPTATION_TIME_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['TOTAL_ADAPTATION_TIME_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'ADAPTATION_METHOD_BED':
            adapt_method_bed = ''
            if line_data[1] == 'CONSTANT_LENGTH':
                adapt_method_bed = 'Constant length'
            elif line_data[1] == 'CONSTANT_TIME':
                adapt_method_bed = 'Constant time'
            elif line_data[1] == 'DEPTH_DEPENDENT':
                adapt_method_bed = 'Depth Dependent'
            self.sim_data.sediment.attrs['BED_ADAPTATION_METHOD'] = adapt_method_bed
        elif card == 'ADAPTATION_LENGTH_BED':
            self.sim_data.sediment.attrs['BED_ADAPTATION_LENGTH_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['BED_ADAPTATION_LENGTH_UNITS'] = clean_token(line_data[2])
        elif card == 'ADAPTATION_TIME_BED':
            self.sim_data.sediment.attrs['BED_ADAPTATION_TIME_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['BED_ADAPTATION_TIME_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'ADAPTATION_DEPTH_FACTOR_BED':
            self.sim_data.sediment.attrs['BED_ADAPTATION_DEPTH'] = float(line_data[1])
        elif card == 'ADAPTATION_METHOD_SUSPENDED':
            adapt_method_susp = ''
            if line_data[1] == 'CONSTANT_LENGTH':
                adapt_method_susp = 'Constant length'
            elif line_data[1] == 'CONSTANT_TIME':
                adapt_method_susp = 'Constant time'
            elif line_data[1] == 'CONSTANT_ALPHA':
                adapt_method_susp = 'Constant coefficient'
            elif line_data[1] == 'ARMANNI_DISILVIO':
                adapt_method_susp = 'Armanini and Di Silvio'
            elif line_data[1] == 'LIN':
                adapt_method_susp = 'Lin'
            elif line_data[1] == 'GALLAPPATTI':
                adapt_method_susp = 'Gallappatti'
            self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_METHOD'] = adapt_method_susp
        elif card == 'ADAPTATION_LENGTH_SUSPENDED':
            self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_LENGTH_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_LENGTH_UNITS'] = clean_token(line_data[2])
        elif card == 'ADAPTATION_TIME_SUSPENDED':
            self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_TIME_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_TIME_UNITS'] = card_to_time_unit(line_data[2])
        elif card == 'ADAPTATION_COEFFICIENT_SUSPENDED':
            self.sim_data.sediment.attrs['SUSPENDED_ADAPTATION_COEFFICIENT'] = float(line_data[1])
        # handle the old simplified transport option for a single layer, single size class
        elif card == 'TRANSPORT_GRAIN_SIZE':
            self.add_sediment_size()
            self.sediment_size_values['diameter_value'][-1] = float(line_data[1])
            self.sediment_size_values['diameter_units'][-1] = 'mm'
        elif card == 'SEDIMENT_SIZE_CLASS_BEGIN':
            self.reading_sediment_size = True
            self.add_sediment_size()
        elif card == 'BED_LAYERS_MAX_NUMBER':
            self.sim_data.sediment.attrs['MAX_NUMBER_BED_LAYERS'] = int(line_data[1])
        elif card == 'BED_LAYERS_MIN_THICKNESS':
            self.sim_data.sediment.attrs['MIN_BED_LAYER_THICKNESS_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['MIN_BED_LAYER_THICKNESS_UNITS'] = clean_token(line_data[2])
        elif card == 'BED_LAYERS_MAX_THICKNESS':
            self.sim_data.sediment.attrs['MAX_BED_LAYER_THICKNESS_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.sediment.attrs['MAX_BED_LAYER_THICKNESS_UNITS'] = clean_token(line_data[2])
        elif card == 'MIXING_LAYER_FORMULATION':
            mixing_formulation = '0'
            if line_data[1] == 'AUTOMATIC':
                mixing_formulation = 'Automatic'
            elif line_data[1] == 'CONSTANT':
                mixing_formulation = 'Constant'
            elif line_data[1] == 'DATASET':
                mixing_formulation = 'Dataset'
            self.mixing_layer_thickness_type = mixing_formulation
        elif card == 'MIXING_LAYER_THICKNESS_CONSTANT':
            self.mixing_layer_thickness_const = float(line_data[1])
            if len(line_data) > 2:
                units = clean_token(line_data[2]).lower()
                # convert to meters
                ft_m = 3.048  # feet per meter
                ft_yd = 3.0  # feet per yard
                in_ft = 12.0  # inches per foot
                cm_m = 100.0  # centimeters per meter
                try:
                    units_idx = ['m', 'cm', 'ft', 'in', 'yd'].index(units)
                    unit_conversions = [1.0, 1.0 / cm_m, 1.0 / ft_m, 1.0 / ft_m / in_ft, ft_yd / ft_m]
                    self.mixing_layer_thickness_const = self.mixing_layer_thickness_const * unit_conversions[units_idx]
                except ValueError:
                    pass
            #     self.sim_data.sediment.attrs['CONSTANT_MIXING_LAYER_THICKNESS_UNITS'] = clean_token(line_data[2])
        # Bed layer composition table
        elif card == 'BED_LAYER_BEGIN':
            self.reading_bed_layer = True
            self.add_bed_layer()
        elif card == 'USE_AVALANCHING':
            if len(line_data) < 2 or line_data[1] != 'OFF':
                self.sim_data.sediment.attrs['CALCULATE_AVALANCHING'] = 1
        elif card == 'AVALANCHE_CRITICAL_BEDSLOPE':
            self.sim_data.sediment.attrs['CRITICAL_BED_SLOPE'] = float(line_data[1])
        elif card == 'AVALANCHE_MAX_ITERATIONS':
            self.sim_data.sediment.attrs['MAX_NUMBER_ITERATIONS'] = int(line_data[1])
        # Projection info
        elif card == 'HORIZONTAL_PROJECTION_BEGIN':
            self.reading_horizontal_proj = True
        elif card == 'VERTICAL_PROJECTION_BEGIN':
            self.reading_horizontal_proj = False  # Subsequent DATUM and UNITS cards are for the vertical projection
        elif card == 'DATUM':
            # Sanitize the DATUM. For some reason, there are all sorts of variations for the datum card values. Most
            # likely export bugs in older versions of the interface.
            datum = clean_token(''.join([piece for piece in line_data[1:] if not is_comment(piece)]))
            if self.reading_horizontal_proj:
                # data_objects Projection does not have horizontal datum field in 13.0 version. Have to build the WKT
                # with GDAL directly.
                self._horizontal_datum = datum
            else:
                self.grid_projection.vertical_datum = datum
        elif card in ['SYSTEM', 'COORDINATE_SYSTEM']:
            # Convert CMS-Flow card for State Plane to API version. Geographic and UTM are the same.
            system = line_data[1].upper()
            if system == 'STATE_PLANE':
                self.grid_projection.coordinate_system = 'STATEPLANE'
            elif system == 'NONE':
                self.grid_projection.coordinate_system = 'LOCAL'
            else:  # Geographic and UTM CMS keywords match data_objects keywords
                self.grid_projection.coordinate_system = system
        elif card == 'UNITS':
            if self.reading_horizontal_proj:
                self.grid_projection.horizontal_units = card_to_projection_unit(line_data[1])
            else:
                self.grid_projection.vertical_units = card_to_projection_unit(line_data[1])
        elif card == 'ZONE':
            self.grid_projection.coordinate_zone = int(line_data[1])
        # Boundary conditions
        elif card == 'BOUNDARY_BEGIN':
            self.reading_bc = True
            self.add_bc_arc()
        elif card == 'RUBBLE_MOUND_BEGIN':
            self.reading_rubble_mound = True
            self.add_rubble_mound()
        elif card == 'DREDGE_UPDATE_INTERVAL':
            self.sim_data.dredge.attrs['UPDATE_INTERVAL_VALUE'] = float(line_data[1])
            if len(line_data) > 2:
                self.sim_data.dredge.attrs['UPDATE_INTERVAL_UNITS'] = clean_token(line_data[2])
        elif card == 'OUTPUT_DREDGE_DIAGNOSTICS':
            self.sim_data.dredge.attrs['ENABLE_DIAGNOSTIC'] = 1 if line_data[1] == 'ON' else 0
        elif card == 'DREDGE_OPERATION_BEGIN':
            self.reading_dredge_operation = True
        elif card == 'RUBBLE_MOUND_ID_DATASET':
            pass
        else:
            self._add_advanced_card('-- No Block --', line_data)

    def process_save_point_card(self, tokens):
        """Opens the processes each SAVE_POINT card.

        Args:
            tokens (list(str)): A list of arguments on the SAVE_POINT card_line.
        """
        pt = Point(float(tokens[3]), float(tokens[4]))
        pt.id = len(self.save_points_values['name']) + 1
        name = clean_token(tokens[1])
        num_cards = len(tokens)
        hydro = 0
        sediment = 0
        salinity = 0
        wave = 0
        for idx in range(5, num_cards):
            output_card = tokens[idx]
            if output_card == 'HYDRO':
                hydro = 1
            elif output_card == 'SEDIMENT':
                sediment = 1
            elif output_card == 'SALINITY':
                salinity = 1
            elif output_card == 'WAVE':
                wave = 1
        self.save_points_values['point'].append(pt)
        self.save_points_values['name'] = np.append(self.save_points_values['name'], name)
        self.save_points_values['hydro'] = np.append(self.save_points_values['hydro'], hydro)
        self.save_points_values['sediment'] = np.append(self.save_points_values['sediment'], sediment)
        self.save_points_values['salinity'] = np.append(self.save_points_values['salinity'], salinity)
        self.save_points_values['waves'] = np.append(self.save_points_values['waves'], wave)

    def process_spcards_file(self, filename):
        """Opens the save points card file and reads all SAVE_POINT cards contained.

        Args:
            filename (str): The line in the card file containing the filename.
        """
        lineno = 0
        with open(filename, 'r') as spfile:
            spcard_lines = spfile.readlines()
            for spcard_line in spcard_lines:
                lineno += 1
                spcard_line = spcard_line.strip()
                if not spcard_line or is_comment(spcard_line):
                    continue  # Skip blank lines, comment lines, and old SMS cards
                try:
                    msg = f'Error parsing save point data on line {lineno}: {spcard_line}'
                    tokens = safe_shlex_split(spcard_line, msg, min_tokens=5)
                    if not tokens:
                        return
                    self.process_save_point_card(tokens)

                except Exception:
                    XmLog().instance.exception(f'Unknown error reading line: {lineno}: {spcard_line}')

    def add_single_wave_dataset(self, card_line_orig, dataset_name):
        """Adds a dataset for a single wave condition. Sets the wave type to single wave condition.

        Args:
            card_line_orig (str): The line in the card file.
            dataset_name (str): The dataset name for storage.
        """
        self.sim_data.wave.attrs['WAVE_INFO'] = 'Single wave condition'
        self.parse_dataset_reference(card_line_orig, dataset_name, self.sim_data.wave.attrs)

    def add_bed_layer_dataset(self, card_line_orig, column):
        """Adds a dataset for a cell in the bed layer table.

        Args:
            card_line_orig (str): The line in the card file.
            column (str): The column name in the bed layer table.
        """
        row = len(self.bed_layer_values[column]) - 1
        self.parse_dataset_reference(card_line_orig, column, self.bed_layer_values, row)

    def add_save_points_coverage(self):
        """Build the save points coverage and add it to the Context with coverage-level and point-level widgets.

        """
        if self.save_points_values['point']:
            XmLog().instance.info('Creating save points coverage')
            cov_uuid = str(uuid.uuid4())
            self.save_points_cov = Coverage(
                name=self.save_points_cov_name, uuid=cov_uuid, projection=self.grid_projection
            )
            pts = self.save_points_values.pop('point')
            self.save_points_cov.set_points(pts)
            self.save_points_cov.complete()

            comp_uuid = str(uuid.uuid4())
            save_comp_dir = os.path.join(self.comp_dir, comp_uuid)
            os.makedirs(save_comp_dir, exist_ok=True)
            save_pts_mainfile = os.path.join(save_comp_dir, 'save_points_comp.nc')
            save_pts_data = SavePointsData(save_pts_mainfile)
            save_pts_data.info.attrs['cov_uuid'] = cov_uuid
            ids = [pt.id for pt in pts]
            coords = {'comp_id': ids}
            pt_table = {}
            for key in self.save_points_values.keys():
                pt_table[key] = ('comp_id', self.save_points_values[key])
            save_pts_data.points = xr.Dataset(data_vars=pt_table, coords=coords)

            save_pts_data.info.attrs['next_comp_id'] = max(ids) + 1
            save_pts_data.general.attrs['HYDRO_VALUE'] = self.save_points_cov_values['HYDRO_VALUE']
            save_pts_data.general.attrs['HYDRO_UNITS'] = self.save_points_cov_values['HYDRO_UNITS']
            save_pts_data.general.attrs['SEDIMENT_VALUE'] = self.save_points_cov_values['SEDIMENT_VALUE']
            save_pts_data.general.attrs['SEDIMENT_UNITS'] = self.save_points_cov_values['SEDIMENT_UNITS']
            save_pts_data.general.attrs['SALINITY_VALUE'] = self.save_points_cov_values['SALINITY_VALUE']
            save_pts_data.general.attrs['SALINITY_UNITS'] = self.save_points_cov_values['SALINITY_UNITS']
            save_pts_data.general.attrs['WAVES_VALUE'] = self.save_points_cov_values['WAVES_VALUE']
            save_pts_data.general.attrs['WAVES_UNITS'] = self.save_points_cov_values['WAVES_UNITS']
            save_pts_data.commit()

            # Write component id and save point att ids to a file so we can initialize them in
            # get_initial_display_options
            id_file = os.path.join(save_comp_dir, SAVE_INITIAL_ATT_ID_FILE)
            write_display_option_ids(id_file, ids)
            id_file = os.path.join(save_comp_dir, SAVE_INITIAL_COMP_ID_FILE)
            write_display_option_ids(id_file, ids)
            id_file = os.path.join(save_comp_dir, 'save_point.display_ids')
            write_display_option_ids(id_file, ids)

            self.build_data['save_pts_cov_comp'] = SavePointsComponent(save_pts_mainfile)
            self.build_data['save_pts_cov'] = self.save_points_cov

    def add_dependent_widgets(self):
        """Add widgets that have dependencies after all the cards have been read.

        """
        # Starting date
        if 'STARTING_JDATE' in self.dependent_widgets and 'STARTING_JDATE_HOUR' in self.dependent_widgets:
            hours = self.dependent_widgets['STARTING_JDATE_HOUR']
            py_dt = self.dependent_widgets['STARTING_JDATE']
            py_dt = py_dt.replace(hour=hours)
            self.sim_data.general.attrs['DATE_START'] = py_dt.strftime('%Y-%m-%dT%H:%M:%S')

        # Hot start file toggle
        if 'HOT_START_TIME' not in self.dependent_widgets:
            self.sim_data.general.attrs['USE_HOT_START_OUTPUT_FILE'] = 0
        # Auto hot start interval toggle
        if 'AUTO_HOT_START_INTERVAL' not in self.dependent_widgets:
            self.sim_data.general.attrs['RECURRING_HOT_START_FILE'] = 0

        # add size diameters
        if self.sediment_size_values:
            for key, item in self.sediment_size_values.items():
                self.sediment_size_values[key] = xr.DataArray(item)
            self.sim_data.advanced_sediment_diameters_table = xr.Dataset(data_vars=self.sediment_size_values)

    def build_structure_coverage(self):
        """Builds the polygons. Adds them to xms_data if possible."""
        XmLog().instance.info('Creating structure coverage')

        num_rubble_mounds = len(self.rubble_mound_values['name'])
        num_weirs = len(self.weir_values)
        num_culverts = len(self.culvert_values)
        num_tide_gates = len(self.tide_gate_values)
        if num_rubble_mounds == 0 and num_weirs == 0 and num_culverts == 0 and num_tide_gates == 0:
            return

        builder = CoverageComponentBuilder(StructuresComponent, 'Structures', self.quad_ugrid, self.grid_projection)

        if num_rubble_mounds > 0:
            self._build_structure_rubble_mounds(builder)
        if num_weirs > 0:
            self._build_structure_weirs(builder)
        if num_culverts > 0:
            self._build_structure_culverts(builder)
        if num_tide_gates > 0:
            self._build_structure_tide_gates(builder)

        coverage, component, keywords = builder.build()
        self.build_data['structure_coverage'] = coverage
        self.build_data['structure_component'] = component
        self.build_data['structure_keywords'] = keywords

    def _build_structure_rubble_mounds(self, builder: CoverageComponentBuilder):
        """Build the rubble mound features in a structure coverage."""
        data: CoverageData = builder.data
        section = get_model().polygon_parameters
        dataset_value_to_component_id = {}

        for i in range(len(self.rubble_mound_values['name'])):
            section.clear_values()
            group = section.group('rubble_mound')
            group.is_active = True

            parameter = group.parameter('name')
            parameter.value = self.rubble_mound_values['name'][i]

            parameter = group.parameter('rock_diameter')
            parameter.value = self.rubble_mound_values['rock_diameter'][i]

            parameter = group.parameter('rock_diameter_type')
            index = self.rubble_mound_values['rock_diameter_type'][i]
            parameter.value = parameter.options[index]

            parameter = group.parameter('rock_diameter_dataset')
            parameter.value = self.rubble_mound_values['ROCK_DIAMETER_DATASET'][i]

            parameter = group.parameter('porosity')
            parameter.value = self.rubble_mound_values['porosity'][i]

            parameter = group.parameter('porosity_type')
            index = self.rubble_mound_values['porosity_type'][i]
            parameter.value = parameter.options[index]

            parameter = group.parameter('porosity_dataset')
            parameter.value = self.rubble_mound_values['STRUCTURE_POROSITY_DATASET'][i]

            parameter = group.parameter('base_depth')
            parameter.value = self.rubble_mound_values['base_depth'][i]

            parameter = group.parameter('base_depth_type')
            index = self.rubble_mound_values['base_depth_type'][i]
            parameter.value = parameter.options[index]

            parameter = group.parameter('base_depth_dataset')
            parameter.value = self.rubble_mound_values['STRUCTURE_BASE_DEPTH_DATASET'][i]

            parameter = group.parameter('calc_method')
            parameter.value = self.rubble_mound_values['calculation_method'][i]

            component_id = data.add_feature(TargetType.polygon, section.extract_values(), 'rubble_mound')
            dataset_value_to_component_id[i + 1] = component_id

        needed_length = self.quad_ugrid.ugrid.cell_count
        if self.rubble_mound_cells.size < needed_length:
            append_length = needed_length - self.rubble_mound_cells.size
            self.rubble_mound_cells = np.pad(self.rubble_mound_cells, (0, append_length))
        builder.add_polygons(self.rubble_mound_cells.tolist(), dataset_value_to_component_id, null_value=0)

    def _build_structure_weirs(self, builder: CoverageComponentBuilder):
        """Build the weirs in a structure coverage."""
        data: CoverageBaseData = builder.data
        types = ['weir'] * len(self.weir_values)
        component_ids = data.add_features(TargetType.arc, self.weir_values, types)
        for weir_cells, component_id in zip(self.weirs, component_ids, strict=True):
            builder.add_cell_string(weir_cells, component_id)

    def _build_structure_culverts(self, builder: CoverageComponentBuilder):
        """Build the culverts in a structure coverage."""
        data: CoverageBaseData = builder.data
        types = ['culvert'] * len(self.culvert_values)
        component_ids = data.add_features(TargetType.arc, self.culvert_values, types)
        for culvert, component_id in zip(self.culverts, component_ids, strict=True):
            builder.add_cell_string(culvert, component_id)

    def _build_structure_tide_gates(self, builder: CoverageComponentBuilder):
        """Build the tide gates in a structure coverage."""
        data: CoverageBaseData = builder.data
        types = ['tide_gate'] * len(self.tide_gate_values)
        component_ids = data.add_features(TargetType.arc, self.tide_gate_values, types)
        for tide_gate, component_id in zip(self.tide_gates, component_ids, strict=True):
            builder.add_cell_string(tide_gate, component_id)

    def read_files(self):
        """Reads the cmcards files and any other files referenced therein."""
        XmLog().instance.info('Reading cards')
        with open(self.filename, 'r') as file:
            card_lines = file.readlines()
            for card_line in card_lines:
                self.lineno += 1
                self.current_orig_line = card_line  # To keep indentation level in advanced card section.
                card_line = card_line.strip()
                if not card_line or is_comment(card_line):
                    continue  # Skip blank lines, comment lines, and old SMS cards
                try:
                    self.read_card(card_line)
                except Exception:
                    XmLog().instance.exception(f'Unknown error reading line: {self.lineno}: {card_line}')

        try:
            XmLog().instance.info('Preparing to read other model input files')
            self._build_grid_wkt()
            self.add_dependent_widgets()
        except Exception:
            XmLog().instance.exception('Unknown error occurred.')

        try:
            XmLog().instance.info('Reading geometry')
            self.read_geometry()
        except Exception:
            XmLog().instance.exception('Unknown error reading geometry.')

        try:
            XmLog().instance.info('Reading boundary conditions')
            self.read_boundary_conditions()
        except Exception:
            XmLog().instance.exception('Unknown error reading boundary conditions.')

        try:
            XmLog().instance.info('Building simulation data')

            self._add_h5_dependent_tables()
            self._add_advanced_cards()
            self.build_structure_coverage()
        except Exception:
            XmLog().instance.exception('Unknown error building simulation data.')

        try:
            self.add_save_points_coverage()
        except Exception:
            XmLog().instance.exception('Unknown error creating save points coverage.')

        XmLog().instance.info('Finished reading')

    def get_sms_data(self, query=None):
        """Get the path and filename of the .cmcards file. Get the simulation Context vertex id.

        Returns:
            bool: True on success, False if error encountered
        """
        self.xms_data = XmsData(query)

        # Read file is the XMDF solution
        self.filename = self.xms_data.file_to_read

        if not os.path.isfile(self.filename):
            XmLog().instance.error(f'.cmcards file not found - {self.filename}')
            return False

        # Add the simulation and its component
        comp_uuid = str(uuid.uuid4())
        self.sim_comp_dir = os.path.join(self.comp_dir, comp_uuid)
        os.makedirs(self.sim_comp_dir, exist_ok=True)
        sim_mainfile = os.path.join(self.sim_comp_dir, 'simulation_comp.nc')
        sim_comp = SimComponent(sim_mainfile)
        self.sim_data = sim_comp.data
        self.build_data['sim_comp'] = sim_comp
        return True

    def add_build_data(self):
        """Adds build data at the end of reading everything."""
        if not self.xms_data:
            return

        # Add the main simuation
        self.xms_data.add_simulation(self.build_data['sim_comp'], self.sim_data.output.attrs['SIMULATION_LABEL'])

        # Add the tidal simulation(s)
        for tidal_comp, tidal_name in self.build_data['tidal_sims']:
            tides_uuid = self.xms_data.add_simulation(tidal_comp, tidal_name, False)
            self.xms_data.link_item(tides_uuid)

        # Add the coverage(s)
        if self.build_data['bc_cov']:
            self.xms_data.add_coverage(self.build_data['bc_cov'], self.build_data['bc_cov_comp'])
            self.xms_data.link_item(self.build_data['bc_cov'].uuid)
        if self.build_data['save_pts_cov']:
            self.xms_data.add_coverage(self.build_data['save_pts_cov'], self.build_data['save_pts_cov_comp'])
            self.xms_data.link_item(self.build_data['save_pts_cov'].uuid)
        if self.build_data['structure_coverage']:
            component = self.build_data['structure_component']

            self.xms_data.add_coverage(self.build_data['structure_coverage'], component)
            self.xms_data.link_item(self.build_data['structure_coverage'].uuid)
        if self.build_data['activity_cov']:
            self.xms_data.add_generic_coverage(self.build_data['activity_cov'])
            self.xms_data.link_item(self.build_data['activity_cov'].m_cov.uuid)

        # Add the ugrids and datasets
        for ugrid in self.build_data['ugrids']:
            cogrid = read_grid_from_file(ugrid.cogrid_file)
            self.xms_data.add_grid(cogrid, ugrid.name, ugrid.projection)
        if self.quad_tree:
            self.xms_data.link_item(self.quad_tree.uuid)
        for dataset in self.build_data['datasets']:
            self.xms_data.add_dataset(dataset)

    def send_sms_data(self):
        """Send the imported data to SMS."""
        self.sim_data.commit()
        self.add_build_data()
        self.xms_data.send()

    def read(self, query=None):
        """Top-level function that triggers the read of the .cmcards file.

        A new simulation will be created in SMS. Any accompanying model-native files in the same directory will
        also be written.

        """
        try:
            if self.get_sms_data(query):
                self.read_files()
        except Exception:
            XmLog().instance.exception('Error importing CMS-Flow files:')
