"""This module defines data for the boundary conditions hidden component."""

# 1. Standard Python modules
import datetime
import os
from pathlib import Path

# 2. Third party modules
import numpy as np
import pandas as pd
from PySide2.QtCore import QDateTime, Qt
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.guipy.time_format import ISO_DATETIME_FORMAT, qdatetime_to_datetime

# 4. Local modules
from xms.cmsflow.components.id_files import UNINITIALIZED_COMP_ID

SUPPORTED_CONSTITUENTS = [
    'M2', 'S2', 'N2', 'K1', 'M4', 'O1', 'M6', 'MK3', 'S4', 'MN4', 'NU2', 'S6', 'MU2', '2N2', 'OO1', 'LAM2', 'S1', 'M1',
    'J1', 'MM', 'SSA', 'SA', 'MSF', 'MF', 'RHO', 'Q1', 'T2', 'R2', '2Q1', 'P1', '2SM2', 'M3', 'L2', '2MK3', 'K2', 'M8',
    'MS4'
]


class BCData(XarrayBase):
    """Manages data file for the hidden save arcs component."""
    def __init__(self, data_file: str | Path):
        """
        Initializes the data class.

        Args:
            data_file: The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        data_file = str(data_file)
        self._filename = data_file
        self._info = None
        self._arcs = None
        self._flow_curves = dict()
        self._wse_forcing_curves = dict()
        self._wse_offset_curves = dict()
        self._salinity_curves = dict()
        self._temperature_curves = dict()
        self._harmonic_tables = dict()
        self._tidal_tables = dict()
        self.deleted_comp_ids = set()  # Set this to be a list of deleted component ids in the coverage before vacuuming
        # Create the default file before calling super because we have our own attributes to write.
        created = self._get_default_datasets(data_file)
        super().__init__(data_file)
        if created:
            self.add_bc_arc_atts()
            self.commit()
        self._check_for_simple_migrations()

    @property
    def wse_forcing_geometry(self) -> str:
        """The UUID of the geometry for extracted WSEs."""
        return self.info.attrs.get('wse_forcing_geometry', '')

    @wse_forcing_geometry.setter
    def wse_forcing_geometry(self, value: str):
        """The UUID of the geometry for extracted WSEs."""
        self.info.attrs['wse_forcing_geometry'] = value

    @property
    def wse_forcing_wse_source(self) -> str:
        """The UUID of the dataset for extracted WSE elevation."""
        return self.info.attrs.get('wse_forcing_wse_source', '')

    @wse_forcing_wse_source.setter
    def wse_forcing_wse_source(self, value: str):
        """The UUID of the dataset for extracted WSE elevation."""
        self.info.attrs['wse_forcing_wse_source'] = value

    @property
    def wse_forcing_velocity_source(self) -> str:
        """The UUID of the dataset for extracted WSE velocity."""
        return self.info.attrs.get('wse_forcing_velocity_source', '')

    @wse_forcing_velocity_source.setter
    def wse_forcing_velocity_source(self, value: str):
        """The UUID of the dataset for extracted WSE velocity."""
        self.info.attrs['wse_forcing_velocity_source'] = value

    def _get_default_datasets(self, data_file):
        """Create default datasets if needed.

        Args:
            data_file (str): Name of the data file. If it doesn't exist, it will be created.

        Returns:
            Returns True if datasets were created.
        """
        if not os.path.exists(data_file) or not os.path.isfile(data_file):
            info = {
                'FILE_TYPE': 'CMSFLOW_BC',
                # 'VERSION': pkg_resources.get_distribution('cmsflow').version,
                'cov_uuid': '',
                'arc_display_uuid': '',
                'next_comp_id': 0,
                'proj_dir': os.path.dirname(os.environ.get('XMS_PYTHON_APP_PROJECT_PATH', '')),
            }
            self._info = xr.Dataset(attrs=info)

            arc_table = {
                'name': ('comp_id', np.array([], dtype=object)),
                'bc_type': ('comp_id', np.array([], dtype=object)),
                'flow_source': ('comp_id', np.array([], dtype=object)),
                'constant_flow': ('comp_id', np.array([], dtype=float)),
                'flow_curve': ('comp_id', np.array([], dtype=int)),
                'specify_inflow_direction': ('comp_id', np.array([], dtype=int)),
                'flow_direction': ('comp_id', np.array([], dtype=float)),
                'flow_conveyance': ('comp_id', np.array([], dtype=float)),
                'wse_source': ('comp_id', np.array([], dtype=object)),
                'wse_const': ('comp_id', np.array([], dtype=float)),
                'wse_forcing_curve': ('comp_id', np.array([], dtype=int)),
                'use_velocity': ('comp_id', np.array([], dtype=int)),
                'parent_cmsflow': ('comp_id', np.array([], dtype=object)),
                'parent_adcirc_14_uuid': ('comp_id', np.array([], dtype=object)),
                'parent_adcirc_14': ('comp_id', np.array([], dtype=object)),
                'parent_adcirc_solution_type': ('comp_id', np.array([], dtype=object)),
                'parent_adcirc_63': ('comp_id', np.array([], dtype=object)),
                'parent_adcirc_64': ('comp_id', np.array([], dtype=object)),
                'parent_adcirc_solution_wse': ('comp_id', np.array([], dtype=object)),
                'parent_adcirc_solution': ('comp_id', np.array([], dtype=object)),
                'parent_adcirc_start_time': ('comp_id', np.array([], dtype=object)),
                'wse_offset_type': ('comp_id', np.array([], dtype=object)),
                'wse_offset_const': ('comp_id', np.array([], dtype=float)),
                'wse_offset_curve': ('comp_id', np.array([], dtype=int)),
                'use_salinity_curve': ('comp_id', np.array([], dtype=int)),
                'salinity_curve': ('comp_id', np.array([], dtype=int)),
                'use_temperature_curve': ('comp_id', np.array([], dtype=int)),
                'temperature_curve': ('comp_id', np.array([], dtype=int)),
                'harmonic_table': ('comp_id', np.array([], dtype=int)),
                'tidal_table': ('comp_id', np.array([], dtype=int)),
            }
            coords = {'comp_id': np.array([], dtype=int)}
            self._arcs = xr.Dataset(data_vars=arc_table, coords=coords)
            self.commit()
            return True
        else:
            return False

    def _check_for_simple_migrations(self):
        """Check for easy fixes we can make to the file to avoid doing it during DMI project migration."""
        if 'parent_adcirc_solution_wse' not in self.arcs.variables:
            num_bc_arcs = self.arcs.parent_adcirc_solution.size
            self.arcs['parent_adcirc_solution_wse'] = ('comp_id', np.array([''] * num_bc_arcs, dtype=object))
        if 'parent_adcirc_14_uuid' not in self.arcs.variables:
            num_bc_arcs = self.arcs.parent_adcirc_solution.size
            self.arcs['parent_adcirc_14_uuid'] = ('comp_id', np.array([''] * num_bc_arcs, dtype=object))
        if 'extracted_grid_uuid' not in self.arcs.variables:
            num_bc_arcs = self.arcs.parent_adcirc_solution.size
            self.arcs['extracted_grid_uuid'] = ('comp_id', np.array([''] * num_bc_arcs, dtype=object))
        if 'extracted_wse_uuid' not in self.arcs.variables:
            num_bc_arcs = self.arcs.parent_adcirc_solution.size
            self.arcs['extracted_wse_uuid'] = ('comp_id', np.array([''] * num_bc_arcs, dtype=object))
        if 'extracted_velocity_uuid' not in self.arcs.variables:
            num_bc_arcs = self.arcs.parent_adcirc_solution.size
            self.arcs['extracted_velocity_uuid'] = ('comp_id', np.array([''] * num_bc_arcs, dtype=object))

        # Make sure the reftime string representation is locale agnostic.
        for i in range(self.arcs.sizes['comp_id']):
            dt_str = self.arcs.parent_adcirc_start_time[i].item()
            try:  # Try to parse using current locale
                # We used to use Qt's representation of an ISO date, but we want to be consistent with other libraries.
                qreftime = QDateTime.fromString(dt_str, Qt.ISODate)
                reftime = qdatetime_to_datetime(qreftime)
            except Exception:
                try:  # Try to parse using ISO format
                    reftime = datetime.datetime.strptime(dt_str, ISO_DATETIME_FORMAT)
                except Exception:  # Set default to 01/01/2000 epoch
                    reftime = datetime.datetime(2000, 1, 1)
            self.arcs.parent_adcirc_start_time[i] = reftime.strftime(ISO_DATETIME_FORMAT)

        version = self.info.attrs.get('adcirc_file_version', 0)
        self.info.attrs['adcirc_file_version'] = 1
        if version == 0:
            proj_dir = self.info.attrs['proj_dir']
            _make_absolute(proj_dir, self.arcs, 'parent_cmsflow')
            _make_absolute(proj_dir, self.arcs, 'parent_adcirc_14')
            _make_absolute(proj_dir, self.arcs, 'parent_adcirc_63')
            _make_absolute(proj_dir, self.arcs, 'parent_adcirc_64')

    @staticmethod
    def _clean_dataset_keys(dataset):
        """Removes '/' from the dataset keys for safe storage in h5.

        Args:
            dataset (xr.Dataset): The dataset that may have bad keys.

        Returns:
            A new xr.Dataset with sanitized keys.
        """
        column_names = dataset.keys()
        rename_dict = {}
        for col in column_names:
            if col.find('/') >= 0:
                rename_dict[col] = col.replace('/', '%slash%')
        if rename_dict:
            dataset = dataset.rename(rename_dict)
        return dataset

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming
        if self._arcs is not None:
            self._arcs.close()
            self._drop_h5_groups(['arcs'])
            self._arcs.to_netcdf(self._filename, group='arcs', mode='a')
        # write the flow curves
        for curve_id, data in self._flow_curves.items():
            grp = self._flow_curve_group_name(curve_id)
            self._drop_h5_groups([grp])
            if data is not None:
                self._clean_dataset_keys(data).to_netcdf(self._filename, group=grp, mode='a')
        # write the water surface elevation forcing curves
        for curve_id, data in self._wse_forcing_curves.items():
            grp = self._wse_forcing_curve_group_name(curve_id)
            self._drop_h5_groups([grp])
            if data is not None:
                data.to_netcdf(self._filename, group=grp, mode='a')
        # write the water surface elevation offset curves
        for curve_id, data in self._wse_offset_curves.items():
            grp = self._wse_offset_curve_group_name(curve_id)
            self._drop_h5_groups([grp])
            if data is not None:
                data.to_netcdf(self._filename, group=grp, mode='a')
        # write the salinity curves
        for curve_id, data in self._salinity_curves.items():
            grp = self._salinity_curve_group_name(curve_id)
            self._drop_h5_groups([grp])
            if data is not None:
                data.to_netcdf(self._filename, group=grp, mode='a')
        # write the temperature curves
        for curve_id, data in self._temperature_curves.items():
            grp = self._temperature_curve_group_name(curve_id)
            self._drop_h5_groups([grp])
            if data is not None:
                data.to_netcdf(self._filename, group=grp, mode='a')
        # write the harmonic tables
        for table_id, data in self._harmonic_tables.items():
            grp = self._harmonic_table_group_name(table_id)
            self._drop_h5_groups([grp])
            if data is not None:
                data.to_netcdf(self._filename, group=grp, mode='a')
        # write the tidal tables
        for table_id, data in self._tidal_tables.items():
            grp = self._tidal_table_group_name(table_id)
            self._drop_h5_groups([grp])
            if data is not None:
                data.to_netcdf(self._filename, group=grp, mode='a')

    def vacuum(self):
        """Rewrite all SimData to a new/wiped file to reclaim disk space.

        All BC datasets that need to be written to the file must be loaded into memory before calling this method.

        """
        if self._info is None:
            self._info = self.get_dataset('info', False)
        if self._arcs is None:
            self._arcs = self.get_dataset('arcs', False)
        for curve in self._arcs['flow_curve']:
            curve = int(curve)
            if curve > 0:
                grp = self._flow_curve_group_name(curve)
                if curve not in self._flow_curves or self._flow_curves[curve] is None:
                    self._flow_curves[curve] = self.get_dataset(grp, False)
        for curve in self._arcs['wse_forcing_curve']:
            curve = int(curve)
            if curve > 0:
                grp = self._wse_forcing_curve_group_name(curve)
                if curve not in self._wse_forcing_curves or self._wse_forcing_curves[curve] is None:
                    self._wse_forcing_curves[curve] = self.get_dataset(grp, False)
        for curve in self._arcs['wse_offset_curve']:
            curve = int(curve)
            if curve > 0:
                grp = self._wse_offset_curve_group_name(curve)
                if curve not in self._wse_offset_curves or self._wse_offset_curves[curve] is None:
                    self._wse_offset_curves[curve] = self.get_dataset(grp, False)
        for curve in self._arcs['salinity_curve']:
            curve = int(curve)
            if curve > 0:
                grp = self._salinity_curve_group_name(curve)
                if curve not in self._salinity_curves or self._salinity_curves[curve] is None:
                    self._salinity_curves[curve] = self.get_dataset(grp, False)
        for curve in self._arcs['temperature_curve']:
            curve = int(curve)
            if curve > 0:
                grp = self._temperature_curve_group_name(curve)
                if curve not in self._temperature_curves or self._temperature_curves[curve] is None:
                    self._temperature_curves[curve] = self.get_dataset(grp, False)
        for table in self._arcs['harmonic_table']:
            table = int(table)
            if table > 0:
                grp = self._harmonic_table_group_name(table)
                if table not in self._harmonic_tables or self._harmonic_tables[table] is None:
                    self._harmonic_tables[table] = self.get_dataset(grp, False)
        for table in self._arcs['tidal_table']:
            table = int(table)
            if table > 0:
                grp = self._tidal_table_group_name(table)
                if table not in self._tidal_tables or self._tidal_tables[table] is None:
                    self._tidal_tables[table] = self.get_dataset(grp, False)
        try:
            os.remove(self._filename)
        except Exception:
            pass
        self.commit()  # Rewrite all datasets

    @property
    def arcs(self):
        """Load the arcs dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the arcs datasets in the main file

        """
        if self._arcs is None:
            self._arcs = self.get_dataset('arcs', False)
        return self._arcs

    @arcs.setter
    def arcs(self, dset):
        """Setter for the arcs attribute."""
        if dset:
            self._arcs = dset

    def update_bc_arc(self, comp_id, new_atts):
        """Update the BC arc attributes of a BC arc.

        Args:
            comp_id (int): Component id of the BC arc to update
            new_atts (xarray.Dataset): The new attributes for the BC arc

        """
        comp_ids = list(self.arcs.comp_id.values)
        if comp_id in comp_ids:
            comp_idx = comp_ids.index(comp_id)
            var_names = list(self.arcs.keys())
            df = self.arcs.to_dataframe()
            for var_name in var_names:
                df.loc[comp_idx, var_name] = new_atts[var_name].item()
            self.arcs = df.to_xarray()

    def add_bc_arc_atts(self, dset=None) -> int:
        """Add the bc arc attribute dataset for a arc.

        Args:
            dset (xarray.Dataset): The attribute dataset to concatenate. If not provided, a new Dataset of default
                attributes will be generated.


        Returns:
            The newly generated component id
        """
        try:
            new_comp_id = self.info.attrs['next_comp_id'].item()
            self.info.attrs['next_comp_id'] += 1  # Increment the unique XMS component id.
            if dset is None:  # Generate a new default Dataset
                dset = self._get_new_arc_atts(new_comp_id)
            else:  # Update the component id of an existing Dataset
                dset.coords['comp_id'] = [new_comp_id for _ in dset.coords['comp_id']]
            self._arcs = xr.concat([self.arcs, dset], 'comp_id')
            self._arcs = self.arcs.to_dataframe().to_xarray()  # fix for bug 15117
            return new_comp_id
        except Exception:
            return UNINITIALIZED_COMP_ID

    @staticmethod
    def _get_new_arc_atts(comp_id):
        """Get a new dataset with default attributes for a bc arc.

        Args:
            comp_id (int): The unique XMS component id of the bc arc. If UNINITIALIZED_COMP_ID, a new one is
                generated.

        Returns:
            (xarray.Dataset): A new default dataset for a BC arc. Can later be concatenated to persistent dataset.

        """
        arc_table = {
            'name': ('comp_id', np.array(['(none selected)'], dtype=object)),
            'bc_type': ('comp_id', np.array(['Unassigned'], dtype=object)),
            'flow_source': ('comp_id', np.array(['Constant'], dtype=object)),
            'constant_flow': ('comp_id', np.array([0.0], dtype=float)),
            'flow_curve': ('comp_id', np.array([0], dtype=int)),
            'specify_inflow_direction': ('comp_id', np.array([0], dtype=int)),
            'flow_direction': ('comp_id', np.array([0.0], dtype=float)),
            'flow_conveyance': ('comp_id', np.array([0.667], dtype=float)),
            'wse_source': ('comp_id', np.array(['Constant'], dtype=object)),
            'wse_const': ('comp_id', np.array([0.0], dtype=float)),
            'wse_forcing_curve': ('comp_id', np.array([0], dtype=int)),
            'use_velocity': ('comp_id', np.array([0], dtype=int)),
            'parent_cmsflow': ('comp_id', np.array(['(none selected)'], dtype=object)),
            'parent_adcirc_14_uuid': ('comp_id', np.array([''], dtype=object)),
            'parent_adcirc_14': ('comp_id', np.array(['(none selected)'], dtype=object)),
            'parent_adcirc_solution_type': ('comp_id', np.array(['ASCII'], dtype=object)),
            'parent_adcirc_63': ('comp_id', np.array(['(none selected)'], dtype=object)),
            'parent_adcirc_64': ('comp_id', np.array(['(none selected)'], dtype=object)),
            'parent_adcirc_solution': ('comp_id', np.array(['(none selected)'], dtype=object)),
            'parent_adcirc_solution_wse': ('comp_id', np.array(['(none selected)'], dtype=object)),
            'parent_adcirc_start_time': ('comp_id', np.array(['2000-01-01 00:00:00'], dtype=object)),
            'extracted_grid_uuid': ('comp_id', np.array([''], dtype=object)),
            'extracted_wse_uuid': ('comp_id', np.array([''], dtype=object)),
            'extracted_velocity_uuid': ('comp_id', np.array([''], dtype=object)),
            'wse_offset_type': ('comp_id', np.array(['Constant'], dtype=object)),
            'wse_offset_const': ('comp_id', np.array([0.0], dtype=float)),
            'wse_offset_curve': ('comp_id', np.array([0], dtype=int)),
            'use_salinity_curve': ('comp_id', np.array([0], dtype=int)),
            'salinity_curve': ('comp_id', np.array([0], dtype=int)),
            'use_temperature_curve': ('comp_id', np.array([0], dtype=int)),
            'temperature_curve': ('comp_id', np.array([0], dtype=int)),
            'harmonic_table': ('comp_id', np.array([0], dtype=int)),
            'tidal_table': ('comp_id', np.array([0], dtype=int)),
        }
        coords = {'comp_id': [comp_id]}
        ds = xr.Dataset(data_vars=arc_table, coords=coords)
        return ds

    def flow_curve_from_id(self, curve_id, create_default=True):
        """Gets the flow curve from the curve id.

        Args:
            curve_id (int): curve id
            create_default (bool): True if a default curve should be created if none is found.

        Returns:
            xarray.Dataset: The flow curve dataset
        """
        return self._curve_from_dictionary(
            curve_id, self._flow_curves, self._default_flow_curve, self._flow_curve_group_name, create_default
        )

    def wse_forcing_curve_from_id(self, curve_id, create_default=True):
        """Gets the wse forcing curve from the curve id.

        Args:
            curve_id (int): curve id
            create_default (bool): True if a default curve should be created if none is found.

        Returns:
            xarray.Dataset: The wse forcing curve dataset
        """
        return self._curve_from_dictionary(
            curve_id, self._wse_forcing_curves, self._default_wse_forcing_curve, self._wse_forcing_curve_group_name,
            create_default
        )

    def wse_offset_curve_from_id(self, curve_id, create_default=True):
        """Gets the wse offset curve from the curve id.

        Args:
            curve_id (int): curve id
            create_default (bool): True if a default curve should be created if none is found.

        Returns:
            xarray.Dataset: The wse offset curve dataset
        """
        return self._curve_from_dictionary(
            curve_id, self._wse_offset_curves, self._default_wse_offset_curve, self._wse_offset_curve_group_name,
            create_default
        )

    def salinity_curve_from_id(self, curve_id, create_default=True):
        """Gets the salinity curve from the curve id.

        Args:
            curve_id (int): curve id
            create_default (bool): True if a default curve should be created if none is found.

        Returns:
            xarray.Dataset: The salinity curve dataset
        """
        return self._curve_from_dictionary(
            curve_id, self._salinity_curves, self._default_salinity_curve, self._salinity_curve_group_name,
            create_default
        )

    def temperature_curve_from_id(self, curve_id, create_default=True):
        """Gets the temperature curve from the curve id.

        Args:
            curve_id (int): curve id
            create_default (bool): True if a default curve should be created if none is found.

        Returns:
            xarray.Dataset: The temperature curve dataset
        """
        return self._curve_from_dictionary(
            curve_id, self._temperature_curves, self._default_temperature_curve, self._temperature_curve_group_name,
            create_default
        )

    def harmonic_table_from_id(self, table_id, create_default=True):
        """Gets the harmonic table from the table id.

        Args:
            table_id (int): table id
            create_default (bool): True if a default table should be created if none is found.

        Returns:
            xarray.Dataset: The harmonic table dataset
        """
        return self._curve_from_dictionary(
            table_id, self._harmonic_tables, self.default_harmonic_table, self._harmonic_table_group_name,
            create_default
        )

    def tidal_table_from_id(self, table_id, create_default=True):
        """Gets the tidal table from the table id.

        Args:
            table_id (int): table id
            create_default (bool): True if a default table should be created if none is found.

        Returns:
            xarray.Dataset: The tidal table dataset
        """
        return self._curve_from_dictionary(
            table_id, self._tidal_tables, self.default_tidal_table, self._tidal_table_group_name, create_default
        )

    def set_flow_forcing_curve(self, curve_id, curve):
        """Set the flow forcing curve.

        Args:
            curve_id (int): The id of the curve.
            curve (xarray.Dataset): The dataset representing the curve.
        """
        self._flow_curves[curve_id] = curve

    def set_wse_forcing_curve(self, curve_id, curve):
        """Set the wse forcing curve.

        Args:
            curve_id (int): The id of the curve.
            curve (xarray.Dataset): The dataset representing the curve.
        """
        self._wse_forcing_curves[curve_id] = curve

    def set_wse_offset_curve(self, curve_id, curve):
        """Set the wse offset curve.

        Args:
            curve_id (int): The id of the curve.
            curve (xarray.Dataset): The dataset representing the curve.
        """
        self._wse_offset_curves[curve_id] = curve

    def set_salinity_curve(self, curve_id, curve):
        """Set the salinity curve.

        Args:
            curve_id (int): The id of the curve.
            curve (xarray.Dataset): The dataset representing the curve.
        """
        self._salinity_curves[curve_id] = curve

    def set_temperature_curve(self, curve_id, curve):
        """Set the temperature curve.

        Args:
            curve_id (int): The id of the curve.
            curve (xarray.Dataset): The dataset representing the curve.
        """
        self._temperature_curves[curve_id] = curve

    def set_harmonic_table(self, table_id, table):
        """Set the harmonic table.

        Args:
            table_id (int): The id of the table.
            table (xarray.Dataset): The dataset representing the table.
        """
        self._harmonic_tables[table_id] = table

    def set_tidal_table(self, table_id, table):
        """Set the tidal table.

        Args:
            table_id (int): The id of the table.
            table (xarray.Dataset): The dataset representing the table.
        """
        self._tidal_tables[table_id] = table

    def _curve_from_dictionary(
        self, curve_id, curve_dict, default_curve_method, curve_group_name_method, create_default=True
    ):
        """Gets the curve from the curve id.

        Args:
            curve_id (int): curve id
            curve_dict (dict): A dictionary of curve ids to xr.Dataset
            default_curve_method (method): The method to call to get a default curve.
            curve_group_name_method (method): The method ot call to get the name of the curve group.
            create_default (bool): True if a default curve should be created if none is found.

        Returns:
            xarray.Dataset: The curve dataset
        """
        curve_id = int(curve_id)
        # we don't want it added to the dictionary if its id = 1
        if curve_id == -1:
            return default_curve_method()
        if curve_id not in curve_dict:
            # load from the file if the curve exists
            grp = curve_group_name_method(curve_id)
            dset = self.get_dataset(grp, False)
            if dset:
                column_names = dset.keys()
                rename_dict = {}
                for col in column_names:
                    if col.find('%slash%') >= 0:
                        rename_dict[col] = col.replace('%slash%', '/')
                if rename_dict:
                    dset = dset.rename(rename_dict)
            curve_dict[curve_id] = dset
            if curve_dict[curve_id] is None and create_default:
                # create a default curve
                curve_dict[curve_id] = default_curve_method()
        return curve_dict[curve_id]

    @staticmethod
    def _flow_curve_group_name(curve_id):
        """Gets the h5 group where the curve is stored.

        Args:
            curve_id (int): curve id

        Returns:
            (str): The h5 group name
        """
        return f'flow_curves/{curve_id}'

    @staticmethod
    def _default_flow_curve():
        """Creates a xarray.Dataset for a flow curve."""
        default_data = {
            'Time (hrs)': [0.0],
            'Flow (m^3/s Total, not per cell)': [0.0],
        }
        return pd.DataFrame(default_data).to_xarray()

    @staticmethod
    def _wse_forcing_curve_group_name(curve_id):
        """Gets the h5 group where the curve is stored.

        Args:
            curve_id (int): curve id

        Returns:
            (str): The h5 group name
        """
        return f'wse_forcing_curves/{curve_id}'

    @staticmethod
    def _default_wse_forcing_curve():
        """Creates a xarray.Dataset for a water surface elevation curve."""
        default_data = {
            'Time (hrs)': [0.0],
            'WSE (m)': [0.0],
        }
        return pd.DataFrame(default_data).to_xarray()

    @staticmethod
    def _wse_offset_curve_group_name(curve_id):
        """Gets the h5 group where the curve is stored.

        Args:
            curve_id (int): curve id

        Returns:
            (str): The h5 group name
        """
        return f'wse_offset_curves/{curve_id}'

    @staticmethod
    def _default_wse_offset_curve():
        """Creates a xarray.Dataset for a water surface elevation offset curve."""
        default_data = {
            'Time (hrs)': [0.0],
            'WSE Offset (m)': [0.0],
        }
        return pd.DataFrame(default_data).to_xarray()

    @staticmethod
    def _salinity_curve_group_name(curve_id):
        """Gets the h5 group where the curve is stored.

        Args:
            curve_id (int): curve id

        Returns:
            (str): The h5 group name
        """
        return f'salinity_curves/{curve_id}'

    @staticmethod
    def _default_salinity_curve():
        """Creates a xarray.Dataset for a salinity curve."""
        default_data = {
            'Time (hrs)': [0.0],
            'Salinity (ppt)': [0.0],
        }
        return pd.DataFrame(default_data).to_xarray()

    @staticmethod
    def _temperature_curve_group_name(curve_id):
        """Gets the h5 group where the curve is stored.

        Args:
            curve_id (int): curve id

        Returns:
            (str): The h5 group name
        """
        return f'temperature_curves/{curve_id}'

    @staticmethod
    def _default_temperature_curve():
        """Creates a xarray.Dataset for a temperature curve."""
        default_data = {
            'Time (hrs)': [0.0],
            'Temperature (dec C)': [0.0],
        }
        return pd.DataFrame(default_data).to_xarray()

    @staticmethod
    def _harmonic_table_group_name(table_id):
        """Gets the h5 group where the table is stored.

        Args:
            table_id (int): table id

        Returns:
            (str): The h5 group name
        """
        return f'harmonic_tables/{table_id}'

    @staticmethod
    def default_harmonic_table():
        """Creates a xarray.Dataset for a harmonic table."""
        default_data = {
            'speed': [0.0],
            'amplitude': [0.0],
            'phase': [0.0],
        }
        return pd.DataFrame(default_data).to_xarray()

    @staticmethod
    def _tidal_table_group_name(table_id):
        """Gets the h5 group where the table is stored.

        Args:
            table_id (int): table id

        Returns:
            (str): The h5 group name
        """
        return f'tidal_tables/{table_id}'

    @staticmethod
    def default_tidal_table():
        """Creates a xarray.Dataset for a tidal table."""
        default_data = {
            'constituent': ['M2'],
            'amplitude': [0.0],
            'phase': [0.0],
        }
        return pd.DataFrame(default_data).to_xarray()

    def arc_uses_extracted_wse_forcing(self, component_id: int) -> bool:
        """
        Check whether an arc uses Extracted WSE forcing.

        Args:
            component_id: Component ID of the arc to check.

        Returns:
            Whether the arc uses Extracted WSE forcing.
        """
        attributes = self.arcs.loc[{'comp_id': component_id}]
        if attributes['bc_type'].item() == 'WSE-forcing' and attributes['wse_source'].item() == 'Extracted':
            return True
        return False

    def clean_up_old_comp_ids(self, used_comp_ids):
        """Cleans up old component IDs from the 'arcs' attribute of the BCData class.

        Args:
            used_comp_ids (list): A list of component IDs to keep.
        """
        mask = self.arcs.comp_id.isin(used_comp_ids)
        self.arcs = self.arcs.where(mask, drop=True)


def _make_absolute(base, container, key):
    # xarray assigns a maximum string length to its arrays. If prepending the directory makes the total path longer
    # than whatever maximum xarray picked for this array, it just silently truncates the string. We force xarray to
    # forget the length here to prevent that.
    container[key] = container[key].astype(object)
    values = container[key]
    for i in range(len(values)):
        value = values[i].item()
        if value == '' or value == '(none selected)':
            continue
        if not os.path.isabs(value):
            value = os.path.join(base, value)
        value = os.path.normpath(value)
        values[i] = value
