"""Map Boundary Conditions coverage locations and attributes to the CMS-Flow domain."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import (read_display_options_from_json, write_display_options_to_json)
from xms.components.display.display_options_io import write_display_option_line_locations
from xms.data_objects.parameters import Component
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.line_style import LineStyle
from xms.guipy.data.target_type import TargetType
from xms.snap.snap_exterior_arc import SnapExteriorArc

# 4. Local modules
from xms.cmsflow.components.mapped_bc_component import MappedBcComponent


class BcMapper:
    """Class for mapping boundary conditions coverage to a quadtree for CMS-Flow."""
    def __init__(self, coverage_mapper, generate_snap):
        """Constructor."""
        self._snap_arc = SnapExteriorArc()
        if coverage_mapper._small_quadtree:
            self._snap_arc.set_grid(grid=coverage_mapper._small_quadtree, target_cells=True)
        else:
            self._snap_arc.set_grid(grid=coverage_mapper._quadtree, target_cells=True)
        self._arc_to_grid_points = {}
        self._logger = coverage_mapper._logger
        self._wkt = coverage_mapper._wkt
        self._generate_snap = generate_snap
        self._bc_coverage = coverage_mapper._bc_cov
        self._bc_component = coverage_mapper._bc_comp
        self.bc_mapped_comp_uuid = None
        self.bc_mapped_comp_display_uuid = None
        self._new_comp_unique_name = 'Mapped_Bc_Component'
        self._bc_component_file = self._bc_component.main_file

    @property
    def snapper(self):
        """Gets the snapper used for boundary condition arcs.

        Returns:
            The arc snapper used for boundary condition arcs.
        """
        return self._snap_arc

    def do_map(self):
        """Creates the mapped boundary conditions component."""
        self._get_grid_points_from_arcs()
        if self._generate_snap:
            self._create_component_folder_and_copy_display_options()
            self._create_drawing()
            # Create the data_objects component
            do_comp = Component(
                main_file=self._comp_main_file,
                name=f'Snapped {self._bc_coverage.name} display',
                model_name='CMS-Flow',
                unique_name=self._new_comp_unique_name,
                comp_uuid=os.path.basename(os.path.dirname(self._comp_main_file))
            )
            comp = MappedBcComponent(self._comp_main_file)
            return do_comp, comp
        return None, None  # pragma: no cover

    def _create_component_folder_and_copy_display_options(self):
        """Creates the folder for the mapped bc component and copies the display options from the bc coverage."""
        if self.bc_mapped_comp_uuid is None:
            comp_uuid = str(uuid.uuid4())  # pragma: no cover
        else:
            comp_uuid = self.bc_mapped_comp_uuid
        self._logger.info('Creating component folder')
        bc_comp_path = os.path.dirname(self._bc_component_file)
        self._comp_path = os.path.join(os.path.dirname(bc_comp_path), comp_uuid)

        if os.path.exists(self._comp_path):
            shutil.rmtree(self._comp_path)  # pragma: no cover
        os.mkdir(self._comp_path)

        bc_comp_display_file = os.path.join(bc_comp_path, 'bc_display_options.json')
        comp_display_file = os.path.join(self._comp_path, 'bc_display_options.json')
        if os.path.isfile(bc_comp_display_file):
            shutil.copyfile(bc_comp_display_file, comp_display_file)
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            json_dict = read_display_options_from_json(comp_display_file)
            if self.bc_mapped_comp_display_uuid is None:
                json_dict['uuid'] = str(uuid.uuid4())  # pragma: no cover
            else:
                json_dict['uuid'] = self.bc_mapped_comp_display_uuid
            json_dict['comp_uuid'] = comp_uuid
            json_dict['is_ids'] = 0
            categories.from_dict(json_dict)
            categories.projection = {'wkt': self._wkt}

            # Set all snapped arcs to be dashed and thick by default. Keep current color.
            for category in categories.categories:
                category.options.style = LineStyle.DASHEDLINE
                category.options.width = 4
                category.label_on = False

            write_display_options_to_json(comp_display_file, categories)
            self._comp_main_file = comp_display_file
        else:
            self._logger.info('Could not find bc_display_options.json file')  # pragma: no cover

    def _create_drawing(self):
        """Uses cell ids to get cell point coords to draw lines for bc mapped to cells."""
        for disp_name, arcs_list in self._arc_to_grid_points.items():
            filename = os.path.join(self._comp_path, f'{disp_name}.display_ids')
            write_display_option_line_locations(filename, arcs_list)

    def _get_grid_points_from_arcs(self):
        """Uses xmssnap to get the points from arcs."""
        self._logger.info('Boundary conditions coverage to quadtree.')
        bc_data = self._bc_component.data
        df = bc_data.arcs.to_dataframe()
        bc_type_to_display = {
            'Unassigned': 'unassigned_arcs',
            'Flow rate-forcing': 'flow_arcs',
            'WSE-forcing': 'wse_arcs'
        }
        comp_id_to_bc_id_name = {
            int(comp_id): bc_type_to_display[bc_type]
            for comp_id, bc_type in zip(list(df.index.values), df['bc_type'])
        }
        arcs = self._bc_coverage.arcs
        for arc in arcs:
            snap_output = self._snap_arc.get_snapped_points(arc)
            if snap_output:
                points = [item for sublist in snap_output['location'] for item in sublist]

                comp_id = self._bc_component.get_comp_id(TargetType.arc, arc.id)
                if comp_id is None or comp_id < 0:
                    display_name = 'unassigned_arcs'
                else:
                    display_name = comp_id_to_bc_id_name[comp_id]

                if display_name not in self._arc_to_grid_points:
                    self._arc_to_grid_points[display_name] = []
                self._arc_to_grid_points[display_name].append(points)
