"""Class for managing interprocess communication with XMS."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import cached_property

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file

# 4. Local modules
from xms.cmswave.components.structures_component import StructuresComponent


class XmsData:
    """Class for managing interprocess communication with XMS."""
    def __init__(self, query=None):
        """Constructor."""
        self._query = query
        self._sim_uuid = None
        self._sim_item = None
        self._do_ugrid = None
        self._cogrid = None

    @property
    def query(self):
        """Returns the xmsapi interprocess communication object.

        Notes:
            If constructed in a component runner script, Query should be passed in on construction. If constructed in
            an export or import script, creation of the Query can happen here. Only one Query/XmsData object should be
            constructed per script.
        """
        if self._query is None:
            self._query = Query()
        return self._query

    @property
    def sim_uuid(self):
        """Returns the simulation's UUID (not it's component's UUID)."""
        if self._sim_uuid is None:
            sim_item = self.sim_item
            self._sim_uuid = sim_item.uuid
        return self._sim_uuid

    @property
    def sim_item(self):
        """Returns the simulation's TreeNode."""
        if self._sim_item is None:
            sim_uuid = self.query.current_item_uuid()
            sim_item = tree_util.find_tree_node_by_uuid(self.query.project_tree, sim_uuid)
            if sim_item is None:
                sim_uuid = self.query.parent_item_uuid()
                sim_item = tree_util.find_tree_node_by_uuid(self.query.project_tree, sim_uuid)
            self._sim_item = sim_item
        return self._sim_item

    @cached_property
    def do_ugrid(self):
        """Retrieves the simulation's data_objects UGrid."""
        if self._do_ugrid is None:
            xms_types = ['TI_CGRID2D_PTR', 'TI_MESH2D_PTR', 'TI_UGRID_PTR']
            tree_node = tree_util.descendants_of_type(
                self.sim_item, allow_pointers=True, xms_types=xms_types, only_first=True
            )
            if tree_node is None:
                return None
            self._do_ugrid = self.query.item_with_uuid(tree_node.uuid)
            if not self._do_ugrid:
                raise RuntimeError('Unable to retrieve domain mesh from SMS.')
        return self._do_ugrid

    @cached_property
    def cogrid(self):
        """Retrieves the simulation's domain CoGrid."""
        if self._cogrid is None:
            do_ugrid = self.do_ugrid
            self._cogrid = read_grid_from_file(do_ugrid.cogrid_file)
        return self._cogrid

    @cached_property
    def _structures_coverage(self):
        """Returns the simulation's structures data_objects coverage and component if it has one else None."""
        tree_node = tree_util.descendants_of_type(
            self.sim_item,
            coverage_type='Structures',
            model_name='CMS-Wave',
            recurse=False,
            allow_pointers=True,
            only_first=True
        )
        if tree_node is None:
            return None, None
        coverage = self.query.item_with_uuid(tree_node.uuid)
        do_component = self.query.item_with_uuid(
            tree_node.uuid, model_name='CMS-Wave', unique_name='StructuresComponent'
        )
        component = StructuresComponent(do_component.main_file)
        self.query.load_component_ids(component, polygons=True)
        return coverage, component

    @property
    def structures_coverage(self):
        """Helper to get the structures coverage."""
        coverage, _component = self._structures_coverage
        return coverage

    @property
    def structures_component(self):
        """Helper to get the structures component."""
        _coverage, component = self._structures_coverage
        return component
