"""Worker thread for the solution importer."""

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.data_objects.parameters import Point
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.cmswave.file_io.solution_reader import SolutionReader


class SolutionImportWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing CMS-Wave solutions."""
    def __init__(self, filenames, parent, query):
        """Construct the worker.

        Args:
            filenames (:obj:`list[str]`): The solution files to read. Pass empty list if a single solution file read.
            parent (:obj:`Optional[QWidget]`): Parent of the QThread, probably the hidden parent dialog created by
                XMS. Pass None if a single file solution import.
            query (:obj:`Optional[Query]`): The XMS Query. Pass None if this is a single solution read
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._query = query
        self._filenames = filenames
        self._reader = None

    def _do_work(self):
        """Thread runner that imports the fort.15."""
        try:
            self._reader = SolutionReader()
            self._reader.win_cont = self.parent()
            if self._query is None:  # Setup for a single solution file read
                self._reader.setup_query()
                self._filenames = [self._reader.query.read_file]
            else:  # Set the passed in Query if a simulation run solution load
                self._reader.query = self._query

            for filename in self._filenames:
                self._reader.filename = filename
                self._reader.read()
        except Exception:
            logging.getLogger('xms.cmswave').exception('Error(s) encountered while reading CMS-Wave solution files.')

    def send(self):
        """Send data to XMS if import is successful."""
        # Only send if this is a single solution import. If a simulation run solution read. Query.send() will get called
        # in xmscomponents.
        if self._reader and self._reader.query and self._reader.read_data and not self._query:
            self._reader.query.send()

    def get_spectral_grid_info(self):
        """Get the spectral grid point info and orientation."""
        if self._reader.eng_reader is None:
            return
        if self._reader.eng_reader.get_point_info is False:
            return
        # prompt the user for grid point info and orientation
        from xms.gmi.data.generic_model import GenericModel
        gm = GenericModel()
        pp = gm.point_parameters
        gp = pp.add_group(group_name='grid_props', label='Grid Properties', is_active=True)
        gp.add_float(name='grid_origin_x', label='Spectra Location X', default=0.0)
        gp.add_float(name='grid_origin_y', label='Spectra Location y', default=0.0)
        gp.add_float(name='grid_rotation', label='Grid Orientation Angle (CCW from East)', default=0.0)
        dlg_name = 'xms.cmswave.feedback.solution_import_runner.grid_properties'
        from xms.gmi.gui.group_set_dialog import GroupSetDialog
        dlg = GroupSetDialog(
            parent=self.parent(),
            section=pp,
            get_curve=None,
            add_curve=None,
            is_interior=False,
            dlg_name=dlg_name,
            window_title='Spectra Grid Info',
            multi_select_message='',
            show_groups=False
        )
        if dlg.exec():
            # update the spec_grid with the new pt location
            spec_grid = self._reader.eng_reader.spec_grids['0 0'][0]
            x = gp.parameter('grid_origin_x').value
            y = gp.parameter('grid_origin_y').value
            pt_loc = f"{x} {y}"
            spec_pt = Point(x, y)
            spec_pt.id = 1
            # create the new spec_grid
            self._reader.eng_reader.spec_grids = {}
            self._reader.eng_reader.spec_grids[pt_loc] = (spec_grid, spec_pt)
            # build the spectral coverage (delayed from read_spectral_vals)
            self._reader.eng_reader.build_spec_cov()
            spec_cov = self._reader.eng_reader.spec_cov
            spec_cov.m_cov.name = self._reader.eng_reader.cov_name
            # update the grid orientation
            spec_cov.m_points[1][0].m_rectGrid.angle = gp.parameter('grid_rotation').value
            self._reader.query.add_coverage(spec_cov)
            self._reader.query.read_data = True
            self._reader.read_data = True
            self._reader.output_coverages.append(spec_cov)


def import_solution_with_feedback(filenames, parent, query):
    """Driver for the solution import script with a feedback dialog.

    Args:
        filenames (:obj:`list[str]`): The solution files to read. Pass empty list if a single solution file read.
        parent (:obj:`Optional[QWidget]`): Parent of the QThread, probably the hidden parent dialog created by XMS.
            Pass None if a single file solution import.
        query (:obj:`Optional[Query]`): The XMS Query. Pass None if this is a single solution read
    """
    ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    main_hwnd = None
    if query is None:  # This is a single file solution read
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        parent = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, parent)  # Keep the timer in scope

    worker = SolutionImportWorkerThread(filenames, parent, query)
    display_text = {
        'title': 'Reading CMS-Wave Solution',
        'working_prompt': 'Reading CMS-Wave solution files. Please wait...',
        'error_prompt': 'Error(s) encountered reading CMS-Wave solution. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered reading CMS-Wave solution. Review log output for more details.',
        'success_prompt': 'Successfully read CMS-Wave solution.',
        'note': '',
        'auto_load': 'Auto load simulation into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, parent)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.get_spectral_grid_info()
        worker.send()  # Send the data to SMS
    if main_hwnd is not None:
        win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
