"""Qt delegate with a validator for edit fields."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QColor
from PySide2.QtWidgets import QStyledItemDelegate

# 3. Aquaveo modules

# 4. Local modules
from xms.cmswave.data.structures_data import StructuresData


class ModDisplayDelegate(QStyledItemDelegate):
    """A mod display delegate."""
    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent)

    def _get_mod_text(self, index):
        """Gets the text for the mod.

        Args:
            index (:obj:`QModelIndex`): The index

        Returns:
            (:obj:`str`)
        """
        struct_type = index.model().index(index.row(), StructuresData.COL_CBX_TYPE).data(Qt.DisplayRole)
        mod_text = 'n/a'
        if struct_type == 'Bathymetry modification':
            mod_text = 'Depth'
        elif struct_type == 'Floating breakwater':
            mod_text = 'Draft'
        elif struct_type == 'Highly permeable' or \
                struct_type == 'Semi-permeable':
            mod_text = 'Porous layer thickness'
        elif struct_type == 'Rubble-mound' or struct_type == 'Wall breakwater' or struct_type == 'Wave runup':
            mod_text = 'Elevation'
        return mod_text

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (:obj:`QPainter`): The painter.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if not index.isValid():
            return
        flags = index.flags()
        if not (flags & Qt.ItemIsEnabled):
            painter.fillRect(option.rect, QColor(240, 240, 240))
        mod_text = self._get_mod_text(index)
        flags = index.flags()
        flags |= Qt.AlignHCenter
        flags |= Qt.AlignVCenter
        flags = flags & (~Qt.AlignRight)
        painter.drawText(option.rect, flags, mod_text)
