"""Runs XMS DMI component ActionRequest events."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import argparse
import sys

# 2. Third party modules
# from PySide2.QtWidgets import QApplication

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.guipy.dialogs.dialog_util import ensure_qapplication_exists
from xms.guipy.dialogs.xms_parent_dlg import get_parent_window_container

# 4. Local modules
from xms.components.bases.component_base import ComponentBase
from xms.components.display import windows_gui as win_gui


def run_runner_main(module_name, class_name, method_name, main_file, modal_id, main_id):
    """Runs the XMS DMI component ActionRequest event.

    Args:
        module_name (str): Import module path of the component
        class_name (str): Class name of the component
        method_name (str): Name of the component method to call
        main_file (str): Path to the component instance's main file
        modal_id (int): HWND of the parent dialog
        main_id (int): HWND of the XMS main window
    """
    query = Query()

    params = []
    result = query._impl._instance.Get("parameters")
    if result["parameters"] and result["parameters"][0]:
        params = result["parameters"]
    disp_opts = None
    messages = None
    action_requests = None
    comp_uuid = None
    comp_ids = None
    att_files = None
    # Disable Query.send() calls during the component action method.
    query._impl._instance.SetAllowSend(False)  # Only exposed on C++ exposed object

    try:
        mod = __import__(module_name, fromlist=[class_name])
        klass = getattr(mod, class_name)
        # This script handles both components (which need main-files) and run managers (which don't). Prior to this
        # check, run managers had to have a dummy main-file to keep this script happy. Now we just try constructing
        # it with a main-file and fall back to no parameters if that fails.
        try:
            class_instance = klass(main_file)
        except TypeError:
            class_instance = klass()
        method_to_invoke = getattr(class_instance, method_name)

        if modal_id is None:
            # Most methods don't need any messages or requests. If they return None, assume no follow-up activity.
            messages_and_requests = method_to_invoke(query, params)
            if method_name == 'write_shapefile_atts':  # Get shapefile attribute filenames
                get_att_files_method = class_instance.get_shapefile_att_files
                att_files = get_att_files_method()
        else:
            # QApplication(sys.argv)
            # The line above resulted in the following error:
            # "RuntimeError: Please destroy the QApplication singleton before creating a new QApplication instance."
            # I am replacing it with ensure_qapplication_exists():
            ensure_qapplication_exists()

            int_hwnd = int(modal_id)
            win_cont = get_parent_window_container(int_hwnd)
            xms_mainframe_id = 0
            if main_id:
                xms_mainframe_id = int(main_id)
            _ = win_gui.create_and_connect_raise_timer(xms_mainframe_id, win_cont)  # Keep the timer in scope
            # Most methods don't need any messages or requests. If they return None, assume no follow-up activity.
            messages_and_requests = method_to_invoke(query, params, win_cont) or ([], [])
            win_gui.raise_main_xms_window(xms_mainframe_id)

        if messages_and_requests:  # Old-style methods return a tuple of messages and requests.
            messages, action_requests = messages_and_requests
        elif hasattr(klass, 'messages') and hasattr(klass, 'requests'):  # ComponentWithMenusBase stores them as attrs
            messages = klass.messages
            action_requests = klass.requests
        # If the method didn't return anything, assume it has nothing else to do.

        disp_opts = []
        comp_uuid = None
        comp_ids = []
        if isinstance(class_instance, ComponentBase):
            disp_opts = class_instance.get_display_options()
            comp_uuid, comp_ids = class_instance.get_component_coverage_ids()

    except Exception as ex:
        XmEnv.report_error(ex)

    arg_list = []
    if action_requests:
        # Unwrap the pure Python ActionRequests
        action_requests = [action._instance for action in action_requests]
        arg_list.append({"actions": action_requests})
    if messages:
        arg_list.append({"messages": messages})
    if disp_opts:
        arg_list.append({"display_options": disp_opts})
    if att_files:
        arg_list.append({"shapefile_atts": att_files})
    if comp_uuid and comp_ids:
        arg_list.append({"component_coverage_ids": [comp_uuid, comp_ids]})
    if arg_list:
        # Look for a good place to add UI data. Builder will first look for UI
        # keywords on root build vertices. If there are none, it will look on
        # the reference roots. Prefer the build vertices. Furthermore, look for
        # the last root so any build operations will be performed before processing
        # ActionRequests.
        ultimate_root = query._impl._instance.GetContext().GetRootInstance()
        penultimate_roots = query.get_children_ids(ultimate_root)
        penultimate_edges = query.get_children_names(ultimate_root)

        add_vertex = None
        last_build_vertex = None
        for idx, edge in zip(penultimate_roots, penultimate_edges):
            if edge.startswith('Reference'):  # Keep track of the last reference vertex
                add_vertex = idx
            elif edge.startswith('Build'):
                last_build_vertex = idx  # Keep track of the last build vertex

        # If there are any build edges from the ultimate root, use the last one.
        if last_build_vertex is not None:
            add_vertex = last_build_vertex
        if add_vertex is not None:
            query._impl._instance.Set(arg_list, add_vertex)
        else:  # If no root instances, throw on the current place marks and hope for the best.
            query._impl._instance.Set(arg_list)
    # Re-enable Query.send()
    query._impl._instance.SetAllowSend(True)
    query.send(True)
    sys.exit()


if __name__ == "__main__":
    arguments = argparse.ArgumentParser(description="Component method runner.")
    arguments.add_argument(dest='script', type=str, help='script to run')
    arguments.add_argument(dest='module_name', type=str, help='module of the method to run')
    arguments.add_argument(dest='class_name', type=str, help='class of the method to run')
    arguments.add_argument(dest='method_name', type=str, help='method to run')
    arguments.add_argument(dest='file', type=str, help='main file for the component')
    arguments.add_argument(dest='modal_id', type=int, nargs='?', help='modal id of the parent Qt widget')
    arguments.add_argument(dest='main_id', type=int, nargs='?', help='main frame id of XMS')
    parsed_args = arguments.parse_args()
    run_runner_main(
        module_name=parsed_args.module_name,
        class_name=parsed_args.class_name,
        method_name=parsed_args.method_name,
        main_file=parsed_args.file,
        modal_id=parsed_args.modal_id,
        main_id=parsed_args.main_id
    )
