"""Pooled process that imports known modules for efficiency."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import os
import shlex
import sys
import traceback

# 2. Third party modules
import win32com.client

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv

# 4. Local modules
from xms.components.runners.component_runner import run_runner_main
from xms.components.runners.project_io_event import project_io_main
from xms.components.runners.save_all_event import save_all_main
from xms.components.runners.save_event import save_event_main


def call_component_runner(split_line):
    """Call the component_runner main method.

    Args:
        split_line (list of str): The command line arguments split on whitespace
    """
    module_name = split_line[1]
    class_name = split_line[2]
    method_name = split_line[3]
    main_file = split_line[4].strip('"')
    if len(split_line) > 6:  # Parse the modal dialog arguments.
        modal_id = int(split_line[5])
        main_id = int(split_line[6])
    else:
        modal_id = None
        main_id = None

    # Switch current working directory to the component's folder.
    component_folder = os.path.dirname(main_file)
    if os.path.isdir(component_folder):
        os.chdir(component_folder)

    # Call the component runner script.
    run_runner_main(module_name, class_name, method_name, main_file, modal_id, main_id)


def call_save_event(split_line):
    """Call the save_all_event main method.

    Args:
        split_line (list of str): The command line arguments split on whitespace
    """
    module_name = split_line[1]
    class_name = split_line[2]
    old_main_file = split_line[3].strip('"')
    new_path = split_line[4].strip('"')
    save_type = split_line[5]
    save_event_main(module_name, class_name, old_main_file, new_path, save_type, None)


def call_save_all_event(split_line):
    """Call the save_event main method.

    Args:
        split_line (list of str): The command line arguments split on whitespace
    """
    event = split_line[1]
    save_all_main(event)


def call_project_io_event(split_line):
    """Call the project_io main method.

    Args:
        split_line (list of str): The command line arguments split on whitespace
    """
    db_path = split_line[1].strip('"')
    save_type = split_line[2]
    project_io_main(db_path, save_type)


def call_tool_runner(split_line):
    """Call the tool_runner main method.

    Args:
        split_line (list of str): The command line arguments split on whitespace
    """
    from xms.tool_gui.runners.tool_runner import main  # this should only get called xms.tool_gui is installed
    # xms.tool_gui is not in setup.py
    args = [arg.strip('"') for arg in split_line]
    args.pop()
    main(args)


class PoolProcess:
    """Pools a process until XMS needs it to run."""
    _command_arg_file = None
    _know_import_modules = False
    _counter = 0

    def __init__(self):
        """Constructor."""
        # Build the path of the file XMS is ready to pull us from the pool.
        PoolProcess._command_arg_file = os.path.join(os.getcwd(), f'{os.getpid()}_command_args')
        # Read the known DMI modules we might need to import. File may not exist yet.
        self.read_import_modules_file()

    @staticmethod
    def _log_exception(ex, description):
        """Print an exception traceback to the process debug log file.

        Args:
            ex (Exception): The exception to log
            description (str): Description of how/where the exception was raised
        """
        with open(f'debug_process_pool_{os.getpid()}.txt', 'a') as f_ex:
            f_ex.write(f'{description}\n')
            traceback.print_exception(type(ex), ex, ex.__traceback__, file=f_ex)
        XmEnv.report_error(ex)

    @staticmethod
    def read_import_modules_file():
        """Pre load all known DMI component modules."""
        if PoolProcess._know_import_modules:  # We already read the file
            return

        module_file = os.path.join(os.getcwd(), 'python_import_modules.txt')
        if os.path.isfile(module_file):
            with open(module_file, 'r') as f:  # File exists, check for end card.
                lines = f.readlines()
                if not lines or lines[-1] != 'END_MODULES':
                    return  # File not complete yet

                PoolProcess._know_import_modules = True
                for line in lines[:-1]:  # Each line is a loaded component definition: "module_name class_name"
                    split_line = line.split()
                    try:
                        # Dynamically import all possible modules we may import. Takes significant RAM but reduces lag
                        # in component script startup times.
                        module_name = split_line[0]
                        class_name = split_line[1]
                        _ = __import__(module_name, fromlist=[class_name])
                    except Exception as ex:
                        msg = f'Failed to load module: module_name = {module_name}  class_name = {class_name}'
                        PoolProcess._log_exception(ex, msg)
                        fname = 'process_pool_failed_import.txt'
                        if not os.path.exists(fname):
                            with open(fname, 'w') as f_fail:
                                f_fail.writeline(msg)
                                traceback.print_exception(type(ex), ex, ex.__traceback__, file=f_fail)

    @staticmethod
    def _run_command(line):
        """Parse commandline arguments and run the command.

        Args:
            line (str): The commandline arguments as written by XMS
        """
        split_line = shlex.split(line, posix=False)
        if split_line[0] == 'runner':
            call_component_runner(split_line)
        elif split_line[0] == 'save_all':
            call_save_all_event(split_line)
        elif split_line[0] == 'save':
            call_save_event(split_line)
        elif split_line[0] == 'project_io':
            call_project_io_event(split_line)
        elif split_line[0] == 'run_tool':
            call_tool_runner(split_line)

    @staticmethod
    def _ensure_xms_lives():
        """Kill this process if parent XMS PID no longer exists.

        We need to check if XMS still exists because we actually don't start querying XMS until it is ready for us to
        run. The heartbeat and timeout mechanisms in xmsapi shut down shortly after a crash of the parent XMS process,
        but it won't kill the Python process. In all other scripts, the process explodes the next time a Query call is
        made, which is desirable. We need to be more proactive. Normal exit of XMS aborts us before terminating XMS.
        """
        PoolProcess._counter += 1  # Only check every 100 iterations of the loop (~10 sec) to avoid hogging CPU time.
        if PoolProcess._counter < 100:
            return
        PoolProcess._counter = 0

        # Kill the process if our parent XMS has died.
        xms_alive = False
        xms_pid = int(os.environ.get('XMS_PYTHON_APP_PID', -1))
        wmi = win32com.client.GetObject('winmgmts:')
        for p in wmi.InstancesOf('win32_process'):
            if int(p.Properties_("ProcessId")) == xms_pid:
                xms_alive = True
                break  # Our parent XMS is still alive, continue polling.
        if not xms_alive:
            sys.exit(0)  # Our parent XMS is no longer running, commit suicide.

    @staticmethod
    def _update_environment():
        """Updates environment variables from a file written by XMS before executing command."""
        environ_file = f'{PoolProcess._command_arg_file}_env'
        if os.path.isfile(environ_file):
            with open(environ_file, 'r') as f:
                lines = f.readlines()
                for line in lines:
                    split_line = shlex.split(line, posix=False)  # Each line is a key-value pair
                    if len(split_line) < 2:
                        continue
                    key = split_line[0]
                    value = split_line[1].strip('"')  # Strip quotes off paths
                    os.environ[key] = value

            try:  # Cleanup environment file written by XMS.
                os.remove(environ_file)
            except Exception:
                pass  # Oh well, it's in the XMS temp directory

    @staticmethod
    def poll_loop():
        """Loops until XMS requests us to be removed from the pool and execute a component process.

        Breaks loop when the command line argument file for our PID exists and is completely written.
        """
        PoolProcess.read_import_modules_file()

        if os.path.isfile(PoolProcess._command_arg_file):  # File exists, check if contains a complete line.
            with open(PoolProcess._command_arg_file, 'r') as f:
                lines = f.readlines()
            if lines and lines[0].endswith('END_ARGS'):  # File is complete, run the requested script.
                try:  # Try to clean up the commandline argument file as soon as we read a complete one.
                    os.remove(PoolProcess._command_arg_file)
                except Exception:
                    pass  # Whatever, it is in the XMS temp directory.
                PoolProcess._update_environment()
                PoolProcess._run_command(lines[0])
                sys.exit(0)  # Kill the loop
        PoolProcess._ensure_xms_lives()

    @staticmethod
    def wait_to_run():
        """Begin polling for the command line argument file XMS writes when it is ready for us to run."""
        try:
            # This Query is just so we can start the progress loop. The real initial Context will be reset before
            # XMS pulls us out of the pool. The next Query instantiated will be ready to go.
            query = Query()

            prog = query.xms_agent.session.progress_loop
            prog.set_progress_function(PoolProcess.poll_loop)
            query._impl._instance.SetContext(prog.progress_context._instance)
            prog.start_constant_loop(100)
        except SystemExit:  # Normal exit
            sys.exit(0)
        except Exception as ex:
            PoolProcess._log_exception(ex, 'Unexpected error in pooled process.')
            sys.exit(1)
