"""ObsTargetComponent class. Data for observation targets Coverage."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest

# 4. Local modules
from xms.coverage.components.generic_coverage_component import duplicate_display_opts, GenericCoverageComponent
from xms.coverage.components.obs_target_component_display import ObsTargetComponentDisplay
from xms.coverage.data.obs_target_data import ObsTargetData


class ObsTargetComponent(GenericCoverageComponent):
    """A hidden Dynamic Model Interface (DMI) component for the Generic Coverages observation targets coverage."""
    def __init__(self, main_file, custom_display=None):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
            custom_display (Optional[list[str]]): List of custom display options JSON files to override defaults with.
                Should be size and order of feature types in ObsTargetData.OBS_CATEGORY_* enum. Elements should be
                None for feature types that you don't want to override. Elements should be the absolute path to the
                display options JSON file for the feature type with custom display.
        """
        super().__init__(main_file)
        self.data = ObsTargetData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Observations...', 'open_observations_all'),
            ('Display Options...', 'open_display_options'),
        ]
        self.point_commands = [('Observations...', 'open_observations')]
        self.arc_commands = [('Observations...', 'open_observations')]
        self.arc_group_commands = [('Observations...', 'open_observations')]
        self.polygon_commands = [('Observations...', 'open_observations')]
        self.disp_opts_files = ['', '', '', '']
        # Copy default display options if needed
        display_helper = ObsTargetComponentDisplay(self)
        display_helper.ensure_display_options_exist(custom_display)

    def _set_feature_menu_commands(self):
        """Clear menu commands for feature types not allowed in the coverage if feature type has been restricted."""
        feature_type = self.data.info.attrs['feature_type']
        if feature_type == ObsTargetData.OBS_CATEGORY_POINT:
            self.arc_commands = []
            self.arc_group_commands = []
            self.polygon_commands = []
        elif feature_type == ObsTargetData.OBS_CATEGORY_ARC:
            self.point_commands = []
            self.arc_group_commands = []
            self.polygon_commands = []
        elif feature_type == ObsTargetData.OBS_CATEGORY_ARC_GROUP:
            self.point_commands = []
            self.arc_commands = []
            self.polygon_commands = []
        elif feature_type == ObsTargetData.OBS_CATEGORY_POLY:
            self.point_commands = []
            self.arc_commands = []
            self.arc_group_commands = []

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of 'DUPLICATE', 'PACKAGE', 'SAVE', 'SAVE_AS', 'LOCK'.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            tuple(str, list, list):
                - new_main_file (str): Path to the new component's mainfile
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                  level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            filename = os.path.basename(self.disp_opts_files[ObsTargetData.OBS_CATEGORY_POINT])
            json_dict_point = duplicate_display_opts(new_path, filename)
            filename = os.path.basename(self.disp_opts_files[ObsTargetData.OBS_CATEGORY_ARC])
            json_dict_arc = duplicate_display_opts(new_path, filename)
            filename = os.path.basename(self.disp_opts_files[ObsTargetData.OBS_CATEGORY_ARC_GROUP])
            json_dict_arc_group = duplicate_display_opts(new_path, filename)
            filename = os.path.basename(self.disp_opts_files[ObsTargetData.OBS_CATEGORY_POLY])
            json_dict_poly = duplicate_display_opts(new_path, filename)
            data = ObsTargetData(new_main_file)
            data.info.attrs['cov_uuid'] = ''
            data.info.attrs[f'display_uuid_{ObsTargetData.OBS_CATEGORY_POINT}'] = json_dict_point['uuid']
            data.info.attrs[f'display_uuid_{ObsTargetData.OBS_CATEGORY_ARC}'] = json_dict_arc['uuid']
            data.info.attrs[f'display_uuid_{ObsTargetData.OBS_CATEGORY_ARC_GROUP}'] = json_dict_arc_group['uuid']
            data.info.attrs[f'display_uuid_{ObsTargetData.OBS_CATEGORY_POLY}'] = json_dict_poly['uuid']
            data.commit()

        return new_main_file, messages, action_requests

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            tuple(list, list)`: tuple containing:
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                  level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        action = ActionRequest(
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            method_name='get_initial_display_options'
        )
        return [], [action]

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.

        Returns:
            tuple(list, list): tuple containing:
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                  level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        display_helper = ObsTargetComponentDisplay(self)
        return display_helper.initialize_display(query)

    def get_display_menus(self, selection, lock_state, id_files):
        """This will be called when right-click menus in the main display area of XMS are being created.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            menu_items (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
            that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        self._set_feature_menu_commands()
        return super().get_display_menus(selection, lock_state, id_files)

    def get_double_click_actions_for_selection(self, selection, lock_state, id_files):
        """This will be called when a double-click in the main display area of XMS happened.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            tuple: tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        self._set_feature_menu_commands()
        return super().get_double_click_actions_for_selection(selection, lock_state, id_files)

    def open_display_options(self, query, params, win_cont):
        """Opens the Display Options dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                  level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        display_helper = ObsTargetComponentDisplay(self)
        return display_helper.open_display_options(query, win_cont)

    def open_observations(self, query, params, win_cont):
        """Opens the Observations dialog for a feature.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            tuple(list, list): tuple containing:
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                  level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        display_helper = ObsTargetComponentDisplay(self)
        return display_helper.open_observations(params, win_cont, query)

    def open_observations_all(self, query, params, win_cont):
        """Opens the Observations dialog for a feature.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            tuple(list, list): tuple containing:
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                  level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        display_helper = ObsTargetComponentDisplay(self)
        params = display_helper.get_feature_selection_params(query)
        return display_helper.open_observations(params, win_cont, query)

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (list(tuple)): tuple containing:

                - main_file (str): The absolute path to the main file of the old component this
                  component is being merged from.

                - id_files (dict): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                  Each value is a tuple that may have two absolute file paths or none. The first
                  file is for the ids in XMS on the coverage. The second file contains the ids the
                  old component used for those objects. Both id files should be equal in length.
                  This dictionary is only applicable if the component derives from
                  CoverageComponentBase.

        Returns:
            tuple(list, list): tuple containing:
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                  level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        # Don't think this works from the XMS side
        return [], []
