"""Python wrapping for xms._data_objects.parameters.SpatialVector."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.data_objects.parameters import Projection as PyProjection


class SpatialVector:
    """The pure Python wrapper for C++ exposed xms._data_objects.parameters.SpatialVector base class."""
    def __init__(self, instance, **kwargs):
        """Construct the wrapper.

        Note that this is an abstract class in C++. Need a C++ instance to construct.

        Args:
            instance: (xms._data_objects.parameters.SpatialVector) - C++ wrapped instance of a concrete implementation
                (Coverage, QuadTree, RectilinearGrid, or UGrid).
            **kwargs:
                name (str): Name of the geometric object
                uuid (str): UUID of the geometric object
                projection (PyProjection): Projection of the geometric object
                force_ugrid (bool): True to force SMS to build a UGrid
        """
        self._instance = instance

        name = kwargs.get('name')
        geom_uuid = kwargs.get('uuid')
        projection = kwargs.get('projection')
        if name is not None:
            self._instance.SetName(name)
        if geom_uuid is not None:
            self._instance.SetUUID(geom_uuid)
        if projection is not None:
            self.projection = projection  # Property setter will unwrap
        self.force_ugrid = kwargs.get('force_ugrid', False)

    @property
    def name(self):
        """Returns the spatial vector's name."""
        return self._instance.GetName()

    @name.setter
    def name(self, item_name):
        """Sets the spatial vector's name.

        Args:
            item_name (str): name of the spatial vector item
        """
        self._instance.SetName(item_name)

    @property
    def uuid(self):
        """Returns the spatial vector's UUID."""
        return self._instance.GetUUID()

    @uuid.setter
    def uuid(self, item_uuid):
        """Sets the spatial vector's UUID.

        Args:
            item_uuid (str): UUID of the spatial vector item
        """
        self._instance.SetUUID(item_uuid)

    @property
    def projection(self):
        """Returns the spatial vector's projection as a xms.data_objects.parameters.Projection."""
        return PyProjection(instance=self._instance.GetProjection())

    @projection.setter
    def projection(self, proj):
        """Sets the spatial vector's projection.

        Args:
            proj (xms.data_objects.parameters.Projection): Projection of the spatial vector item
        """
        self._instance.SetProjection(proj._instance)

    @property
    def native_projection(self):
        """Returns the spatial vector's native projection as a xms.data_objects.parameters.Projection.

        Note:
            Only applicable when geometry comes from XMS. In that case, the projection attribute is the current
            display projection, which is what the geometry locations are in when exported. When sending a geometric
            item to XMS to be built, the projection property is assumed to be the geometry's native projection.
        """
        return PyProjection(instance=self._instance.GetNativeProjection())

    @property
    def force_ugrid(self):
        """Returns the spatial vector's CoGrid builder flag.

        Returns:
            see description
        """
        return self._instance.GetForceUGrid()

    @force_ugrid.setter
    def force_ugrid(self, force):
        """Sets the spatial vector's CoGrid builder flag.

        Args:
            force (bool): The CoGrid builder flag
        """
        self._instance.SetForceUGrid(force)

    def complete(self, flag=True):
        """Sets the spatial vector's complete flag.

        Args:
            flag (bool): If the flag is True, the object's data will be written to an H5 file before being sent to the
                XMS process. Sending a geometry or coverage to XMS with this flag set to False is strongly discouraged
                for performance reasons.
        """
        self._instance.Complete(flag)
