"""Dialog for assigning Material and Sediment Material coverage properties."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtWidgets import (
    QAbstractItemView, QDialogButtonBox, QGroupBox, QHBoxLayout, QLabel, QLineEdit, QSizePolicy, QSpacerItem,
    QVBoxLayout
)

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets.category_display_options_list import CategoryDisplayOptionsWidget, Columns

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as consts


class EwnPropertiesDialog(XmsDlg):
    """A dialog to define EWN feature properties and their display options."""
    def __init__(self, parent, display_options, data):
        """Constructor.

        Args:
            parent (:obj:`QWidget`): Parent Qt dialog
            display_options (:obj:`CategoryDisplayOptionList`): The current EWN display options
            data (:obj:`EwnCovData`): The coverage component data
        """
        super().__init__(parent, 'xms.ewn.gui.ewn_properties_dialog')
        self._help_url = 'https://www.xmswiki.com/wiki/SMS:EWN'
        self._data = data
        self._widgets = {}
        self._setup_ui(display_options)

    def _setup_ui(self, display_options):
        """Setup dialog widgets.

        Args:
            display_options (:obj:`CategoryDisplayOptionList`): The current EWN display options
        """
        self.setWindowTitle('EWN Classification Properties')
        self._setup_display_options(display_options)
        self._setup_attributes()
        self._setup_main_layout()

    def _setup_display_options(self, display_options):
        """Adds the display options table view.

        Args:
            display_options (:obj:`CategoryDisplayOptionList`): The current EWN display options
        """
        # Setup the display options widget
        self._widgets['display_options'] = CategoryDisplayOptionsWidget(self)
        display_widget = self._widgets['display_options']
        display_widget.set_show_targets(False)
        display_widget.set_category_lists([display_options])
        display_widget.ui.table_view_categories.setColumnHidden(Columns.FONT_CHECK, True)
        display_widget.ui.table_view_categories.setColumnHidden(Columns.FONT, True)
        display_widget.ui.btn_change_all_label.setVisible(False)
        self._widgets['grp_display_options'] = QGroupBox('Display options')
        self._widgets['layout_display_options'] = QVBoxLayout()
        self._widgets['layout_display_options'].addWidget(display_widget)
        self._widgets['grp_display_options'].setLayout(self._widgets['layout_display_options'])

        # Connect a slot for updating the attribute pane widgets when the selection changes.
        display_widget.ui.table_view_categories.setSelectionBehavior(QAbstractItemView.SelectRows)
        display_widget.ui.table_view_categories.setSelectionMode(QAbstractItemView.SingleSelection)
        selection_model = display_widget.ui.table_view_categories.selectionModel()
        selection_model.currentRowChanged.connect(self.on_feature_type_changed)

    def _setup_attributes(self):
        """Adds the attribute pane widgets."""
        # Setup the feature attributes pane
        float_validator = QxDoubleValidator(parent=self)
        float_validator.setBottom(0.0)
        float_validator.setDecimals(10)
        self._widgets['edt_roughness'] = QLineEdit('0.0')
        self._widgets['edt_roughness'].setValidator(float_validator)
        self._widgets['edt_roughness'].setEnabled(False)
        self._widgets['lbl_roughness'] = QLabel('Default roughness:')
        self._widgets['layout_horiz_roughness'] = QHBoxLayout()
        self._widgets['layout_horiz_roughness'].addWidget(self._widgets['lbl_roughness'])
        self._widgets['layout_horiz_roughness'].addWidget(self._widgets['edt_roughness'])
        self._widgets['spacer_vert_roughness'] = QSpacerItem(40, 20, QSizePolicy.Minimum, QSizePolicy.Expanding)
        self._widgets['layout_vert_attributes'] = QVBoxLayout()
        self._widgets['layout_vert_attributes'].addLayout(self._widgets['layout_horiz_roughness'])
        self._widgets['layout_vert_attributes'].addItem(self._widgets['spacer_vert_roughness'])
        self._widgets['grp_attributes'] = QGroupBox('Feature attributes')
        self._widgets['grp_attributes'].setLayout(self._widgets['layout_vert_attributes'])

    def _setup_main_layout(self):
        """Adds the dialog button bar and sets the top-level dialog layout."""
        # Add the dialog buttons layout
        self._widgets['btn_box'] = QDialogButtonBox()
        btn_flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(btn_flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['btn_horiz_layout'] = QHBoxLayout()
        self._widgets['btn_horiz_layout'].addWidget(self._widgets['btn_box'])

        # Add a main layout.
        self._widgets['main_horiz_layout'] = QHBoxLayout()
        self._widgets['main_horiz_layout'].addWidget(self._widgets['grp_display_options'])
        self._widgets['main_horiz_layout'].addWidget(self._widgets['grp_attributes'])
        self._widgets['main_vert_layout'] = QVBoxLayout()
        self._widgets['main_vert_layout'].addLayout(self._widgets['main_horiz_layout'])
        self._widgets['main_vert_layout'].addLayout(self._widgets['btn_horiz_layout'])
        self.setLayout(self._widgets['main_vert_layout'])

    def on_feature_type_changed(self, selected, _):
        """Updates the feature attributes pane when the selected feature type changes.

        Args:
            selected (:obj:`QModelIndex`): The new selected item
            '_' (:obj:`QModelIndex`): The previously selected item
        """
        # Update the new values
        selected_row = selected.row()
        df = self._data.meta_data
        rough_list = [0.0] + df['Default n'].tolist()
        if selected.isValid() and selected_row >= consts.FEATURE_TYPE_UNASSIGNED:
            self._widgets['edt_roughness'].setVisible(True)
            self._widgets['lbl_roughness'].setVisible(True)
            self._widgets['edt_roughness'].setText(str(rough_list[selected_row]))
        else:
            self._widgets['edt_roughness'].setVisible(False)
            self._widgets['lbl_roughness'].setVisible(False)

    def help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def accept(self):
        """Ensure currently selected feature type's attributes get saved."""
        super().accept()

    def get_category_lists(self):
        """Returns the current category display options lists.

        Returns:
            (:obj:`list[CategoryDisplayOptionList]`): The current category display options list.
        """
        return self._widgets['display_options'].get_category_lists()
