"""ToolInputDialog class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (
    QComboBox, QDialogButtonBox, QGroupBox, QHBoxLayout, QLabel, QLineEdit, QListWidget, QListWidgetItem, QMessageBox,
    QPushButton, QRadioButton, QSizePolicy, QToolBar, QVBoxLayout
)

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.treeitem_selector_datasets import TreeItemSelectorDatasetsDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets import widget_builder

# 4. Local modules


def is_ewn_features_if_coverage(item):
    """Check if a tree item is an EWN Features coverage, but only if it is a coverage.

    Args:
        item (:obj:`TreeNode`): The item to check

    Returns:
        (:obj:`bool`): True if the tree item is an EWN Features coverage type or is not a coverage.
    """
    if item.item_typename == 'TI_COVER':
        if item.coverage_type == 'EWN Features':
            return True
        return False
    return True


def is_scalar_nodes_if_dset(item):
    """Check if a tree item is scalar and node-based but only if it is a dataset.

    Args:
        item (:obj:`TreeNode`): The item to check

    Returns:
        (:obj:`bool`): True if the tree item is scalar or is not a dataset.
    """
    if item.item_typename in ['TI_SFUNC', 'TI_VFUNC', 'TI_SCALAR_DSET', 'TI_VECTOR_DSET']:
        if item.num_components == 1 and item.data_location == 'NODE':
            return True
        return False
    return True


class ToolInputDialog(XmsDlg):
    """A dialog used for selecting inputs of EWN Feature coverage commands."""
    INSERT_FEATURES = 'insert_features'
    ROUGHNESS_DATASET = 'roughness_dataset'

    def __init__(self, pe_tree, source_coverage, command, parent=None, query=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            pe_tree (:obj:`xms.guipy.tree.tree_node.TreeNode`): The XMS project explorer
            source_coverage (:obj:`str`): UUID of the EWN coverage this command is coming from
            command (:obj:`str`): The EWN coverage command to get user input for. One of:
                INSERT_FEATURES, ROUGHNESS_DATASET
            parent (Something derived from :obj:`QWidget`): The parent window.
            query (:obj:`Query`): XMS interprocess communication. Only needed if gathering inputs for the roughness
                dataset generation command.
        """
        super().__init__(parent, 'xms.ewn.gui.insert_ewn_features')
        self._help_url = 'https://www.xmswiki.com/wiki/SMS:EWN'
        self._source_coverage = source_coverage
        self._command = command
        self._query = query
        self._pe_tree = pe_tree
        self._ewn_pe_tree = None
        self._widgets = {}

        self._target_geometry = None
        self._lock_dataset_uuid = None
        self._initial_roughness_uuid = ''  # UUID of initial roughness input dataset
        self._initial_roughness_timestep = 0  # Timestep index of initial roughness input dataset
        self._is_cartesian = None

        self._filter_tree_to_ewn_coverages()
        self._setup_widgets()

    def _get_input_coverage_uuids(self):
        """Returns a list of the currently selected input EWN coverage UUIDs."""
        cov_list = self._widgets['lst_ewn_covs']
        return [
            cov_list.item(row).data(Qt.UserRole) for row in range(cov_list.count())
            if cov_list.item(row).data(Qt.UserRole)  # '(none selected)' item's data is empty string
        ]

    def _filter_tree_to_ewn_coverages(self):
        """Creates a copy of the project explorer tree and filters out non-EWN coverages."""
        creator = tree_util.ProjectExplorerTreeCreator()
        self._ewn_pe_tree = creator.copy(self._pe_tree)
        tree_util.filter_project_explorer(self._ewn_pe_tree, is_ewn_features_if_coverage)

    def _get_dataset_selector_tree(self):
        """Get a dataset selector project explorer tree."""
        geom_item = tree_util.find_tree_node_by_uuid(self._pe_tree, self._target_geometry)
        creator = tree_util.ProjectExplorerTreeCreator()
        dataset_tree = creator.copy(geom_item)
        if dataset_tree:
            pass
            # tree_util.filter_project_explorer(dataset_tree, is_scalar_nodes_if_dset)
        return dataset_tree

    def _add_input_coverage(self, cov_uuid):
        """Adds an input EWN Features coverage to the list.

        Args:
            cov_uuid (str): UUID of the input coverage to add
        """
        tree_path = tree_util.build_tree_path(self._ewn_pe_tree, cov_uuid) if cov_uuid else '(none selected)'
        new_item = QListWidgetItem(tree_path)
        new_item.setData(Qt.UserRole, cov_uuid)
        self._widgets['lst_ewn_covs'].addItem(new_item)

    def _setup_widgets(self):
        """Add widgets to the dialog."""
        self.setWindowTitle(
            'Insert EWN Features' if self._command == self.INSERT_FEATURES else 'Generate Roughness Data Set'
        )
        self._setup_target_geometry_widgets()
        self._setup_initial_roughness_widgets()
        self._setup_ewn_feature_coverages_widgets()
        self._setup_output_widgets()
        self._setup_dialog_layout()

    def _setup_target_geometry_widgets(self):
        """Add widgets for selecting the target geometry."""
        # Set up the geometry selector group
        self._widgets['btn_target_geom'] = QPushButton('Select...')
        self._widgets['btn_target_geom'].clicked.connect(self.select_target_geometry)
        self._widgets['lbl_selected_geom'] = QLabel('(none selected)')
        self._widgets['lbl_selected_geom_size_policy'] = QSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        self._widgets['lbl_selected_geom'].setSizePolicy(self._widgets['lbl_selected_geom_size_policy'])
        self._widgets['layout_horiz_geom'] = QHBoxLayout()
        self._widgets['layout_horiz_geom'].addWidget(self._widgets['btn_target_geom'])
        self._widgets['layout_horiz_geom'].addWidget(self._widgets['lbl_selected_geom'])

        self._widgets['btn_target_geom_lock'] = QPushButton('Lock Dataset...')
        self._widgets['btn_target_geom_lock'].clicked.connect(self.select_target_geometry_lock)
        self._widgets['btn_target_geom_lock'].setEnabled(False)
        self._widgets['lbl_selected_geom_lock'] = QLabel('(none selected)')
        self._widgets['lbl_selected_geom_lock_size_policy'] = QSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        self._widgets['lbl_selected_geom_lock'].setSizePolicy(self._widgets['lbl_selected_geom_lock_size_policy'])
        self._widgets['lbl_selected_geom_lock'].setEnabled(False)
        self._widgets['layout_horiz_geom_lock'] = QHBoxLayout()
        self._widgets['layout_horiz_geom_lock'].addWidget(self._widgets['btn_target_geom_lock'])
        self._widgets['layout_horiz_geom_lock'].addWidget(self._widgets['lbl_selected_geom_lock'])

        if self._command == self.ROUGHNESS_DATASET:
            # Don't show lock stuff if doing roughness dataset
            self._widgets['btn_target_geom_lock'].setVisible(False)
            self._widgets['lbl_selected_geom_lock'].setVisible(False)

        self._widgets['grp_target_geom'] = QGroupBox('Target geometry')
        self._widgets['layout_vert_geom'] = QVBoxLayout()
        self._widgets['layout_vert_geom'].addLayout(self._widgets['layout_horiz_geom'])
        self._widgets['layout_vert_geom'].addLayout(self._widgets['layout_horiz_geom_lock'])
        if self._command == self.INSERT_FEATURES:
            float_validator = QxDoubleValidator(parent=self)
            float_validator.setBottom(0.0)
            float_validator.setTop(1.0)
            self._widgets['lbl_transition'] = QLabel('Element area change limit (fast 0.0 <---> 1.0 slow):')
            self._widgets['edt_transition'] = QLineEdit('0.5')
            self._widgets['edt_transition'].setValidator(float_validator)
            size_policy = self._widgets['edt_transition'].sizePolicy()
            size_policy.setHorizontalPolicy(QSizePolicy.MinimumExpanding)
            self._widgets['edt_transition'].setSizePolicy(size_policy)
            self._widgets['layout_horiz_transition'] = QHBoxLayout()
            self._widgets['layout_horiz_transition'].addWidget(self._widgets['lbl_transition'])
            self._widgets['layout_horiz_transition'].addWidget(self._widgets['edt_transition'])
            self._widgets['layout_vert_geom'].addLayout(self._widgets['layout_horiz_transition'])
        self._widgets['grp_target_geom'].setLayout(self._widgets['layout_vert_geom'])

    def _setup_ewn_feature_coverages_widgets(self):
        """Add widgets for selecting the input EWN Feature coverages."""
        # Set up the EWN coverage list.
        self._widgets['btn_ewn_covs'] = QPushButton('Select...')
        self._widgets['btn_ewn_covs'].clicked.connect(self.select_ewn_coverages)

        # Add a list of the selected EWN Feature coverages. For now, initialize with the coverage that the command
        # originated from. When this is moved to a tool, this will probably not be valid.
        self._widgets['lst_ewn_covs'] = QListWidget()
        self._add_input_coverage(self._source_coverage)

        # Add Up/Down toolbar so the user can sort the input coverages by priority.
        self._widgets['coverage_toolbar'] = QToolBar()
        buttons = [
            [':resources/icons/row-up.svg', 'Move up', self.on_btn_up],
            [':resources/icons/row-down.svg', 'Move down', self.on_btn_down],
        ]
        widget_builder.setup_toolbar(self._widgets['coverage_toolbar'], buttons)
        self._widgets['layout_horiz_toolbar'] = QHBoxLayout()
        self._widgets['layout_horiz_toolbar'].addWidget(self._widgets['btn_ewn_covs'])
        self._widgets['layout_horiz_toolbar'].addWidget(self._widgets['coverage_toolbar'])
        # Add a note about what the order of the input EWN Feature coverages means.
        self._widgets['lbl_coverage_priority'] = QLabel(
            'Note: If multiple input EWN Feature coverages are specified, priority is assigned to the coverages in '
            'descending order.'
        )
        self._widgets['lbl_coverage_priority'].setWordWrap(True)

        self._widgets['layout_vert_ewn_covs'] = QVBoxLayout()
        self._widgets['layout_vert_ewn_covs'].addLayout(self._widgets['layout_horiz_toolbar'])
        self._widgets['layout_vert_ewn_covs'].addWidget(self._widgets['lst_ewn_covs'])
        self._widgets['layout_vert_ewn_covs'].addWidget(self._widgets['lbl_coverage_priority'])
        self._widgets['grp_ewn_covs'] = QGroupBox('EWN Feature coverages')
        self._widgets['grp_ewn_covs'].setLayout(self._widgets['layout_vert_ewn_covs'])

    def _setup_initial_roughness_widgets(self):
        """Add widgets for selecting initial values when generating a roughness data set."""
        self._widgets['grp_initial_roughness'] = QGroupBox('Initial values')
        self._widgets['initial_roughness_option_layout'] = QVBoxLayout()
        self._widgets['initial_roughness_input_layout'] = QVBoxLayout()
        grp_box = self._widgets['grp_initial_roughness']
        option_layout = self._widgets['initial_roughness_option_layout']
        input_layout = self._widgets['initial_roughness_input_layout']

        # Set up the initial roughness option radio group
        self._widgets['rbt_constant_roughness'] = QRadioButton('Constant (node)', grp_box)
        self._widgets['rbt_constant_roughness'].setEnabled(True)
        self._widgets['rbt_constant_roughness'].setChecked(True)
        self._widgets['rbt_constant_roughness'].toggled.connect(self.on_initial_roughness_option_toggled)
        self._widgets['rbt_dataset_roughness'] = QRadioButton('Data Set', grp_box)
        self._widgets['rbt_dataset_roughness'].setEnabled(False)  # Not enabled until target geometry is selected
        self._widgets['rbt_dataset_roughness'].setChecked(False)
        self._widgets['rbt_dataset_roughness'].toggled.connect(self.on_initial_roughness_option_toggled)
        option_layout.addWidget(self._widgets['rbt_constant_roughness'])
        option_layout.addWidget(self._widgets['rbt_dataset_roughness'])

        # Set up the constant initial roughness line edit widget
        float_validator = QxDoubleValidator(parent=self)
        float_validator.setBottom(0.0)
        float_validator.setDecimals(10)
        self._widgets['cbx_dataset_location'] = QComboBox()
        self._widgets['cbx_dataset_location'].addItem('at points', 'points')
        self._widgets['cbx_dataset_location'].addItem('at cells', 'cells')
        self._widgets['edt_constant_roughness'] = QLineEdit('0.0')
        self._widgets['edt_constant_roughness'].setValidator(float_validator)
        self._widgets['layout_horiz_constant_roughness'] = QHBoxLayout()
        self._widgets['layout_horiz_constant_roughness'].addWidget(self._widgets['cbx_dataset_location'])
        self._widgets['layout_horiz_constant_roughness'].addWidget(self._widgets['edt_constant_roughness'])

        # Set up the initial roughness dataset selector.
        self._widgets['btn_initial_roughness'] = QPushButton('Select...')
        self._widgets['btn_initial_roughness'].setEnabled(False)
        self._widgets['btn_initial_roughness'].clicked.connect(self.select_initial_roughness)
        self._widgets['lbl_initial_roughness'] = QLabel('(none selected)')
        self._widgets['layout_horiz_initial_roughness'] = QHBoxLayout()
        self._widgets['layout_horiz_initial_roughness'].addWidget(self._widgets['btn_initial_roughness'])
        self._widgets['layout_horiz_initial_roughness'].addWidget(self._widgets['lbl_initial_roughness'])

        input_layout.addLayout(self._widgets['layout_horiz_constant_roughness'])
        input_layout.addLayout(self._widgets['layout_horiz_initial_roughness'])

        # Setup the groupbox
        self._widgets['initial_roughness_horiz_layout'] = QHBoxLayout()
        self._widgets['initial_roughness_horiz_layout'].addLayout(option_layout)
        self._widgets['initial_roughness_horiz_layout'].addLayout(input_layout)
        self._widgets['grp_initial_roughness'].setLayout(self._widgets['initial_roughness_horiz_layout'])

    def _setup_output_widgets(self):
        """Add widgets for specifying output options."""
        # Roughness dataset name
        self._widgets['output_dataset_horiz_layout'] = QHBoxLayout()
        self._widgets['lbl_roughness_name'] = QLabel('Data set name:')
        self._widgets['edt_roughness_name'] = QLineEdit('EWN Roughness')
        self._widgets['output_dataset_horiz_layout'].addWidget(self._widgets['lbl_roughness_name'])
        self._widgets['output_dataset_horiz_layout'].addWidget(self._widgets['edt_roughness_name'])

        self._widgets['output_dataset_vert_layout'] = QVBoxLayout()
        self._widgets['output_dataset_vert_layout'].addLayout(self._widgets['output_dataset_horiz_layout'])
        self._widgets['grp_output_options'] = QGroupBox('Output')
        self._widgets['grp_output_options'].setLayout(self._widgets['output_dataset_vert_layout'])

    def _setup_dialog_layout(self):
        """Adds the OK/Cancel buttons and sets the dialog's main layout."""
        # Set up the OK/Cancel button box
        self._widgets['btn_box'] = QDialogButtonBox()
        flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)

        # Set the top-level dialog layout
        self._widgets['layout_vert_main'] = QVBoxLayout()
        self._widgets['layout_vert_main'].addWidget(self._widgets['grp_target_geom'])
        self._widgets['layout_vert_main'].addWidget(self._widgets['grp_initial_roughness'])
        self._widgets['layout_vert_main'].addWidget(self._widgets['grp_ewn_covs'])
        self._widgets['layout_vert_main'].addWidget(self._widgets['grp_output_options'])
        self._widgets['layout_vert_main'].addWidget(self._widgets['btn_box'])
        self.setLayout(self._widgets['layout_vert_main'])

        # Hide the initial roughness groupbox if not generating a roughness dataset
        generating_roughness = self._command == self.ROUGHNESS_DATASET
        self._widgets['grp_initial_roughness'].setVisible(generating_roughness)
        self._widgets['grp_output_options'].setVisible(generating_roughness)

    def _clear_dataset_selector(self):
        """Clears the initial roughness dataset selection."""
        self._initial_roughness_uuid = ''
        self._initial_roughness_timestep = 0
        self._widgets['lbl_initial_roughness'].setText('(none selected)')

    def _move_row(self, up):
        """Moves a row up or down.

        Args:
            up (:obj:`bool`): True if moving up, else False
        """
        current_row = self._widgets['lst_ewn_covs'].currentRow()
        if current_row < 0:
            return
        current_item = self._widgets['lst_ewn_covs'].takeItem(current_row)
        new_row = current_row - 1 if up else current_row + 1
        self._widgets['lst_ewn_covs'].insertItem(new_row, current_item)
        self._widgets['lst_ewn_covs'].setCurrentRow(new_row)

    def on_btn_up(self):
        """Called when the Up button is clicked. Moves rows up in the table."""
        self._move_row(True)

    def on_btn_down(self):
        """Called when the Down button is clicked. Moves rows down in the table."""
        self._move_row(False)

    def on_initial_roughness_option_toggled(self):
        """Slot called when the initial roughness radio button option is changed."""
        constant_selected = self._widgets['rbt_constant_roughness'].isChecked()
        self._widgets['edt_constant_roughness'].setEnabled(constant_selected)
        self._widgets['cbx_dataset_location'].setEnabled(constant_selected)
        self._widgets['btn_initial_roughness'].setEnabled(not constant_selected)

    def select_target_geometry(self):
        """Slot to display the target geometry tree item selector."""
        # Display a tree item selector dialog.
        selectable_types = ['TI_MESH2D', 'TI_UGRID_SMS', 'TI_CGRID2D']

        selector_dlg = TreeItemSelectorDlg(
            title='Select EWN Geometry',
            target_type='',
            pe_tree=self._pe_tree,
            previous_selection=self._target_geometry,
            parent=self,
            selectable_xms_types=selectable_types
        )

        old_target = self._target_geometry
        if selector_dlg.exec():
            self._target_geometry = selector_dlg.get_selected_item_uuid()
            if self._target_geometry:
                self._widgets['lbl_selected_geom'].setText(
                    tree_util.build_tree_path(self._pe_tree, self._target_geometry)
                )
                self._widgets['rbt_dataset_roughness'].setEnabled(True)
                if self._target_geometry != old_target:  # Clear initial dataset selection if target geometry changed
                    self._clear_dataset_selector()
                tree_item = tree_util.find_tree_node_by_uuid(self._pe_tree, self._target_geometry)
                self._is_cartesian = tree_item.item_typename == 'TI_CGRID2D'
                self._enable_lock_selection()
            else:  # No target geometry selected, clear and disable the initial dataset selection
                self._widgets['lbl_selected_geom'].setText('(none selected)')
                self._widgets['rbt_constant_roughness'].setChecked(True)
                self._widgets['rbt_dataset_roughness'].setEnabled(False)
                self._clear_dataset_selector()
                self._disable_lock_selection()

    def select_target_geometry_lock(self):
        """Select a ugrid or mesh dataset."""
        if self._is_cartesian:
            msgbox = QMessageBox(
                QMessageBox.Warning, 'SMS', 'Locking is not supported for Cartesian grids.', QMessageBox.Ok, self
            )
            msgbox.exec()
            return

        selector_dlg = TreeItemSelectorDlg(
            title='Select Lock Dataset',
            target_type='',
            pe_tree=self._get_dataset_selector_tree(),
            previous_selection=self._lock_dataset_uuid,
            parent=self,
            show_root=True,
            selectable_xms_types=['TI_SFUNC', 'TI_SCALAR_DSET']
        )

        if selector_dlg.exec():
            lock_dataset_uuid = selector_dlg.get_selected_item_uuid()
            if not lock_dataset_uuid:  # No lock dataset selected
                self._clear_lock_selection()
                return
            tree_path = tree_util.build_tree_path(self._pe_tree, lock_dataset_uuid)
            self._widgets['lbl_selected_geom_lock'].setText(tree_path)
            self._lock_dataset_uuid = lock_dataset_uuid

    def _enable_lock_selection(self):
        self._widgets['btn_target_geom_lock'].setEnabled(True)
        self._widgets['lbl_selected_geom_lock'].setEnabled(True)

    def _disable_lock_selection(self):
        self._clear_lock_selection()
        self._widgets['btn_target_geom_lock'].setEnabled(False)
        self._widgets['lbl_selected_geom_lock'].setEnabled(False)

    def _clear_lock_selection(self):
        self._lock_dataset_uuid = None
        self._widgets['lbl_selected_geom_lock'].setText('(none selected)')

    def select_ewn_coverages(self):
        """Slot to display the EWN Features coverages tree item selector."""
        # Display a tree item selector dialog.
        selector_dlg = TreeItemSelectorDlg(
            title='Select EWN Feature Coverages',
            target_type='',
            pe_tree=self._ewn_pe_tree,
            previous_selection=self._get_input_coverage_uuids(),
            parent=self,
            allow_multi_select=True,
            selectable_xms_types=['TI_COVER']
        )

        if selector_dlg.exec():
            selected_uuids = selector_dlg.get_selected_item_uuid()
            self._widgets['lst_ewn_covs'].clear()
            if not selected_uuids:
                self._add_input_coverage('')
                return
            for selected_uuid in selected_uuids:
                self._add_input_coverage(selected_uuid)

    def select_initial_roughness(self):
        """Slot to display the initial roughness dataset tree item selector.

        A target geometry must be specified before this slot can get called.
        """
        # Display a tree item selector dialog. Trim the tree to the selected geometry. This button is disabled if no
        # target geometry has been selected. There should always be at least the scalar Z dataset available for
        # selection under any given geometry.
        selector_dlg = TreeItemSelectorDatasetsDlg(
            title='Select Initial Roughness Data Set',
            pe_tree=self._get_dataset_selector_tree(),
            selected_dataset=self._initial_roughness_uuid,
            selected_time_step=self._initial_roughness_timestep,
            query=self._query,
            show_root=True,
            parent=self,
        )

        if selector_dlg.exec():
            initial_dataset = selector_dlg.get_selected_item_uuid()
            if initial_dataset:
                tree_path = tree_util.build_tree_path(self._pe_tree, initial_dataset)
                self._widgets['lbl_initial_roughness'].setText(
                    f'{tree_path}{selector_dlg.get_selected_time_step_string()}'
                )
                self._initial_roughness_uuid = initial_dataset
                self._initial_roughness_timestep = max(selector_dlg.get_selected_time_step_index(), 0)
            else:
                self._widgets['lbl_initial_roughness'].setText('(none selected)')
                self._initial_roughness_uuid = ''
                self._initial_roughness_timestep = 0

    def accept(self):
        """Don't allow OK exit if required inputs have not been selected."""
        # Check that a target geometry has been selected.
        if not self._target_geometry:
            msgbox = QMessageBox(
                QMessageBox.Warning, 'SMS', 'Select a target Mesh 2D or UGrid geometry.', QMessageBox.Ok, self
            )
            msgbox.exec()
            return

        # Check that at least one EWN Features coverage has been selected.
        if len(self._get_input_coverage_uuids()) < 1:
            msgbox = QMessageBox(
                QMessageBox.Warning, 'SMS', 'At least one EWN Features coverage must be selected.', QMessageBox.Ok, self
            )
            msgbox.exec()
            return

        # Ensure initial roughness dataset selected if applicable.
        applicable = self._command == self.ROUGHNESS_DATASET and self._widgets['rbt_dataset_roughness'].isChecked()
        if applicable and not self._initial_roughness_uuid:
            msg = 'Select an initial roughness data set or switch to the constant initial roughness option.'
            msgbox = QMessageBox(QMessageBox.Warning, 'SMS', msg, QMessageBox.Ok, self)
            msgbox.exec()
            return

        super().accept()

    def help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def tool_inputs(self):
        """Retrieve user inputs from the dialog.

        Returns:
            (:obj:`dict`): Inputs for the tool command
        """
        bias = None
        if 'edt_transition' in self._widgets:
            bias = float(self._widgets['edt_transition'].text())
        inputs = {
            'target_geometry': self._target_geometry,
            'lock_dataset_uuid': self._lock_dataset_uuid,
            'ewn_coverages': self._get_input_coverage_uuids(),
            'dataset_location': self._widgets['cbx_dataset_location'].currentData(),
            'bias': bias
        }
        # Swap the order of the input EWN Feature coverages. The coverage with the highest priority should be
        # processed last. In the GUI, coverages with higher priority are higher in the list.
        inputs['ewn_coverages'].reverse()

        if self._command == self.ROUGHNESS_DATASET:
            dset_name = self._widgets['edt_roughness_name'].text()
            inputs['output_dset_name'] = dset_name if dset_name else 'EWN Roughness'
            if self._widgets['rbt_constant_roughness'].isChecked():  # Constant initial roughness value
                inputs['constant_roughness'] = float(self._widgets['edt_constant_roughness'].text())
            else:
                inputs['initial_roughness_uuid'] = self._initial_roughness_uuid
                inputs['initial_roughness_timestep'] = self._initial_roughness_timestep
        return inputs
