"""EwnProcess class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import math

# 2. Third party modules
import numpy as np
from shapely.geometry import CAP_STYLE, JOIN_STYLE, LineString, Point, Polygon

# 3. Aquaveo modules
from xms.constraint.ugrid_builder import UGridBuilder
from xms.coverage.grid.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.extractor.ugrid_2d_polyline_data_extractor import UGrid2dPolylineDataExtractor
from xms.grid.geometry import geometry as xmg
from xms.interp.interpolate.interp_linear import InterpLinear
import xms.mesher
from xms.mesher import meshing
from xms.mesher.meshing import mesh_utils
from xms.tool.algorithms.geometry.geometry import run_parallel_points_in_polygon

# 4. Local modules
import xms.ewn.data.ewn_cov_data_consts as consts


class EwnProcess:
    """Class to do calculations related to inserting an EWN feature into a 2d ugrid."""
    def __init__(self, ugrid, is_geographic, is_levee, bias, lock_dataset=None, is_cartesian=False):
        """Initializes the class.

        Args:
            ugrid (:obj:`xms.grid.ugrid.UGrid`): 2d Unstructured grid
            is_geographic (:obj:`bool`): true if coordinates are geographic
            is_levee (:obj:`bool`): true if the coverage is a levee coverage
            bias (:obj:`float`): bias used in xms.mesher for speed of element size transition (0.0-1.0)
            lock_dataset (:obj:`list`): A lock dataset used for locking nodes.
            is_cartesian (:obj:`bool`): true if the grid is cartesian
        """
        self.show_transition_warning = True
        self._polygon_data = None
        self._ugrid = ugrid
        self.is_cartesian = is_cartesian
        self._is_geographic = is_geographic
        self._is_levee = is_levee
        self._lock_dataset = lock_dataset if lock_dataset else [0] * self._ugrid.point_count
        self._logger = logging.getLogger('xms.ewn')
        bias = max(1e-5, bias)
        bias = min(1.0 - 1e-5, bias)
        self._mesher_bias = bias
        self._debug = False

        # polygon where ewn transects intersect ugrid
        self._is_convex = None
        self.shapely_poly = None
        self.shapely_poly_buffered = None
        self.average_spacing = None
        self.transect_polygon = None
        self.non_intersecting_transects = None
        self.transition_polygon = None
        self.ewn_ugrid = None
        self.stitched_ugrid = None
        self._transition_polygon_ugrid_idxs = None
        self._transition_poly_holes_ugrid_idxs = None
        self._transition_polygon_ugrid_remove_pts = None
        self._transition_polygon_ugrid_remove_cells = None
        self._stitch_new_to_old_pt_idx = None
        self._stitch_old_to_new_pt_idx = None
        self._input_poly_segment_distances = None
        self._transect_distances = None
        self._stitched_ugrid_smooth_size = None
        self._lock_dataset_cell_idxes = set()
        self._locked_cells = None
        self._previous_transect_polys = []
        self._ugrid_locs = None
        self._ewn_locs = None
        self._transect_cell_idxes = None
        self.cgrid_elevations = None

        # polygon properties
        self._poly_classification = None
        self._poly_points = None
        self._constant_elev = None
        self._specify_slope = None
        self._slope = None
        self._max_dist = None
        self._method = None
        self._poly_id = None
        self._transition_method = None
        self._transition_distance = None
        self._quadtree_refinment_length = None
        self._transition_poly_points = None
        self._max_transition_poly_pts = None
        self._locked_polys = None

        # ugrid data extractor used to intersect the slope transects
        self._polyline_extractor = None
        self._setup_extractor()

    def set_polygon_data(self, polygon_data):
        """Set the data for the current polygon.

        Args:
            polygon_data (:obj:`dict`): Holds the outside polygon points and polygon attributes
        """
        # reset with new polygon
        self.transect_polygon = None
        self.non_intersecting_transects = []
        self.transition_polygon = None
        self.ewn_ugrid = None
        self.stitched_ugrid = None
        self._transition_polygon_ugrid_idxs = None
        self._transition_poly_holes_ugrid_idxs = None
        self._transition_polygon_ugrid_remove_pts = None
        self._transition_polygon_ugrid_remove_cells = None
        self._stitch_new_to_old_pt_idx = None
        self._stitch_old_to_new_pt_idx = None
        self._input_poly_segment_distances = None
        self._transect_distances = None
        self._stitched_ugrid_smooth_size = None
        self._locked_cells = None
        self._previous_transect_polys = []

        self._poly_points = polygon_data['polygon_outside_pts_ccw']
        self._poly_classification = polygon_data['classification']
        self._constant_elev = polygon_data['elevation']
        self._specify_slope = False if int(polygon_data['specify_slope']) == 0 else True
        self._slope = polygon_data['slope']
        self._max_dist = polygon_data['max_distance']
        self._method = polygon_data['elevation_method']
        self._poly_id = polygon_data['polygon_id']
        self._transition_method = polygon_data['transition_method']
        self._transition_distance = polygon_data['transition_distance']
        self._quadtree_refinment_length = polygon_data.get('quadtree_refinment_length', None)
        self._transition_poly_points = polygon_data.get('transition_poly_pts', None)
        self._max_transition_poly_pts = None
        self._previous_transect_polys = polygon_data.get('transect_polys', [])

        self._initialize_variable_z()
        self._calc_max_transition_polygon()

    def _calc_max_transition_polygon(self):
        """Find the transition polygon from the original mesh and user inputs."""
        self._logger.info('Generating transition polygon.')
        if self._max_transition_poly_pts is None:
            # get bounding box of input polygon
            xcoord, ycoord, _ = zip(*self._poly_points)
            if self._transition_method == consts.TRANSITION_METHOD_POLYGON and \
                    self._transition_poly_points is not None:
                xcoord, ycoord, _ = zip(*self._transition_poly_points)
            box_min = [min(xcoord), min(ycoord)]
            box_max = [max(xcoord), max(ycoord)]
            dist = 0.0
            if self._transition_method == consts.TRANSITION_METHOD_FACTOR:
                dist = self._transition_distance
                length = math.sqrt((box_max[0] - box_min[0])**2 + (box_max[1] - box_min[1])**2)
                dist = length * dist
            elif self._transition_method == consts.TRANSITION_METHOD_DISTANCE:
                if self._is_geographic:
                    latitude = (box_max[1] + box_min[1]) / 2
                    dist = meters_to_decimal_degrees(self._transition_distance, latitude)
                else:
                    dist = self._transition_distance
            # create a polygon offset by this distance
            box_min = [box_min[0] - dist, box_min[1] - dist]
            box_max = [box_max[0] + dist, box_max[1] + dist]
            if self._transition_method == consts.TRANSITION_METHOD_POLYGON and \
                    self._transition_poly_points is not None:
                self._max_transition_poly_pts = self._transition_poly_points + [self._transition_poly_points[0]]
            else:
                self._max_transition_poly_pts = [
                    (box_min[0], box_min[1], 0.0), (box_max[0], box_min[1], 0.0), (box_max[0], box_max[1], 0.0),
                    (box_min[0], box_max[1], 0.0), (box_min[0], box_min[1], 0.0)
                ]
        self._calc_locked_cells()

    def _calc_locked_cells(self):
        """Flag the cells outside the transition polygon so that they will not be edited.

        Also, if we are processing polygons in the same coverage then we don't want to edit ewn
        features that were already inserted. The outer boundary of the previously inserted features
        is in the  "_previous_transect_polys" variable.
        """
        self._locked_cells = [0] * self._ugrid.cell_count
        max_transition_poly_pts = self._get_ugrid_points_in_polygon(self._max_transition_poly_pts)
        self._polyline_extractor.set_polyline(self._max_transition_poly_pts)
        _ = self._polyline_extractor.extract_data()
        cell_idxs = set(self._polyline_extractor.cell_indexes)
        for idx in max_transition_poly_pts:
            adj_cells = self._ugrid.get_point_adjacent_cells(idx)
            for cell_idx in adj_cells:
                cell_idxs.add(cell_idx)

        self._transect_cell_idxes = set()
        for poly in self._previous_transect_polys:
            ug_pts_set = set(self._get_ugrid_points_in_polygon(poly))
            for idx in ug_pts_set:
                adj_cells = self._ugrid.get_point_adjacent_cells(idx)
                for cell_idx in adj_cells:
                    cell_pts_set = set(self._ugrid.get_cell_points(cell_idx))
                    if cell_pts_set.issubset(ug_pts_set):
                        self._transect_cell_idxes.add(cell_idx)
            # raise error if input polygon points are inside of any previous transect polys
            for pt in self._poly_points:
                if xmg.point_in_polygon_2d(poly, pt) > -1:
                    raise RuntimeError('Input polygon inside of previous ewn transect polygon.')

        # Locked Cells
        # Loop through each cell
        for i in range(self._ugrid.cell_count):
            # Get the points around the cell
            cell_pts = set(self._ugrid.get_cell_points(i))
            locked_cell_pts_count = 0
            # Loop through each point around the cell
            for cell_pt in cell_pts:
                # If the pt is locked increase the count
                if self._lock_dataset[cell_pt] != 0:
                    locked_cell_pts_count += 1
            # if the number of cell points is the same as the number of
            # locked points the cell is locked
            if len(cell_pts) == locked_cell_pts_count:
                self._lock_dataset_cell_idxes.add(i)

        for idx in range(self._ugrid.cell_count):
            if idx not in cell_idxs:
                self._locked_cells[idx] = 1
            if idx in self._transect_cell_idxes:
                self._locked_cells[idx] = 2
            if idx in self._lock_dataset_cell_idxes:
                self._locked_cells[idx] = 3

    def _get_ugrid_points_in_polygon(self, poly):
        """Get a list of ugrid point indexes that are inside the polygon.

        Args:
            poly (:obj:`list`): list of x, y, z coordinates

        Returns:
            (:obj:`list`): list of ugrid point indexes
        """
        # find all ugrid points in the transect_offset polygon
        poly = np.asarray([(p[0], p[1]) for p in poly])
        locs_2d = np.asarray([(p[0], p[1]) for p in self._ugrid.locations])
        result = run_parallel_points_in_polygon(locs_2d, poly)
        inside_points = [i for i, flag in enumerate(result) if flag]
        return inside_points

    def insert_feature(self):
        """Inserts the current polygon into the ugrid."""
        if self._specify_slope:
            if self._slope == 0:
                self._logger.error('Specified slope must be greater than 0.')
                raise RuntimeError
            if self._max_dist == 0:
                self._logger.error('Maximum slope distance must be greater than 0.')
                raise RuntimeError

        self.calc_transect_polygon()

        if self.is_cartesian:
            self._logger.info('Cartesian grid detected.')
            self._update_cartesian_elevations()
            return

        self.calc_transition_polygon()
        self.calc_ewn_ugrid()
        self.calc_stitched_ugrid()
        self.expand_transition_polygon()
        self.calc_ewn_ugrid()
        self.calc_stitched_ugrid()

    def calc_transect_polygon(self):
        """Creates a polygon where the ewn slope transects intersect the background ugrid."""
        self._logger.info('Generating transect polygon.')

        if self._specify_slope:
            self._logger.info('Specified slope is enabled...')
            self._create_transects_from_polygon_points()
            self._clip_transects_to_grid_elevations()
            self._create_transect_polygon()

            transect_shapely = Polygon(self.transect_polygon)
            if not transect_shapely.is_valid:
                self._logger.error(
                    'This polygon is invalid when using slope. Either '
                    'disable slope or simplify the polygon.'
                )
                raise RuntimeError

        else:
            # Setup shapely polygon
            self.shapely_poly = Polygon(self._poly_points)

            # Get average spacing
            spacings = []
            for i in range(len(self.shapely_poly.exterior.coords) - 1):
                point1 = Point(self.shapely_poly.exterior.coords[i])
                point2 = Point(self.shapely_poly.exterior.coords[i + 1])
                spacings.append(point1.distance(point2))
            self.average_spacing = sum(spacings) / len(spacings)
            self.shapely_poly_buffered = self.shapely_poly.buffer(
                distance=self.average_spacing, cap_style=CAP_STYLE.flat, join_style=JOIN_STYLE.mitre
            )

            self.transect_polygon = [(p[0], p[1], 0.0) for p in self.shapely_poly_buffered.exterior.coords]

            redist = meshing.poly_redistribute_points.PolyRedistributePoints()
            redist.set_size_function_from_polygon(self._poly_points, inside_polygons=[], size_bias=0.3)
            self.transect_polygon = redist.redistribute(self.transect_polygon)

        # Make sure we are within the bounding box of the transition polygon
        poly = np.asarray([(p[0], p[1]) for p in self._max_transition_poly_pts])
        locs_2d = np.asarray([(p[0], p[1]) for p in self.transect_polygon])
        result = run_parallel_points_in_polygon(locs_2d, poly)

        if False in result:
            self._logger.error('Transect polygon outside of max transition polygon bounding box.')
            raise RuntimeError

        # use the polyline extractor to see if we go outside of the mesh
        self._polyline_extractor.set_polyline(self.transect_polygon)  # set the line segment on the data extractor
        grid_elev = self._polyline_extractor.extract_data()  # grid elevations at intersection locations

        if not self._specify_slope:
            locations = [list(x) for x in self._polyline_extractor.extract_locations]
            for i in range(len(self.transect_polygon) - 1):
                extract_index = locations.index(list(self.transect_polygon[i]))
                self.transect_polygon[i][2] = grid_elev[extract_index]

        for elev in grid_elev:
            if math.isnan(elev):
                msg = f'Unable to insert polygon id: {self._poly_id} because it is too close to, ' \
                    'or outside of, the mesh boundary.'
                self._logger.error(msg)
                raise RuntimeError

        for poly in self._previous_transect_polys:
            # raise error if transect points are inside of any previous transect polys
            if Polygon(poly).intersects(Polygon(self.transect_polygon)):
                self._logger.error(
                    'Cannot continue inserting features. some features are too close to complete transition.'
                )
                raise RuntimeError

    def calc_transition_polygon(self):
        """Creates a polygon for the transition zone between the transect polygon and unmodified portion of the ugrid.
        """
        self._logger.info('Generating transition polygon.')
        if self.transect_polygon is None:
            self._logger.error('Transect polygon must be defined to calculate transition polygon.')
            return
        # offset the transect polygon
        if self.shapely_poly_buffered:
            transect_offset = [
                [p[0], p[1], 0.0] for p in self.shapely_poly_buffered.buffer(
                    distance=self.average_spacing, cap_style=CAP_STYLE.flat, join_style=JOIN_STYLE.mitre
                ).exterior.coords
            ]
        else:
            new_pt_vectors, distances = _calc_polygon_offset_vectors_distances(self.transect_polygon)
            transect_offset = []
            for idx in range(len(self.transect_polygon) - 1):
                pt = self.transect_polygon[idx]
                uv = new_pt_vectors[idx]
                transect_offset.append([pt[0] + uv[0] * distances[idx], pt[1] + uv[1] * distances[idx], pt[2]])
        xcoord, ycoord, _ = zip(*transect_offset)
        box_min = [min(xcoord), min(ycoord)]
        box_max = [max(xcoord), max(ycoord)]
        inside_pts = []
        pts_flag = [0] * self._ugrid.point_count
        for i, pt in enumerate(self._ugrid.locations):
            if (i + 1) % 10000 == 0:
                self._logger.info('Transect polygon generation: 10,000 points processed.')
            if self._pt_in_box(pt, box_min, box_max):
                if xmg.point_in_polygon_2d(transect_offset, pt) == 1:
                    pts_flag[i] = 1
                    inside_pts.append(i)
        self._transition_polygon_ugrid_remove_pts = pts_flag
        # get all cells attached to the found points
        cell_category = [0] * self._ugrid.cell_count
        for i, pt_idx in enumerate(inside_pts):
            if i % 10000 == 0:
                self._logger.info('Transect polygon generation: 10,000 points processed.')  # pragma no cover
            for cell_idx in self._ugrid.get_point_adjacent_cells(pt_idx):
                if self._locked_cells[cell_idx] == 0:
                    cell_category[cell_idx] = 1
        # if no points were in the polygon then find the cells that the ewn feature is in
        transect_offset.append(transect_offset[0])
        self._polyline_extractor.set_polyline(transect_offset)
        cell_idxs = self._polyline_extractor.cell_indexes
        for idx in cell_idxs:
            if self._locked_cells[idx] == 0:
                cell_category[idx] = 1

        # Check to see if we are editing locked nodes
        if len(self._lock_dataset_cell_idxes) > 0:
            # Get the polygons from the dataset for the locked areas.
            ug_builder = UGridBuilder()
            ug_builder.set_is_2d()
            ug_builder.set_ugrid(self._ugrid)
            co_grid = ug_builder.build_grid()

            poly_builder = GridCellToPolygonCoverageBuilder(
                co_grid=co_grid, dataset_values=self._locked_cells, projection=None, coverage_name='temp'
            )
            lock_polys = poly_builder.find_polygons()
            # Check polygon of each set of lock nodes
            self._locked_polys = []
            for i, poly in enumerate(lock_polys[3]):
                points_locked = False
                lock_holes = []
                # Get the holes for the lock polygons
                if len(poly) > 1:
                    for hole_idxs in poly[1:]:
                        lock_holes.append([self._ugrid.locations[x] for x in hole_idxs])
                # Get the locations for the outer polygon of the hole
                outer_poly = [self._ugrid.locations[x] for x in poly[0]]
                # Check each point of the polygon to see if it is in the transition poly
                for lock_point in outer_poly:
                    if xmg.point_in_polygon_2d(self._poly_points, lock_point) > -1:
                        points_locked = True
                # Check each point of the transect_polygon and see if it is in the lock poly
                for pt in self.transect_polygon:
                    if xmg.point_in_polygon_2d(outer_poly, pt) > -1:
                        in_hole = False
                        # If it is see if it is in a hole
                        for hole in lock_holes:
                            if xmg.point_in_polygon_2d(hole, pt) > -1:
                                in_hole = True
                                break
                        if not in_hole:
                            points_locked = True
                if points_locked:
                    self._locked_polys.append(i)

            if len(self._locked_polys) > 0:
                self._logger.error('Unable to maintain locked nodes. Some nodes in the new feature area were locked.')
                self._logger.error('Locked nodes in the transition polygon were not honored.')
                for poly_id in self._locked_polys:
                    locked_polygon = [self._ugrid.locations[x] for x in lock_polys[3][poly_id]][0]
                    # Loop throug the cells
                    for i in range(self._ugrid.cell_count):
                        # Get each point around the cell
                        cell_pts = set(self._ugrid.get_cell_points(i))
                        # Loop through each cell point
                        for cell_pt in cell_pts:
                            # If any cell_pt is in the lock polygon, remove the cell from the lock indexes
                            if xmg.point_in_polygon_2d(locked_polygon, self._ugrid.locations[cell_pt]) > -1 \
                                    and i in self._lock_dataset_cell_idxes:
                                self._lock_dataset_cell_idxes.remove(i)
                                self._locked_cells[i] = 0
                                cell_category[i] = 1

        self._debug_write_points_to_temp_file(transect_offset)
        self._transition_polygon_ugrid_remove_cells = cell_category
        self._polygon_from_ugrid_cell_categories()

    def calc_ewn_ugrid(self):
        """Generate a ugrid from the defined polygons."""
        self._logger.info('Generating EWN mesh.')
        ug_locs = self._ugrid.locations
        transition_size_function = None
        if self._stitched_ugrid_smooth_size:
            _, dist = _calc_polygon_offset_vectors_distances(self.transect_polygon)
            transition_size_function = [[p[0], p[1], dist[i]] for i, p in enumerate(self.transect_polygon[:-1])]
            for ug_idx in self._transition_polygon_ugrid_idxs:
                stitch_idx = self._stitch_old_to_new_pt_idx[ug_idx]
                loc = ug_locs[ug_idx]
                transition_size_function.append([loc[0], loc[1], self._stitched_ugrid_smooth_size[stitch_idx]])
        mesh_poly_pts = self._poly_points[:-1]
        transect_poly_pts = self.transect_polygon[:-1]
        transition_poly_pts = self.transition_polygon[:-1]
        interp_pts = np.append(np.array(mesh_poly_pts), transect_poly_pts, axis=0)
        interp_pts = np.append(np.array(transition_poly_pts), interp_pts, axis=0)
        # add mesh points that are being removed that are inside the transition polygon but outside the transect poly
        set_idx = set(self._transition_polygon_ugrid_idxs)
        pts_to_add = []
        for i, flag in enumerate(self._transition_polygon_ugrid_remove_pts):
            if i in set_idx:
                continue
            if flag == 1:
                loc = ug_locs[i]
                if xmg.point_in_polygon_2d(transect_poly_pts, loc) == -1:
                    pts_to_add.append(loc)
                    if transition_size_function:
                        stitch_idx = self._stitch_old_to_new_pt_idx[i]
                        if stitch_idx > -1:  # skip points that are not in the stitched ugrid
                            size = self._stitched_ugrid_smooth_size[stitch_idx]
                            transition_size_function.append([loc[0], loc[1], size])
        if len(pts_to_add) > 0:
            interp_pts = np.append(np.array(pts_to_add), interp_pts, axis=0)
        linear = InterpLinear(points=interp_pts)

        linear_size = None
        if transition_size_function:
            linear_size = InterpLinear(points=transition_size_function)

        transition_in_polys = [transect_poly_pts[::-1]]
        if self._transition_poly_holes_ugrid_idxs is not None:
            locs = self._ugrid.locations
            for loop in self._transition_poly_holes_ugrid_idxs:
                rev_loop = loop[::-1]
                transition_in_polys.append([locs[idx] for idx in rev_loop])

        bias = self._mesher_bias
        mp = list(mesh_poly_pts)
        flag = True
        mp.reverse()
        meshing_inputs = [
            meshing.PolyInput(
                outside_polygon=mp, elev_function=linear, bias=bias, remove_internal_four_triangle_points=flag
            ),  # , fix_point_connections=True),
            meshing.PolyInput(
                outside_polygon=transect_poly_pts,
                inside_polygons=[mp[::-1]],
                elev_function=linear,
                bias=bias,
                remove_internal_four_triangle_points=flag
            ),  # , fix_point_connections=True),
            meshing.PolyInput(
                outside_polygon=transition_poly_pts,
                inside_polygons=transition_in_polys,
                elev_function=linear,
                bias=bias,
                size_function=linear_size,
                remove_internal_four_triangle_points=flag
            )  # , fix_point_connections=True),
        ]
        if self._is_levee:
            meshing_inputs.pop(0)
        self.ewn_ugrid = xms.mesher.generate_mesh(polygon_inputs=meshing_inputs)

    def calc_stitched_ugrid(self):
        """Stitches the ewn ugrid and the _ugrid together."""
        self._logger.info('Merging EWN mesh into target mesh.')
        if self.ewn_ugrid is None:
            self._logger.error('Ewn ugrid must be defined to calculate stitched ugrid.')
            return
        self._ugrid_locs = self._ugrid.locations
        self._ewn_locs = self.ewn_ugrid.locations
        self._calc_ewn_grid_new_to_old_pt_idx()
        stitched_pts = self._calc_stitched_points()
        stitched_cells = self._calc_stiched_cell_stream()
        self.stitched_ugrid = xms.grid.ugrid.UGrid(stitched_pts, stitched_cells)
        self._ugrid_locs = None
        self._ewn_locs = None

    def _calc_ewn_grid_new_to_old_pt_idx(self):
        """Fill in array of indexes to points in the original ugrid that are also in the ewn grid."""
        # the indexes of the old ugrid that are present in the new ewn grid are in the
        # self._transition_polygon_ugrid_idxs and self._transition_poly_holes_ugrid_idxs
        # make a dict to look up location and idx of boundary points
        pt_dict = dict()
        for idx in self._transition_polygon_ugrid_idxs:
            p = self._ugrid_locs[idx]
            pt_dict[(p[0], p[1])] = idx
        if self._transition_poly_holes_ugrid_idxs is not None:
            for loop in self._transition_poly_holes_ugrid_idxs:
                for idx in loop:
                    p = self._ugrid_locs[idx]
                    pt_dict[(p[0], p[1])] = idx
        self._stitch_new_to_old_pt_idx = [-1] * len(self._ewn_locs)
        self._stitch_old_to_new_pt_idx = [-1] * len(self._ugrid_locs)
        for idx, p in enumerate(self._ewn_locs):
            if (p[0], p[1]) in pt_dict:
                old_idx = pt_dict[(p[0], p[1])]
                self._stitch_new_to_old_pt_idx[idx] = old_idx
                self._stitch_old_to_new_pt_idx[old_idx] = idx

    def _calc_stitched_points(self):
        """Create new list of the points for the stiched grid (see calc_stitched_ugrid).

        Returns:
            (:obj:`list`): x,y,z coords of stitched ugrid points
        """
        # add the ugrid points that are not being removed and update old to new idx
        stitched_pts = [pt for pt in self._ewn_locs]
        for ug_idx, pt in enumerate(self._ugrid_locs):
            if (ug_idx + 1) % 10000 == 0:
                self._logger.info('Merging meshes- 10,000 points processed.')
            if self._stitch_old_to_new_pt_idx[ug_idx] != -1:  # skip points found above
                continue
            if self._transition_polygon_ugrid_remove_pts[ug_idx] == 0:
                stitched_pts.append(pt)
                stitched_idx = len(stitched_pts) - 1
                self._stitch_old_to_new_pt_idx[ug_idx] = stitched_idx
                self._stitch_new_to_old_pt_idx.append(ug_idx)
        return stitched_pts

    def _calc_stiched_cell_stream(self):
        """Creates new cell stream for the stitched ugrid (see calc_stitched_ugrid).

        Returns:
            (:obj:`list[int]`): list of ints representing ugrid cell stream.
        """
        # create new cell stream
        stitched_cells = [c for c in self.ewn_ugrid.cellstream]
        cell_idx = -1
        cnt = 0
        ug_cellstream = self._ugrid.cellstream
        while cnt < len(ug_cellstream):
            cell_idx += 1
            if (cell_idx + 1) % 10000 == 0:
                self._logger.info('Merging meshes- 10,000 cells processed.')
            cell_type = ug_cellstream[cnt]
            num_pts = ug_cellstream[cnt + 1]
            start = cnt + 2
            end = start + num_pts
            cell_pts = ug_cellstream[start:end]
            cnt = end
            if self._transition_polygon_ugrid_remove_cells[cell_idx] != 1:
                add_cell = [cell_type, num_pts] + [self._stitch_old_to_new_pt_idx[old_idx] for old_idx in cell_pts]
                stitched_cells.extend(add_cell)
        return stitched_cells

    def expand_transition_polygon(self):
        """Expand the transition polygon by smoothing the size function on the stitched grid."""
        self._logger.info('Expanding transition polygon.')
        if self.stitched_ugrid is None:
            self._logger.error('Stitched ugrid must be defined to expand the transition polygon.')
            return
        stitched_size_func = mesh_utils.size_function_from_edge_lengths(self.stitched_ugrid)
        min_size = min(stitched_size_func)
        bias = max(1e-5, self._mesher_bias)
        smooth_size_func = mesh_utils.smooth_size_function_ugrid(
            ugrid=self.stitched_ugrid,
            sizes=stitched_size_func,
            size_ratio=1 - bias,
            min_size=min_size,
            anchor_to='min',
            points_flag=()
        )
        self._stitched_ugrid_smooth_size = smooth_size_func
        tol = min_size * 0.1
        pts_to_visit = list(self._transition_polygon_ugrid_idxs)
        transition_warning = False
        for ug_idx in pts_to_visit:
            stitch_idx = self._stitch_old_to_new_pt_idx[ug_idx]
            # skip if point is already marked to be removed
            if self._transition_polygon_ugrid_remove_pts[ug_idx] == 1:
                continue
            # check if this point should be removed
            if (stitched_size_func[stitch_idx] - smooth_size_func[stitch_idx]) > tol:
                # can't remove point if it is connected to any locked cell
                locked_cell = False
                for cell_idx in self._ugrid.get_point_adjacent_cells(ug_idx):
                    if self._locked_cells[cell_idx] != 0:
                        locked_cell = True
                        if self._locked_cells[cell_idx] == 1:
                            transition_warning = True
                        break
                if locked_cell:
                    continue
                self._transition_polygon_ugrid_remove_pts[ug_idx] = 1
                for cell_idx in self._ugrid.get_point_adjacent_cells(ug_idx):
                    self._transition_polygon_ugrid_remove_cells[cell_idx] = 1
                # add neighbor points to points to visit if not
                adj_pts = self.stitched_ugrid.get_point_adjacent_points(stitch_idx)
                for p in adj_pts:
                    neigh_ug_idx = self._stitch_new_to_old_pt_idx[p]
                    if neigh_ug_idx != -1 and self._transition_polygon_ugrid_remove_pts[neigh_ug_idx] == 0:
                        pts_to_visit.append(neigh_ug_idx)  # noqa B038 editing a loop's mutable iterable

        if transition_warning and self.show_transition_warning:
            msg = 'Size function transition has been clipped to the specified maximum transition zone.'
            self._logger.warning(msg)
        self._polygon_from_ugrid_cell_categories()

    def _initialize_variable_z(self):
        """Load variable polygon Z values if option is enabled."""
        elev = self._constant_elev
        if self._is_levee:
            elev = 0.0
        if self._method == consts.ELEVATION_METHOD_Z_OFFSET or self._is_levee:
            # Setup an extractor to get the mesh elevations at the EWN 2D polygon locations.
            self._polyline_extractor.set_polyline(self._poly_points)
            locs = self._polyline_extractor.extract_locations
            grid_elev = self._polyline_extractor.extract_data()  # grid elevations at intersection locations
            extract_idx = 0
            for poly_point in self._poly_points:
                idx = None
                for i in range(extract_idx, len(locs)):
                    if (poly_point == locs[i]).all():
                        idx = i
                        break
                if idx is not None:
                    poly_point[2] = grid_elev[idx] + elev  # Apply the offset to the source elevation
                    extract_idx = idx

    def _polygon_from_ugrid_cell_categories(self):
        # generate a polygon from the attached cells
        self._logger.info('Generating polygon from mesh cells.')
        ug_builder = UGridBuilder()
        ug_builder.set_is_2d()
        ug_builder.set_ugrid(self._ugrid)
        co_grid = ug_builder.build_grid()
        poly_builder = GridCellToPolygonCoverageBuilder(
            co_grid=co_grid,
            dataset_values=self._transition_polygon_ugrid_remove_cells,
            projection=None,
            coverage_name='temp'
        )
        out_polys = poly_builder.find_polygons()
        self.transition_polygon = [self._ugrid.locations[idx] for idx in out_polys[1][0][0]]
        self._transition_polygon_ugrid_idxs = out_polys[1][0][0]
        if len(self._previous_transect_polys) > 0:
            # if we have process features in the same coverage previously we want to preserve those so we will
            # preserve any "holes" in the transition polygon because those may contain the previous features.
            if len(out_polys[1][0]) > 1:
                self._transition_poly_holes_ugrid_idxs = out_polys[1][0][1:]
        else:
            idx_set = set(self._transition_polygon_ugrid_idxs)
            # check for holes in this polygon and mark all associated points and cells to be removed
            for i in range(1, len(out_polys[1][0])):
                pts_to_visit = list(out_polys[1][0][i])
                for ug_idx in pts_to_visit:
                    if ug_idx in idx_set:  # skip points that are in the transition polygon
                        continue
                    if self._transition_polygon_ugrid_remove_pts[ug_idx] == 1:
                        continue
                    self._transition_polygon_ugrid_remove_pts[ug_idx] = 1
                    for cell_idx in self._ugrid.get_point_adjacent_cells(ug_idx):
                        self._transition_polygon_ugrid_remove_cells[cell_idx] = 1
                    # add neighbor points to points to visit if not
                    adj_pts = self._ugrid.get_point_adjacent_points(ug_idx)
                    for p in adj_pts:
                        if p in idx_set:  # skip points that are in the transition polygon
                            continue
                        if self._transition_polygon_ugrid_remove_pts[p] == 0:
                            pts_to_visit.append(p)  # noqa B038 editing a loop's mutable iterable
        # check for points that are connected to cells that are all being removed
        for idx, val in enumerate(self._transition_polygon_ugrid_remove_pts):
            if val == 0:
                adj_cells = self._ugrid.get_point_adjacent_cells(idx)
                remove_cells = [self._transition_polygon_ugrid_remove_cells[i] for i in adj_cells]
                if remove_cells.count(0) == 0:  # all cells being removed
                    self._transition_polygon_ugrid_remove_pts[idx] = 1
        self._debug_write_points_to_temp_file(self.transition_polygon)

    def _pt_in_box(self, pt, box_min, box_max):
        """See if a point is in or on a box.

        Args:
            pt (x, y, z): location
            box_min (x, y): min corner of box
            box_max (x, y): max corner of box

        Returns:
            (:obj:`bool`): true if the point is in or on the box
        """
        return not (pt[0] < box_min[0] or pt[1] < box_min[1] or pt[0] > box_max[0] or pt[1] > box_max[1])

    def _setup_extractor(self):
        """Creates a data extractor for the UGrid."""
        self._logger.info('Setting up data extractor for mesh')
        locs = self._ugrid.locations
        ug_elevations = [pt[2] for pt in locs]
        ug_activity = [1] * len(ug_elevations)
        self._polyline_extractor = UGrid2dPolylineDataExtractor(ugrid=self._ugrid, scalar_location='points')
        self._polyline_extractor.set_grid_scalars(ug_elevations, ug_activity, 'points')

    def _create_transects_from_polygon_points(self):
        """Create transects that radiate out from each polygon point to the max distance specified by user."""
        self._logger.info('Generating transects from input EWN polygon.')
        # get unit vector directions for each point
        new_pt_vectors, distances = _calc_polygon_offset_vectors_distances(self._poly_points)
        self._input_poly_segment_distances = distances
        self._debug_write_points_to_temp_file(new_pt_vectors)
        # compute line segment
        self._transects = []
        use_computed_distance = self._is_levee or not self._specify_slope
        for i in range(len(new_pt_vectors)):
            pt = self._poly_points[i]
            uv = new_pt_vectors[i]
            self._transects.append((pt[0], pt[1], pt[2]))
            d = self._max_dist if not use_computed_distance else distances[i]
            if self._is_geographic and not use_computed_distance:
                d = meters_to_decimal_degrees(d, latitude=pt[1])
            self._transects.append((pt[0] + uv[0] * d, pt[1] + uv[1] * d, pt[2]))
        self._debug_write_points_to_temp_file(self._transects)

    def _clip_transects_to_grid_elevations(self):
        """Clip the transects to the elevations of the input unstructured grid."""
        self._logger.info('Intersecting transects with mesh elevations.')
        for i in range(0, len(self._transects), 2):
            seg = self._transects[i:i + 2]  # process each segment - 2 points
            self._polyline_extractor.set_polyline(seg)  # set the line segment on the data extractor
            if self._is_levee or not self._specify_slope:
                grid_elev = self._polyline_extractor.extract_data()  # grid elevations at intersection locations
                self._transects[i + 1] = (seg[1][0], seg[1][1], grid_elev[-1])
                continue
            locs = self._polyline_extractor.extract_locations  # get locations where segment intersects ugrid
            t_vals = _locations_to_parametric_values(seg, locs)  # parametric values for intersection locations
            grid_elev = self._polyline_extractor.extract_data()  # grid elevations at intersection locations
            fill_segment = seg[0][2] > grid_elev[0]  # determine if this is a cut or fill
            poly_elev = self._constant_elev if self._method == consts.ELEVATION_METHOD_CONSTANT else seg[0][2]
            slope = self._slope
            dist = self._max_dist
            if fill_segment:
                elev_at_max_dist = poly_elev - slope * dist
            else:
                elev_at_max_dist = poly_elev + slope * dist
            seg[1] = (seg[1][0], seg[1][1], elev_at_max_dist)
            # elevations at intersection locations from user specified slope
            elevations = [poly_elev - (poly_elev - elev_at_max_dist) * t for t in t_vals]

            # check if there is no intersection
            if min(elevations) > max(grid_elev) or max(elevations) < min(grid_elev):
                self.non_intersecting_transects.append(seg)
                msg = f'Segment in slope zone of EWN feature does not intersect the mesh at locations: ' \
                      f'{seg[0]}, {seg[1]}.'
                self._logger.info(msg)
            else:
                fill_segment = elevations[0] > grid_elev[0]  # determine if this is a cut or fill
                for idx in range(len(grid_elev)):
                    intersects = False
                    if fill_segment and grid_elev[idx] > elevations[idx]:
                        intersects = True
                    elif not fill_segment and grid_elev[idx] < elevations[idx]:
                        intersects = True

                    if intersects:
                        # get the parametric value where intersection occurs
                        elev0 = elevations[idx - 1]
                        elev1 = elevations[idx]
                        grid_elev0 = grid_elev[idx - 1]
                        grid_elev1 = grid_elev[idx]
                        dz0 = abs(elev0 - grid_elev0)
                        dz1 = abs(elev1 - grid_elev1)
                        t = dz0 / (dz0 + dz1)
                        new_z = elev0 + t * (elev1 - elev0)
                        # calculate the coordinate of the intersection
                        new_t = t_vals[idx - 1] + (t_vals[idx] - t_vals[idx - 1]) * t
                        new_x = seg[0][0] + new_t * (seg[1][0] - seg[0][0])
                        new_y = seg[0][1] + new_t * (seg[1][1] - seg[0][1])
                        self._transects[i + 1] = (new_x, new_y, new_z)
                        break
        self._debug_write_points_to_temp_file(self._transects)
        if self.non_intersecting_transects:
            self._logger.error(
                'One or more segments in the slope zone do not intersect with the mesh. '
                'Adjust slope settings to ensure segments instersect with the mesh.'
            )
            raise RuntimeError

    def _create_transect_polygon(self):
        """Create a polygon from the clipped transect end points.

        Also, redistribute the new polygon to match the spacing of the input polygon.
        """
        self._logger.info('Creating polygon from transect intersection with mesh elevations.')

        redist = meshing.poly_redistribute_points.PolyRedistributePoints()
        redist.set_size_function_from_polygon(
            outside_polygon=self._poly_points, inside_polygons=(()), size_bias=self._mesher_bias
        )

        if self._is_levee or not self._specify_slope:
            self.transect_polygon = [[p[0], p[1], p[2]] for p in self._transects[1::2]]
            self.transect_polygon.append(self.transect_polygon[0])
            self.transect_polygon = redist.redistribute(self.transect_polygon)
            self._set_grid_elevations_on_transect_polygon()
            return

        min_dist = self._max_dist
        if self._is_geographic:
            min_dist = meters_to_decimal_degrees(self._max_dist, self._transects[0][1])
        min_dist_idx = -1
        self._transect_distances = []
        for i in range(0, len(self._transects), 2):
            p0 = self._transects[i]
            p1 = self._transects[i + 1]
            dist = math.sqrt((p0[0] - p1[0])**2 + (p0[1] - p1[1])**2)
            self._transect_distances.append(dist)
            if dist < min_dist:
                min_dist = dist
                min_dist_idx = i

        # starting at position 1 get every other point
        self.transect_polygon = [[p[0], p[1], p[2]] for p in self._transects[1::2]]
        # rotate the transect_polygon about the min_dist_idx
        self.transect_polygon = self.transect_polygon[min_dist_idx:] + self.transect_polygon[:min_dist_idx]
        self._transect_distances = self._transect_distances[min_dist_idx:] + self._transect_distances[:min_dist_idx]
        self.transect_polygon.append(self.transect_polygon[0])  # make a ring - last point = first point

        # Redistribute the transect polygon
        # find points that can be redistributed. These points are at least the length of the size function
        # away from the input polygon
        redist_pts = [0] * len(self.transect_polygon)
        for i in range(1, len(self.transect_polygon) - 1):
            # if dist is < input polygon segment length then don't redistribute the point
            if self._transect_distances[i] >= self._input_poly_segment_distances[i]:
                redist_pts[i] = 1
        # find the start and end indexes for lines that can be redistributed
        start_end_idxs = []
        i = 0
        while i < len(redist_pts) - 1:
            start = i
            end = redist_pts.index(0, i + 1)
            start_end_idxs.append([start, end])
            i = end
        new_transect_poly = []
        for item in start_end_idxs:
            start = item[0]
            end = item[1]
            # add all redistributed points except the last one
            grid_diagonal_length = LineString(self._ugrid.extents).length
            if self._is_geographic:
                grid_diagonal_length = abs(decimal_degrees_to_meters(grid_diagonal_length, self._ugrid.extents[0][0]))
            if self._specify_slope and self._max_dist > grid_diagonal_length:
                self._logger.error(
                    f'Specified maximum slope distance ({self._max_dist}) is '
                    f'more than is allowable. Maximum allowable is: {grid_diagonal_length:.2f}'
                )
                raise RuntimeError
            redist_line = redist.redistribute(self.transect_polygon[start:end + 1])
            if len(redist_line) < 1:
                redist_line = self.transect_polygon[start:end + 1]
            new_transect_poly.extend([x for x in redist_line[:-1]])
        new_transect_poly.append(new_transect_poly[0])
        self.transect_polygon = new_transect_poly
        # self.transect_polygon = redist.redistribute(self.transect_polygon)

        self._set_grid_elevations_on_transect_polygon()
        self._debug_write_points_to_temp_file(self.transect_polygon)

    def _set_grid_elevations_on_transect_polygon(self):
        """Intersects the transect polygon with the ugrid and sets the elevation."""
        self._polyline_extractor.set_polyline(self.transect_polygon)  # set the line segment on the data extractor
        locs = self._polyline_extractor.extract_locations  # get locations where segment intersects ugrid
        grid_elev = self._polyline_extractor.extract_data()  # grid elevations at intersection locations

        idx = 0
        extract_index = 0
        for tpt in self.transect_polygon:
            for i in range(extract_index, len(locs)):
                if (tpt == locs[i]).all():
                    idx = i
                    break
            tpt[2] = grid_elev[idx]
            extract_index = idx

    def _debug_write_points_to_temp_file(self, pts):  # pragma no cover
        """Writes points to a temp file for debugging.

        Args:
            pts (:obj:`list[tuple(x,y,z)]`): xyz coordinates of points

        """
        if not self._debug:
            return
        with open('c:/temp/pts.txt', 'w') as file:
            for p in pts:
                file.write(f'{p[0]} {p[1]} {p[2]}\n')

    def _update_cartesian_elevations(self):
        """Compute a cell map and an elevation for the changing cells."""
        self._transect_cell_idxes = set()
        ug_pts_set = set(self._get_ugrid_points_in_polygon(self.transect_polygon))
        for idx in ug_pts_set:
            adj_cells = self._ugrid.get_point_adjacent_cells(idx)
            for cell_idx in adj_cells:
                cell_pts_set = set(self._ugrid.get_cell_points(cell_idx))
                if cell_pts_set.issubset(ug_pts_set):
                    self._transect_cell_idxes.add(cell_idx)

        self.cgrid_elevations = [None] * len(self._transect_cell_idxes)
        for i, cell_idx in enumerate(self._transect_cell_idxes):
            self.cgrid_elevations[i] = (cell_idx, self._constant_elev, self._method)


def _locations_to_parametric_values(segment, locations):
    """Compute the parametric value of points in 'locations' on the line defined by 'segment'.

    Args:
        segment (:obj:`list[tuples (x,y,z)]`): xyz coords of points
        locations (:obj:`list[tuples (x,y,z)]`): xyz coords of points

    Returns:
        (:obj:`list[float]`): parametric value of each point in 'locations' on line defined by segment
    """
    dx = segment[1][0] - segment[0][0]
    dy = segment[1][1] - segment[0][1]
    if abs(dx) > abs(dy):
        x = segment[0][0]
        parametric_values = [(loc[0] - x) / dx for loc in locations]
    else:
        y = segment[0][1]
        parametric_values = [(loc[1] - y) / dy for loc in locations]
    return parametric_values


def get_direction_from_edges(prev_pt, pt, next_pt):
    """Compute an average vector orthogonal to 2 segments defined by 3 points.

    Args:
        prev_pt (:obj:`tuple(x,y,z)`): coordinates
        pt (:obj:`tuple(x,y,z)`): coordinates
        next_pt (:obj:`tuple(x,y,z)`): coordinates

    Returns:
        (:obj:`tuples (x,y)`): unit vector
    """
    if prev_pt is not None and next_pt is not None:
        v1 = (pt[0] - prev_pt[0], pt[1] - prev_pt[1])
        mag_v1 = math.sqrt(v1[0]**2 + v1[1]**2)
        ov1 = (v1[1] / mag_v1, -v1[0] / mag_v1)
        v2 = (next_pt[0] - pt[0], next_pt[1] - pt[1])
        mag_v2 = math.sqrt(v2[0]**2 + v2[1]**2)
        ov2 = (v2[1] / mag_v2, -v2[0] / mag_v2)
        vec_add = ov1[0] + ov2[0], ov1[1] + ov2[1]
        vec_mag = math.sqrt(vec_add[0]**2 + vec_add[1]**2)
        return vec_add[0] / vec_mag, vec_add[1] / vec_mag, 0.0
    else:
        if prev_pt is not None:
            v1 = (pt[0] - prev_pt[0], pt[1] - prev_pt[1])
            mag_v1 = math.sqrt(v1[0]**2 + v1[1]**2)
            return v1[1] / mag_v1, -v1[0] / mag_v1, 0.0
        else:
            v2 = (next_pt[0] - pt[0], next_pt[1] - pt[1])
            mag_v2 = math.sqrt(v2[0]**2 + v2[1]**2)
            return v2[1] / mag_v2, -v2[0] / mag_v2, 0.0


def _calc_polygon_offset_vectors_distances(poly_pts):
    """Calculate unit vectors of direction for offsetting a polygon.

    Args:
        poly_pts (:obj:`list`): list of x,y,z for a polygon

    Returns:
        (:obj:`list`): list of x,y,z vectors
    """
    new_pt_vectors = []
    distances = []
    prev_pt = poly_pts[-2]
    for idx in range(len(poly_pts) - 1):
        pt = poly_pts[idx]
        next_pt = poly_pts[idx + 1]
        d1 = math.sqrt((pt[0] - prev_pt[0])**2 + (pt[1] - prev_pt[1])**2)
        d2 = math.sqrt((pt[0] - next_pt[0])**2 + (pt[1] - next_pt[1])**2)
        distances.append((d1 + d2) / 2)
        new_pt_vectors.append(get_direction_from_edges(prev_pt, pt, next_pt))
        prev_pt = pt
    return new_pt_vectors, distances


def _factor_from_latitude(latitude):
    """Computes meters, decimal degrees conversion factor from latitude.

    Args:
        latitude (:obj:`float`): the latitude

    Returns:
        (:obj:`float`): conversion factor
    """
    return 111.32 * 1000 * math.cos(latitude * (math.pi / 180))


def meters_to_decimal_degrees(length_meters, latitude):
    """Convert meters to decimal degrees based on the latitude.

    Args:
        length_meters (:obj:`float`): length in meters
        latitude (:obj:`float`): latitude in decimal degrees

    Returns:
        (:obj:`float`): length in decimal degrees
    """
    return length_meters / _factor_from_latitude(latitude)


def decimal_degrees_to_meters(length_decimal_degrees, latitude):
    """Convert decimal degrees to meters based on the latitude.

    Args:
        length_decimal_degrees (:obj:`float`): length in decimal degrees
        latitude (:obj:`float`): latitude in decimal degrees

    Returns:
        (:obj:`float`): length in meters
    """
    return length_decimal_degrees * _factor_from_latitude(latitude)
