"""Provide progress feedback when running an FUNWAVE model from SMS."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules


class FunwaveTracker:
    """Class to track the progress of a running FUNWAVE model."""
    prog = None
    query = None
    echo_file = None
    last_line_count = 0

    @staticmethod
    def progress_function():
        """Method called inside the progress loop for computing percent done."""
        if not FunwaveTracker.echo_file:
            FunwaveTracker.echo_file = FunwaveTracker.prog.command_line_output_file

        need_progress_update = False
        line_count = 0
        try:  # Parse the command line output to determine current timestep
            with open(FunwaveTracker.echo_file, "r") as f:
                lines = f.readlines()
                line_count = len(lines)
                need_progress_update = line_count > FunwaveTracker.last_line_count
        except Exception:
            pass  # File might not exist yet

        if need_progress_update:  # Update model process progress in SMS Simulation Run Queue
            percent_done = line_count / 4
            FunwaveTracker.query.update_progress_percent(percent_done)
            FunwaveTracker.last_line_count = line_count

    @staticmethod
    def start_tracking():
        """Entry point for the FUNWAVE progress script."""
        FunwaveTracker.query = Query(progress_script=True)
        FunwaveTracker.prog = FunwaveTracker.query.xms_agent.session.progress_loop

        FunwaveTracker.prog.set_progress_function(FunwaveTracker.progress_function)
        FunwaveTracker.prog.start_loop()
