"""This module defines data for the grid hidden component."""

# 1. Standard Python modules
import os

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem as xfs

# 4. Local modules
from xms.gencade.components.id_files import UNINITIALIZED_COMP_ID


class GridData(XarrayBase):
    """Manages data file for the hidden save arcs component."""

    def __init__(self, data_file):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        self._filename = data_file
        self._info = None
        self._points = None
        self.deleted_comp_ids = set()  # Set this to be a list of deleted component ids in the coverage before vacuuming
        # Create the default file before calling super because we have our own attributes to write.
        created = self._get_default_datasets(data_file)
        super().__init__(data_file)
        if created:
            self.add_grid_point_atts()
            self.commit()
        # self._check_for_simple_migrations()

    def _get_default_datasets(self, data_file):
        """Create default datasets if needed.

        Args:
            data_file (:obj:`str`): Name of the data file. If it doesn't exist, it will be created.

        Returns:
            (:obj:`bool`): Returns True if datasets were created.
        """
        if not os.path.exists(data_file) or not os.path.isfile(data_file):
            info = {
                'FILE_TYPE': 'GENCADE_GRID',
                # 'VERSION': pkg_resources.get_distribution('gencade').version,
                'cov_uuid': '',
                'point_display_uuid': '',
                'next_comp_id': 0,
                'proj_dir': os.path.dirname(os.environ.get('XMS_PYTHON_APP_PROJECT_PATH', '')),
                'base_cell_size': 50,
            }
            self._info = xr.Dataset(attrs=info)

            point_table = {
                'point_type': ('comp_id', np.array([], dtype=object)),
                'refine_size_const': ('comp_id', np.array([], dtype=float)),
            }
            coords = {
                'comp_id': np.array([], dtype=int)
            }
            self.points = xr.Dataset(data_vars=point_table, coords=coords)
            self.commit()
            return True
        else:
            return False

    @staticmethod
    def _clean_dataset_keys(dataset):
        """Removes '/' from the dataset keys for safe storage in h5.

        Args:
            dataset (:obj:`xarray.Dataset`): The dataset that may have bad keys.

        Returns:
            (:obj:`xarray.Dataset`): A new xr.Dataset with sanitized keys.
        """
        column_names = dataset.keys()
        rename_dict = {}
        for col in column_names:
            if col.find('/') >= 0:
                rename_dict[col] = col.replace('/', '%slash%')
        if rename_dict:
            dataset = dataset.rename(rename_dict)
        return dataset

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming
        if self._points is not None:
            self._points.close()
            self._drop_h5_groups(['points'])
            self.points.to_netcdf(self._filename, group='points', mode='a')

    def vacuum(self):
        """Rewrite all SimData to a new/wiped file to reclaim disk space.

        All BC datasets that need to be written to the file must be loaded into memory before calling this method.
        """
        if self._info is None:
            self._info = self.get_dataset('info', False)
        if self._points is None:
            self._points = self.get_dataset('points', False)
        xfs.removefile(self._filename)
        self.commit()  # Rewrite all datasets

    @property
    def points(self):
        """Load the points dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the arcs datasets in the main file
        """
        if self._points is None:
            self._points = self.get_dataset('points', False)
        return self._points

    @points.setter
    def points(self, dset):
        """Setter for the points attribute."""
        if dset:
            self._points = dset

    def update_grid_point(self, comp_id, new_atts):
        """Update the Grid point attributes of a grid point.

        Args:
            comp_id (:obj:`int`): Component id of the grid point to update
            new_atts (:obj:`dict`): The new attributes for the grid point
        """
        self.points['point_type'].loc[dict(comp_id=[comp_id])] = new_atts['point_type']
        self.points['refine_size_const'].loc[dict(comp_id=[comp_id])] = new_atts['refine_size_const']

    def add_grid_point_atts(self, dset=None):
        """Add the grid point attribute dataset for a point.

        Args:
            dset (:obj:`xarray.Dataset`): The attribute dataset to concatenate. If not provided,
                a new Dataset of default attributes will be generated.

        Returns:
            (:obj:`tuple(int)`): The newly generated component id
        """
        try:
            new_comp_id = self.info.attrs['next_comp_id'].item()
            self.info.attrs['next_comp_id'] += 1  # Increment the unique XMS component id.
            if dset is None:  # Generate a new default Dataset
                dset = self._get_new_point_atts(new_comp_id)
            else:  # Update the component id of an existing Dataset
                dset.coords['comp_id'] = [new_comp_id for _ in dset.coords['comp_id']]
            self._points = xr.concat([self.points, dset], 'comp_id')
            return new_comp_id
        except Exception:
            return UNINITIALIZED_COMP_ID

    @staticmethod
    def _get_new_point_atts(comp_id):
        """Get a new dataset with default attributes for a grid point.

        Args:
            comp_id (:obj:`int`): The unique XMS component id of the grid point. If UNINITIALIZED_COMP_ID, a new one is
                generated.

        Returns:
            (:obj:`xarray.Dataset`): A new default dataset for a grid point. Can later be concatenated to
            persistent dataset.
        """
        point_table = {
            'point_type': ('comp_id', np.array(['Unassigned'], dtype=object)),
            'refine_size_const': ('comp_id', np.array([1.0], dtype=float)),
        }
        coords = {
            'comp_id': [comp_id]
        }
        ds = xr.Dataset(data_vars=point_table, coords=coords)
        return ds
