"""The Structure properties dialog."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.gencade.gui.grid_options_dlg_ui import Ui_GridOptions


class GridOptionsDlg(XmsDlg):
    """A dialog for editing grid coverage properties."""

    def __init__(self, parent, dlg_data):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (:obj:`QWidget`): Parent window
            dlg_data (:obj:`dict`): data for the dialog
        """
        super().__init__(parent, 'xms.gencade.gui.grid_options_dlg')
        self._help_url = 'https://cirpwiki.info/wiki/GenCade_2.0_Grid_Definition'
        self._units = dlg_data['units']
        self._dlg_data = dlg_data
        self.ui = Ui_GridOptions()
        self.ui.setupUi(self)
        self.dbl_validator = None
        self._setup_ui()

    def _setup_ui(self):
        """Sets up the widgets for the UI."""
        self.dbl_validator = QxDoubleValidator(parent=self, bottom=1.0)
        self._add_validators()
        self._add_button_box()
        self._load_data()

    def _load_data(self):
        """Populate the widgets."""
        self.ui.base_cell_size_label.setText(f'Base cell size {self._dlg_data["units"]}')
        self.ui.base_cell_size.setText(str(self._dlg_data['base_cell_size']))

    def _add_validators(self):
        """Adds widgets to the main layout."""
        self.ui.base_cell_size.setValidator(self.dbl_validator)

    def _add_button_box(self):
        """Adds the button box to the bottom of the dialog."""
        self.ui.buttonBox.accepted.connect(self.accept)
        self.ui.buttonBox.rejected.connect(self.reject)
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def get_comp_data(self):
        """Returns the options set in the dialog.

        Returns:
            (:obj:`dict{'base_cell_size': ###}`)
        """
        return {
            'base_cell_size': float(self.ui.base_cell_size.text()),
        }
