"""GenericModelCreatorBase class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from datetime import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel, GmiName, Group, Parameter, Section

# 4. Local modules
from xms.gssha.data import mapping_tables


class GenericModelCreatorBase:
    """Base class for creating GenericModel objects.

    By convention, parameter names are the same as the GSSHA card and are upper case. If there is no matching GSSHA
    card, the parameter names are lower case.
    """
    def __init__(self):
        """Initializes the class."""
        self._gm = None
        self._section: Section | None = None  # Current section (e.g. global_parameters, model_parameters)
        self._group: 'Group | None' = None  # Current group
        self._descriptions: dict[str, str]

    def create(self) -> GenericModel:
        """Creates and returns the generic model.

        Returns:
            (GenericModel): The generic model.
        """
        raise NotImplementedError()  # pragma no cover - don't test this

    def _desc(self, name: str) -> str:
        """Returns the description associated with the parameter name."""
        return self._descriptions[name]

    def _add_group(self, group_name: GmiName, label: str) -> Group:
        """Adds a group."""
        self._group = self._section.add_group(group_name, label, description=self._desc(group_name))
        return self._group

    def _date_time(self, name: GmiName, label: str, default: datetime) -> Parameter:
        """Helper function to add a date_time parameter."""
        return self._group.add_date_time(name, label, default, self._desc(name))

    def _int(
        self, name: GmiName, label: str, default: int, low: int = -2147483648, high: int = 2147483647
    ) -> Parameter:
        """Helper function to add an integer parameter."""
        return self._group.add_integer(name, label, default, low, high, self._desc(name))

    def _float(
        self,
        name: GmiName,
        label: str,
        default: float,
        low: float = -1.79769e+308,
        high: float = 1.79769e+308
    ) -> Parameter:
        """Helper function to add a float parameter."""
        return self._group.add_float(name, label, default, low, high, self._desc(name))

    def _xy_series(self, name: GmiName, label: str, axis_titles: list[str]):
        """Helper function to add a xy series parameter."""
        return self._group.add_curve(name, label, axis_titles, description=self._desc(name))

    def _option(self, name: GmiName, label: str, default: str, options: list | None = None) -> Parameter:
        """Helper function to add an option parameter."""
        return self._group.add_option(name, label, default, options, self._desc(name))

    def _dataset(self, name: GmiName, label: str) -> Parameter:
        """Helper function to add an option parameter."""
        return self._group.add_dataset(name, label, self._desc(name))

    def _bool(self, name: GmiName, label: str, default: bool) -> Parameter:
        """Helper function to add a boolean parameter."""
        return self._group.add_boolean(name, label, default, self._desc(name))

    def _text(self, name: GmiName, label: str, default: str = '', required: bool = False) -> Parameter:
        """Helper function to add a text parameter."""
        return self._group.add_text(name, label, default, self._desc(name), required)

    # def _checkbox_curve(
    #     self,
    #     name: GmiName,
    #     label: str,
    #     checkbox_label: str,
    #     axis_titles: list[str],
    #     modes: tuple[Curve, Curve],
    #     default_check_state: bool,
    #     default_constant: float = 0.0,
    #     low: float = -1.79769e+308,
    #     high: float = 1.79769e+308,
    #     other_description_key: str = '',
    # ):
    #     """Helper function to add a checkbox_curve parameter."""
    #     description = self._descriptions[other_description_key] if other_description_key else ''
    #     return self._group.add_checkbox_curve(
    #         name=name,
    #         label=label,
    #         checkbox_label=checkbox_label,
    #         axis_titles=axis_titles,
    #         modes=modes,
    #         default_check_state=default_check_state,
    #         default_constant=default_constant,
    #         low=low,
    #         high=high,
    #         description=self._desc(name),
    #         checkbox_description=description
    #     )

    def _table(self, name: str, default, table_definition):
        """Adds a table parameter."""
        self._group.add_table(name, '', default, table_definition, self._desc(name), display_table=True)

    def _mapping_table(self, mapping_table_name: str):
        """Adds a mapping table parameter."""
        table_definition = mapping_tables.table_def(mapping_table_name)
        self._table(mapping_tables.table_param(mapping_table_name), [], table_definition)

    def _flags(self, flags: dict, key: str) -> dict:
        """Helper function to return a copy of flags dict with one of the flags set to True."""
        flags_copy = flags.copy()
        flags_copy[key] = True
        return flags_copy
