"""Qt delegate for a curve editor button displaying a curve preview."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules
from PySide2.QtCore import QEvent, QPointF, QSize, Qt
from PySide2.QtGui import QBrush, QPen, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor

# 4. Local modules


class CurveButtonDelegate(QStyledItemDelegate):
    """Qt delegate for a curve editor button displaying a curve preview."""
    def __init__(self, get_dataframe_method, set_dataframe_method, parent=None):
        """Initializes the class.

        Args:
            get_dataframe_method (obj): The method that returns a dataframe, and new curve id, given a curve id.
            set_dataframe_method (obj): The method that sets a dataframe with a curve id.
            parent (QObject): The parent object.
        """
        super().__init__(parent)
        self.parent_dlg = parent
        self.pt_cache = {}
        self.size_cache = {}
        self.get_dataframe_method = get_dataframe_method
        self.set_dataframe_method = set_dataframe_method
        self.series_name = ''
        self.dialog_title = ''
        self.icon = None

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton('Edit Curve...')
            height = option.rect.height()
            width = option.rect.width()
            min_dim = min(height, width)
            icon_size = QSize(min_dim, min_dim)
            btn.setIconSize(icon_size)
            curve_id = int(index.data())
            if curve_id <= 0:
                points = []
            elif curve_id not in self.pt_cache or self.size_cache[curve_id] != icon_size:
                df, curve_id = self.get_dataframe_method(curve_id)
                points = self._get_curve_preview_points(df, icon_size)
                self.pt_cache[curve_id] = points
                self.size_cache[curve_id] = icon_size
            else:
                points = self.pt_cache[curve_id]
            if points:
                # Set the pen and draw the border rectangle.
                old_pen = painter.pen()
                pen = QPen()
                pen.setStyle(Qt.SolidLine)
                pen.setColor(Qt.black)
                painter.setPen(pen)
                # add the offsets
                points = copy.deepcopy(points)
                min_x = points[0].x() + option.rect.x()
                min_y = points[0].y() + option.rect.y()
                max_x = min_x
                max_y = min_y
                for pt in points:
                    x = pt.x() + option.rect.x()
                    y = pt.y() + option.rect.y()
                    pt.setX(x)
                    pt.setY(y)
                    min_x = min(min_x, x)
                    min_y = min(min_y, y)
                    max_x = max(max_x, x)
                    max_y = max(max_y, y)
                painter.drawRect(min_x, min_y, max_x - min_x, max_y - min_y)

                # draw the line
                pen.setColor(Qt.red)
                painter.setPen(pen)
                painter.drawPolyline(points)

                # reset the pen
                painter.setPen(old_pen)
            else:
                btn.setFixedWidth(width)
                btn.setFixedHeight(height)

                pix = QPixmap(option.rect.size())
                btn.render(pix)
                painter.drawPixmap(option.rect.topLeft(), pix)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Called when the XY series editor button is clicked.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                curve_id = int(index.data())
                del_cache = curve_id in self.pt_cache
                df, curve_id = self.get_dataframe_method(curve_id)
                # xy_series dialog
                dlg = XySeriesEditor(
                    data_frame=df,
                    series_name=self.series_name,
                    dialog_title=self.dialog_title,
                    icon=self.icon,
                    parent=self.parent_dlg
                )
                if dlg.exec_():
                    df = dlg.model.data_frame
                    self.set_dataframe_method(curve_id, df)
                    index.model().setData(index, curve_id)
                    if del_cache:
                        del self.pt_cache[curve_id]
                return True

        return super().editorEvent(event, model, option, index)

    @staticmethod
    def _get_curve_preview_points(dataframe, graph_size):
        """Gets the points of the curve to draw onto the button.

        Args:
            dataframe (pandas.DataFrame): The dataframe that holds the curve
            graph_size (QSize): The size of the drawing area on the button.
        """
        zero_tol = 0.000001
        x_col = 0
        y_col = 1

        # Get the drawing area bounds to calculate line locations.
        graph_width = graph_size.width()
        graph_height = graph_size.height()
        xmin = dataframe.iloc[:, x_col].min()
        xmax = dataframe.iloc[:, x_col].max()
        ymin = dataframe.iloc[:, y_col].min()
        ymax = dataframe.iloc[:, y_col].max()
        # add 10% margin and reset dx & dy
        dx = (xmax - xmin) * 0.1
        dy = (ymax - ymin) * 0.1
        xmin -= dx
        xmax += dx
        ymin -= dy
        ymax += dy
        dx = xmax - xmin
        dy = ymax - ymin
        # set scale factors
        sx = graph_width / dx if dx > zero_tol else 0.0
        sy = graph_height / dy if dy > zero_tol else 0.0
        point_draw = []
        for x_data, y_data in zip(dataframe.iloc[:, x_col], dataframe.iloc[:, y_col]):
            if sx == 0.0:
                x = 0.5 * graph_width
            else:
                x = (x_data - xmin) * sx

            if sy == 0.0:
                y = 0.5 * graph_height
            else:
                y = (y_data - ymax) * sy * -1.0

            point_draw.append(QPointF(x, y))
        return point_draw
