"""Specialization of QxCbxDelegate that makes it easier to display pretty text in comboboxes."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QApplication, QComboBox, QStyle, QStyleOptionComboBox

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate

# 4. Local modules


class UserDataComboboxDelegate(QxCbxDelegate):
    """Delegate that allows for displaying one value while storing another."""
    def __init__(self, strings, data_type=str, parent=None):
        """Initializes the class.

        Args:
            strings (dict): The combobox options. key=display text, value=user data
            data_type (type): The type of the user data. Will be used to cast the user data string.
            parent (QObject): The parent object.
        """
        super().__init__(parent)
        self._strings = strings
        self._data_type = data_type

    def paint(self, painter, option, index):
        """Override the paint event.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return

        # Find the display text for the current value
        current_value = self._data_type(index.data(Qt.EditRole))
        current_text = current_value
        for display_text, model_data in self._strings.items():
            if current_value == model_data:
                current_text = display_text

        cbx_opt = QStyleOptionComboBox()
        cbx_opt.currentText = current_text
        cbx_opt.rect = option.rect
        cbx_opt.state = option.state
        if index.flags() & Qt.ItemIsEnabled:
            cbx_opt.state |= QStyle.State_Enabled
        cbx_opt.editable = False

        QApplication.style().drawComplexControl(QStyle.CC_ComboBox, cbx_opt, painter)
        QApplication.style().drawControl(QStyle.CE_ComboBoxLabel, cbx_opt, painter)

    def createEditor(self, parent, option, index):  # noqa: N802
        """Creates the combobox and populates it.

        Args:
            parent (QWidget): The parent.
            option (QStyleOptionViewItem): The option
            index (QModelIndex): The index

        Returns:
            QWidget
        """
        self.cb = QComboBox(parent)
        for display_text, model_card in self._strings.items():
            self.cb.addItem(display_text, model_card)
        self.cb.currentIndexChanged.connect(self.on_index_changed)
        return self.cb

    def setEditorData(self, editor, index):  # noqa: N802
        """Overload to store the model text instead of the display text.

        Args:
            editor (QWidget): The editor.
            index (QModelIndex): The index.
        """
        if not editor:
            return
        current_value = index.data(Qt.EditRole)
        cbx_index = editor.findData(current_value)
        if cbx_index >= 0:
            editor.blockSignals(True)
            editor.setCurrentIndex(cbx_index)
            editor.blockSignals(False)
            self.commitData.emit(self.cb)
            self.closeEditor.emit(self.cb)
        editor.showPopup()

    def setModelData(self, editor, model, index):  # noqa: N802
        """Overload to store the model text instead of the display text.

        Args:
            editor (QWidget): The editor.
            model (QAbstractItemModel): The model.
            index (QModelIndex): The index
        """
        if not editor:
            return
        cbx_index = editor.currentIndex()
        if cbx_index >= 0:  # Store the model value instead of the display text
            model.setData(index, editor.itemData(cbx_index), Qt.EditRole)
            self.state_changed.emit(index)
