"""Utilities common to other things in xms.guipy.dialogs."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import contextlib
import os
import re
import sys

# 2. Third party modules
from PySide2.QtGui import Qt
from PySide2.QtWidgets import QApplication

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.resources.resources_util import get_resource_path


def ensure_qapplication_exists():
    """Ensures a QApplication singleton exists. We don't have to call .exec_().

    https://stackoverflow.com/questions/11145583

    Returns:
         The QApplication singleton
    """
    app = QApplication.instance()
    if app is None:
        app = QApplication(sys.argv)
    return app


def get_xms_icon():
    """Returns the full path to the XMS window icon of the XMS process that launched this script.

    Note that this method will return empty string when the script is run outside of the XMS environment.
    """
    app_name = os.environ.get('XMS_PYTHON_APP_NAME')
    if app_name:
        return get_resource_path(f':/resources/icons/{app_name}.ico')
    return ''


def process_id_window_title(first_part: str) -> str:
    """Returns the window title to use when debugging which shows the process ID.

    Args:
        first_part (str):  First part of the title (the regular window title).

    Returns:
        (str): See description.
    """
    return f'{first_part} - Process ID: {str(os.getpid())}'


def can_add_process_id() -> bool:
    """Returns true if we can add the process ID to the window title: in dev (version 99.99) and file hack is found."""
    if os.path.isfile('c:/temp/show_python_pid.dbg'):
        return True
        # I don't see a reason to limit this to development versions
        # version = os.environ.get('XMS_PYTHON_APP_VERSION')
        # if version and version.startswith('99.99'):
        #     return True
    return False


@contextlib.contextmanager
def wait_cursor_context():
    """Context manager for a wait cursor."""
    QApplication.setOverrideCursor(Qt.WaitCursor)
    try:
        yield
    finally:
        QApplication.restoreOverrideCursor()


def smart_float(value: str) -> float:
    """Convert a string to a float, handling both comma and dot decimal and thousands separators.

    Examples:
        "1,234.56" -> 1234.56
        "1.234,56" -> 1234.56
        "1 234,56" -> 1234.56
        "1234,56"  -> 1234.56
        "1234.56"  -> 1234.56
    """
    if value is None:
        raise ValueError("Cannot parse None as float.")

    # Remove spaces used as thousand separators
    v = value.strip().replace(" ", "")

    # Case 1: both "." and "," exist
    if "," in v and "." in v:
        # Rightmost one is decimal separator
        if v.rfind(",") > v.rfind("."):
            # Comma is decimal, dot is thousands
            v = v.replace(".", "")
            v = v.replace(",", ".")
        else:
            # Dot is decimal, comma is thousands
            v = v.replace(",", "")

        return float(v)

    # Case 2: only commas
    if "," in v:
        # If comma every 3 digits → thousands separator
        if re.match(r"^\d{1,3}(,\d{3})+$", v):
            v = v.replace(",", "")
            return float(v)
        else:
            # Else comma is decimal
            return float(v.replace(",", "."))

    # Case 3: only dots
    if "." in v:
        # If dot every 3 digits → thousands separator
        if re.match(r"^\d{1,3}(\.\d{3})+$", v):
            v = v.replace(".", "")
            return float(v)
        else:
            return float(v)  # already valid

    # Case 4: no separators
    return float(v)
