"""Dialog for viewing the node ids of an applied boundary condition arc."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QKeySequence
from PySide2.QtWidgets import QAbstractItemView, QApplication, QDialogButtonBox, QListWidget, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules


class CopyableListWidget(QListWidget):
    """Derived class to allow copying from the list."""
    def __init__(self, parent=None):
        """Construct the widget."""
        super().__init__(parent)

    def keyPressEvent(self, event):  # noqa: N802
        """Overload to capture copy key sequence."""
        if event.matches(QKeySequence.Copy):
            item_count = self.count()
            strings = [self.item(i).text() for i in range(item_count)]
            QApplication.clipboard().setText('\n'.join(strings))
        else:
            super().keyPressEvent(event)


class ViewNodeIdsDlg(XmsDlg):
    """A dialog for viewing snapped BC arc node ids."""
    def __init__(self, node_ids, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            node_ids (:obj:`xarray.DataArray`): The node ids to view
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent, 'xmsadcirc.gui.view_node_ids_dlg')
        self.node_ids = node_ids
        self.setWindowTitle('Node IDs')
        self.setup_ui()

    def setup_ui(self):
        """Add the widgets."""
        # Add the node id list widget.
        v_layout = QVBoxLayout()
        list_widget = CopyableListWidget()
        list_widget.addItems([f'{node_id.item()}' for node_id in self.node_ids])
        list_widget.setEditTriggers(QAbstractItemView.NoEditTriggers)
        list_widget.setSelectionMode(QAbstractItemView.ExtendedSelection)
        v_layout.addWidget(list_widget)
        # Add the OK button box.
        btn_box = QDialogButtonBox()
        btn_box.setOrientation(Qt.Horizontal)
        btn_box.setStandardButtons(QDialogButtonBox.Ok)
        btn_box.accepted.connect(self.accept)
        btn_box.rejected.connect(self.reject)
        v_layout.addWidget(btn_box)
        self.setLayout(v_layout)
