"""Helper class for renedering param objects in Qt dialogs."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import pandas as pd
import param
from PySide2.QtCore import QObject, QSortFilterProxyModel, Signal
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QFileDialog, QGroupBox, QHBoxLayout, QHeaderView, QLabel, QLineEdit, QPlainTextEdit,
    QPushButton, QSizePolicy, QSpinBox, QVBoxLayout
)

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy import settings
from xms.guipy.dialogs.dialog_util import smart_float
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.param import param_util
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator
from xms.guipy.widgets.qx_table_view import QxTableView


class ParamQtHelper(QObject):
    """A dialog for assigning materials to polygons."""

    end_do_param_widgets = Signal()

    def __init__(self, parent_dialog):
        """Initializes the class, sets up the ui, and writes the model control values.

        Args:
            parent_dialog (QObject): The paren dialog
        """
        super().__init__()
        self.parent_dialog = parent_dialog
        self.param_dict = dict()
        self.doing_param_widgets = False
        self.param_horizontal_layouts = dict()
        self.param_groups = dict()

    def add_params_to_layout(self, layout, param_parent):
        """Add param objects to the layout.

        Args:
            layout (QBoxLayout): The layout to append to
            param_parent (QObject): Qt parent of the param
        """
        # get param classes ordered by original precedence and add them to the vertical layout
        names, params = param_util.names_and_params_from_class(param_parent)
        pdict = param_util.declared_precedence_dict(param_parent)
        lst = []
        for i in range(len(names)):
            lst.append((float(pdict[names[i]]), params[i], names[i]))
        lst.sort()
        for item in lst:
            self.add_param(layout, item[1], item[2], param_parent)

    def add_param(self, layout, param_obj, param_name, parent_class):  # noqa: C901
        """Add param objects.

        Args:
            layout: The layout to append to
            param_obj: The param object
            param_name: param name of the item
            parent_class: param parent of the item
        """
        layout_in = layout
        ptype = type(param_obj)
        val = getattr(parent_class, param_name)
        if val is None and ptype == param.ClassSelector:
            return
        # widgets = widget_depends[param_name] if widget_depends and param_name in widget_depends else []
        widgets = []

        layout_info = None
        if hasattr(parent_class, 'param_layout') and param_name in parent_class.param_layout:
            layout_info = parent_class.param_layout[param_name]

        if layout_info and layout_info.group_id is not None:
            label = ''
            if layout_info.group_label is not None:
                label = layout_info.group_label
            if layout_info.group_id not in self.param_groups:
                widgets.append(QGroupBox(label))
                layout.addWidget(widgets[-1])
                vertical_layout = QVBoxLayout()
                widgets[-1].setLayout(vertical_layout)
                self.param_groups[layout_info.group_id] = vertical_layout
            layout = self.param_groups[layout_info.group_id]

        label_str = param_obj.label
        if len(label_str) > 0 and ptype != param.Boolean and ptype != param.Action:
            widgets.append(QLabel(label_str + ':'))
            layout.addWidget(widgets[-1])

        if layout_info and layout_info.horizontal_layout:
            if layout_info.horizontal_layout in self.param_horizontal_layouts:
                layout = self.param_horizontal_layouts[layout_info.horizontal_layout]
            else:
                horiz_layout = QHBoxLayout()
                self.param_horizontal_layouts[layout_info.horizontal_layout] = horiz_layout
                layout.addLayout(horiz_layout)
                layout = horiz_layout

        if ptype == param.ObjectSelector:
            widgets.append(QComboBox())
            layout.addWidget(widgets[-1])
            widgets[-1].addItems(param_obj.objects)
            widgets[-1].currentIndexChanged.connect(lambda: self.do_param_widgets(param_obj))
            widgets[-1].currentIndexChanged.connect(lambda: self.on_end_do_param_widgets())
            widgets[-1].setAccessibleName(param_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setCurrentText
            value_getter = widgets[-1].currentText
        elif ptype == param.Number:
            widgets.append(QLineEdit())
            layout.addWidget(widgets[-1])
            validator = QxDoubleValidator(parent=self)
            if param_obj.bounds is not None:
                if param_obj.bounds[0] is not None:
                    validator.setBottom(param_obj.bounds[0])
                if param_obj.bounds[1] is not None:
                    validator.setTop(param_obj.bounds[1])
            widgets[-1].setValidator(validator)
            widgets[-1].editingFinished.connect(lambda: self.do_param_widgets(param_obj))
            widgets[-1].editingFinished.connect(lambda: self.on_end_do_param_widgets())
            widgets[-1].setAccessibleName(param_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = lambda: smart_float(widgets[-1].text())  # noqa: E731
        elif ptype == param.Integer:
            widgets.append(QLineEdit())
            layout.addWidget(widgets[-1])
            widgets[-1].setValidator(QxIntValidator())
            widgets[-1].editingFinished.connect(lambda: self.do_param_widgets(param_obj))
            widgets[-1].editingFinished.connect(lambda: self.on_end_do_param_widgets())
            widgets[-1].setAccessibleName(param_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = lambda: int(widgets[-1].text())  # noqa: E731
        elif ptype == param.Action:
            widgets.append(QPushButton(label_str))
            layout.addWidget(widgets[-1])
            widgets[-1].setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
            widgets[-1].clicked.connect(val)
            widgets[-1].setAccessibleName(param_name)
            value_widget = widgets[-1]
            value_setter = None
            value_getter = None
        elif ptype == param.String:
            if layout_info and not layout_info.show_string:
                value_widget = None
                value_setter = None
                value_getter = None
            elif layout_info and layout_info.dataset_selector_action:
                hor_layout = QHBoxLayout()
                layout.addLayout(hor_layout)
                widgets.append(QPushButton('Select Dataset...'))
                widgets[-1].setAccessibleName(param_name + '_select')
                button = widgets[-1]
                hor_layout.addWidget(button)
                button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
                # connect to the button click
                widgets.append(QLineEdit())
                hor_layout.addWidget(widgets[-1])
                widgets[-1].setReadOnly(True)
                widgets[-1].editingFinished.connect(lambda: self.do_param_widgets(param_obj))
                widgets[-1].editingFinished.connect(lambda: self.on_end_do_param_widgets())
                widgets[-1].setAccessibleName(param_name)
                action = layout_info.dataset_selector_action
                button.clicked.connect(lambda: getattr(action[0], action[1])(parent_class, param_name, widgets[-1]))
                value_widget = widgets[-1]
                value_setter = widgets[-1].setText
                value_getter = widgets[-1].text
            elif layout_info and layout_info.string_is_label:
                widgets.append(QLabel(val))
                layout.addWidget(widgets[-1])
                widgets[-1].setAccessibleName(param_name)
                value_widget = widgets[-1]
                value_setter = widgets[-1].setText
                value_getter = widgets[-1].text
            elif layout_info and layout_info.multiline_edit:
                widgets.append(QPlainTextEdit())
                layout.addWidget(widgets[-1])
                widgets[-1].setAccessibleName(param_name)
                widgets[-1].textChanged.connect(lambda: self.do_param_widgets(param_obj))
                widgets[-1].textChanged.connect(lambda: self.on_end_do_param_widgets())
                widgets[-1].setTabChangesFocus(layout_info.tab_changes_focus)
                value_widget = widgets[-1]
                value_setter = widgets[-1].setPlainText
                value_getter = widgets[-1].toPlainText
            elif layout_info and layout_info.string_is_file_selector:
                hor_layout = QHBoxLayout()
                layout.addLayout(hor_layout)
                widgets.append(QPushButton('Select File...'))
                button = widgets[-1]
                hor_layout.addWidget(button)
                button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
                widgets[-1].setAccessibleName(param_name + '_select')
                # connect to the button click
                widgets.append(QLineEdit())
                hor_layout.addWidget(widgets[-1])
                widgets[-1].setReadOnly(True)
                button.clicked.connect(lambda: self.do_string_file_selector(param_obj, widgets[-1]))
                widgets[-1].setAccessibleName(param_name)
                value_widget = widgets[-1]
                value_setter = widgets[-1].setText
                value_getter = widgets[-1].text
            else:
                widgets.append(QLineEdit())
                layout.addWidget(widgets[-1])
                widgets[-1].editingFinished.connect(lambda: self.do_param_widgets(param_obj))
                widgets[-1].editingFinished.connect(lambda: self.on_end_do_param_widgets())
                widgets[-1].setAccessibleName(param_name)
                value_widget = widgets[-1]
                value_setter = widgets[-1].setText
                value_getter = widgets[-1].text
        elif ptype == param.FileSelector:
            hor_layout = QHBoxLayout()
            layout.addLayout(hor_layout)
            widgets.append(QPushButton('Select File...'))
            button = widgets[-1]
            hor_layout.addWidget(button)
            button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
            widgets[-1].setAccessibleName(param_name + '_select')
            # connect to the button click
            widgets.append(QLineEdit())
            hor_layout.addWidget(widgets[-1])
            widgets[-1].setReadOnly(True)
            # widgets[-1].editingFinished.connect(lambda: self.do_param_widgets(param_obj))
            # widgets[-1].editingFinished.connect(lambda: self.on_end_do_param_widgets())
            button.clicked.connect(lambda: self.do_file_selector(param_obj, widgets[-1]))
            widgets[-1].setText(param_obj.path)
            widgets[-1].setAccessibleName(param_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = widgets[-1].text
        elif ptype == param.Boolean:
            widgets.append(QCheckBox(label_str))
            layout.addWidget(widgets[-1])
            widgets[-1].clicked.connect(lambda: self.do_param_widgets(param_obj))
            widgets[-1].clicked.connect(lambda: self.on_end_do_param_widgets())
            widgets[-1].setAccessibleName(param_name)
            value_widget = widgets[-1]
            value_setter = widgets[-1].setChecked
            value_getter = widgets[-1].isChecked
        elif ptype == param.DataFrame:
            if layout_info and layout_info.xy_series_action:
                widgets.append(QPushButton('XY Series...'))
                button = widgets[-1]
                layout.addWidget(button)
                button.setAccessibleName(param_name)
                button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
                action = layout_info.xy_series_action
                button.clicked.connect(lambda: getattr(action[0], action[1])(parent_class, param_name))
                value_widget = widgets[-1]
                value_setter = None
                value_getter = None
            else:
                df = getattr(parent_class, param_name)
                widgets.append(self.setup_ui_table_view(df))
                layout.addWidget(widgets[-1])
                value_widget = widgets[-1]
                value_widget.setAccessibleName(param_name)
                model = widgets[-1].filter_model.sourceModel()
                table_widget = widgets[-1]
                value_setter = model
                value_getter = model.data_frame
                if layout_info and layout_info.size_policy_min:
                    widgets[-1].setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
                if not layout_info or layout_info.dataframe_rows_spin_box:
                    # add spin box for number of rows
                    hor_layout = QHBoxLayout()
                    layout.addLayout(hor_layout)
                    widgets.append(QLabel('Number of rows:'))
                    hor_layout.addWidget(widgets[-1])
                    widgets.append(QSpinBox())
                    spin = widgets[-1]
                    hor_layout.addWidget(spin)
                    spin.setMinimum(1)
                    spin.setMaximum(100000)
                    spin.setKeyboardTracking(False)
                    spin.setValue(len(df.index))
                    spin.valueChanged.connect(lambda: self.df_numrows_changed(spin, model, parent_class, param_name))
                    spin.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
                    spin.setAccessibleName(param_name + '_spin_box')
                    fmodel = table_widget.filter_model
                    fmodel.dataChanged.connect(lambda: self.df_data_changed(spin, table_widget, fmodel))
                    hor_layout.addStretch()
        elif ptype == param.ClassSelector:
            layout_to_pass = layout_in
            if layout_info is not None and layout_info.group_id is not None:
                layout_to_pass = layout
            val = getattr(parent_class, param_name)
            if val is not None:
                self.add_params_to_layout(layout_to_pass, val)
            value_widget = None
            value_setter = None
            value_getter = None
        else:
            raise RuntimeError(f'Unsupported "param" parameter type: {ptype}')

        self.param_dict[param_obj] = {
            'param_name': param_name,
            'parent_class': parent_class,
            'value_widget': value_widget,
            'value_getter': value_getter,
            'value_setter': value_setter,
            'widget_list': widgets,
        }

    def setup_ui_table_view(self, df: pd.DataFrame) -> QxTableView:
        """Sets up the table view for the output times.

        Args:
            df: Data for the table view model
        """
        widget = QxTableView()
        if not df.index.empty and df.index[0] == 0:
            df.index = df.index + 1  # Start index at 1, not 0
        self.set_table_model(df, widget)
        widget.horizontalHeader().setSectionResizeMode(QHeaderView.ResizeToContents)
        return widget

    def set_table_model(self, df: pd.DataFrame, widget) -> None:
        """Sets up and updates table model while removing Nan columns in DataFrame.

        Args:
            df: The dataframe.
            widget: The Qt table view widget.
        """
        model = QxPandasTableModel(df)
        widget.filter_model = QSortFilterProxyModel(self.parent_dialog)
        widget.filter_model.setSourceModel(model)
        widget.setModel(widget.filter_model)

    def df_data_changed(self, spin, table_widget, model):
        """Sets up the table veiw for the output times.

        Args:
            spin: The spinbox Qt widget
            table_widget: The Qt QxTableView
            model: The Qt model
        """
        if table_widget.pasting:
            return

        row_count = model.rowCount()
        spin.setValue(row_count)

    def df_numrows_changed(self, spin, model, parent_class, param_name):
        """Sets up the table veiw for the output times.

        Args:
            spin: The spinbox Qt widget
            model: The Qt model
            parent_class: The Qt parent
            param_name: Name of the param currently being proccessed
        """
        model_rows = model.rowCount()
        spin_rows = spin.value()
        more_rows = spin_rows - model_rows
        if more_rows > 0:
            model.insertRows(model_rows, more_rows)
        elif more_rows < 0:
            less_rows = -more_rows
            model.removeRows(model_rows - less_rows, less_rows)
        setattr(parent_class, param_name, model.data_frame)

    def do_file_selector(self, param_obj, file_edit_field):
        """Display a file selector dialog.

        Args:
            param_obj: Object to display selector for
            file_edit_field (QLineEdit): Qt widget holding descriptive text
        """
        p_dict = self.param_dict[param_obj]
        curr_filename = getattr(p_dict['parent_class'], p_dict['param_name'])
        path = param_obj.path
        if curr_filename:
            path = os.path.dirname(curr_filename)
        if not os.path.exists(path):
            path = settings.get_file_browser_directory()
        file_filter = 'All files (*.*)'
        dlg = QFileDialog(self.parent_dialog, 'Select File', path, file_filter)
        dlg.setLabelText(QFileDialog.Accept, "Select")
        if dlg.exec_():
            setattr(p_dict['parent_class'], p_dict['param_name'], dlg.selectedFiles()[0])
            file_edit_field.setText(dlg.selectedFiles()[0])
        self.do_param_widgets(param_obj)
        self.on_end_do_param_widgets()

    def do_string_file_selector(self, param_obj, file_edit_field):
        """Display a file selector dialog.

        Args:
            param_obj: Object to display selector for
            file_edit_field (QLineEdit): Qt widget holding descriptive text
        """
        p_dict = self.param_dict[param_obj]
        curr_filename = getattr(p_dict['parent_class'], p_dict['param_name'])
        path = ''
        if curr_filename:
            path = os.path.dirname(curr_filename)
        if not os.path.exists(path):
            path = settings.get_file_browser_directory()
        file_filter = 'All files (*.*)'
        dlg = QFileDialog(self.parent_dialog, 'Select File', path, file_filter)
        dlg.setLabelText(QFileDialog.Accept, "Select")
        if dlg.exec_():
            setattr(p_dict['parent_class'], p_dict['param_name'], dlg.selectedFiles()[0])
            file_edit_field.setText(dlg.selectedFiles()[0])
        self.do_param_widgets(param_obj)
        self.on_end_do_param_widgets()

    def on_end_do_param_widgets(self):
        """Sends a signal after do_param_widgets is called."""
        self.end_do_param_widgets.emit()

    def do_param_widgets(self, param_obj):
        """Create widgets from a param object.

        Args:
            param_obj: param object to create widgets for
        """
        if self.doing_param_widgets:
            return
        self.doing_param_widgets = True

        if param_obj is not None:
            p_dict = self.param_dict[param_obj]
            # get the value for this parameter for its widget and set it in the param class
            val = p_dict['value_getter']()
            setattr(p_dict['parent_class'], p_dict['param_name'], val)

        # setting a param value can trigger changes to other members of the class so we need
        # to set the values of the other params to the widgets
        for par in self.param_dict.keys():
            if par == param_obj:
                continue
            self._set_param_widget_value(par)

        # hide or show widgets base on current precedence value of param objects
        for par in self.param_dict.keys():
            hide = par.precedence < 0
            p_dict = self.param_dict[par]
            for widget in p_dict['widget_list']:
                if hide:
                    widget.hide()
                else:
                    widget.show()

        self.doing_param_widgets = False

    def _set_param_widget_value(self, par):
        """Sets widgets for a param object.

        Args:
            par: param object to set as an attribute.
        """
        p_dict = self.param_dict[par]
        val = getattr(p_dict['parent_class'], p_dict['param_name'])
        if type(par) in [param.Number, param.Integer]:
            val = str(val)
        if type(val) is pd.DataFrame:
            model = p_dict['value_setter']
            if model is not None:
                model.data_frame = val
        else:
            if p_dict['value_setter'] is not None:
                p_dict['value_setter'](val)
